/*
 * Created on Apr 27, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.service;

import java.util.List;

import com.google.gwt.user.client.rpc.RemoteService;
import com.google.gwt.user.client.rpc.RemoteServiceRelativePath;
import com.tandbergtv.cms.portal.content.client.title.model.UIActionTitle;
import com.tandbergtv.cms.portal.content.client.title.model.UIMetadataEnhInfo;
import com.tandbergtv.cms.portal.content.client.title.model.UITitle;
import com.tandbergtv.cms.portal.content.client.title.model.UIViewInfo;
import com.tandbergtv.cms.portal.content.client.title.model.compare.TitleCompareRequest;
import com.tandbergtv.cms.portal.content.client.title.model.compare.UITitleCompareData;
import com.tandbergtv.cms.portal.content.client.title.model.compare.UITitleCompareXmlData;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIAssetFilePath;
import com.tandbergtv.cms.portal.content.client.title.view.metadata.UITitleInfo;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneException;

/**
 * Service interface defining methods used by the Title View
 * 
 * @author Vijay Silva
 */
@RemoteServiceRelativePath("service/content/title/ITitleViewService")
public interface ITitleViewService extends RemoteService {

	/**
	 * Create a new Title saving the title metadata
	 * 
	 * @param title The title to create
	 */
	public UITitle createTitle(UITitle title, String meRequestKey) throws NeptuneException, TitleServiceException,
	        TitleValidationUIException;

	/**
	 * Create a new Title in 'Draft' status saving the title metadata
	 * 
	 * @param title The title to create
	 */
	public UITitle createTitleAsDraft(UITitle title) throws NeptuneException, TitleServiceException;

	/**
	 * Update the title metadata given the root asset for the title and the title ID
	 * 
	 * @param titleId The title ID
	 * @param asset The root asset for the title
	 * @return The updated title
	 */
	public UITitle updateTitleMetadata(UITitle title, Long requestKey, String action,
	        String listOfAssetIds, String meRequestKey) throws NeptuneException, TitleServiceException,
	        TitleValidationUIException;

	/**
	 * Update the title metadata given the root asset for the title and the title ID
	 * 
	 * @param titleId The title ID
	 * @param asset The root asset for the title
	 * @return The updated title
	 */
	public UITitle updateTitleMetadataAsDraft(UITitle title) throws NeptuneException,
	        TitleServiceException;

	/**
	 * Update the title metadata given the root asset for the title and the title ID
	 * 
	 * @param titleId The title ID
	 * @param asset The root asset for the title
	 * @return The updated title
	 */
	public void batchUpdateTitleMetadata(UITitle updates, List<Long> titleIds,
			boolean draft, boolean currentState) throws NeptuneException,
	        TitleServiceException;

	/**
	 * Get the Title Metadata for the given title ID
	 * 
	 * @param titleId The title ID
	 * @return The title metadata
	 */
	public UITitle getTitle(Long titleId) throws NeptuneException;

	/**
	 * Get the Title History for a title given the Title ID
	 * 
	 * @param titleId The title ID
	 * @return The Title History
	 */
	public UITitle getTitleHistory(Long titleId) throws NeptuneException;

	/**
	 * Get the title metadata for a specific revision of the title
	 * 
	 * @param titleId The title ID
	 * @param titleRevision The title revision
	 * @return The root asset for the title of that revision
	 */
	public UITitleInfo getTitleRevisionInfo(Long titleId, String titleRevision) throws NeptuneException, TitleServiceException;

	/**
	 * Roll back the title to the specific revision
	 * 
	 * @param titleId The title ID
	 * @param titleRevision The title revision
	 */
	public void rollbackTitle(Long titleId, String titleRevision) throws NeptuneException,
	        TitleValidationUIException, TitleServiceException;

	/**
	 * Roll back the title to the specific revision without performing any validation checks. Also
	 * sets the title status to DRAFT.
	 * 
	 * @param titleId The title ID
	 * @param titleRevision The title revision
	 */
	public void rollbackTitleAsDraft(Long titleId, String titleRevision) throws NeptuneException,
	        TitleServiceException;

	/**
	 * Builds a list of the File Paths that can be used as values for the File URI widget
	 * 
	 * @return The list of file paths
	 */
	public List<UIAssetFilePath> getAvailableFilePaths() throws NeptuneException,
	        TitleFileIOException;

	/**
	 * Sends a message to workflow to complete a manual task of selecting a title and asset
	 * 
	 * @param titleId Selected title
	 * @param assetId Selected asset
	 * @param requestKey Request key for the message
	 * @throws NeptuneException
	 */
	public void sendSelectAssetNotification(Long titleId, Long assetId, Long requestKey)
	        throws NeptuneException;

	/**
	 * Sends a message to workflow to complete a manual task of validating a title
	 * 
	 * @param requestKey Request key for the message
	 * @throws NeptuneException
	 */
	public void sendValidationSuccessConfirmation(Long requestKey) throws NeptuneException;

	/**
	 * Sends a message to workflow to complete a manual task of resolving a conflict from metadata enhancement search results.
	 * 
	 * @param requestKey Request key for the message
	 * @throws NeptuneException
	 */
	public void sendMEDeconflictSuccessConfirmation(Long requestKey) throws NeptuneException;
	
	
	/**
	 * Builds a preview of the title metadata in the representation of a specification.
	 * 
	 * @param title The title to preview
	 * @param specification The specification name
	 * @return Preview of the title in a specification specific representation
	 */
	public String previewTitle(UIActionTitle title, String specification, boolean useHtml) throws NeptuneException;

	/**
	 * Compares two titles. Both titles must either be different versions of the same title, or have
	 * the original - copy relationship
	 * 
	 * @param request Title compare request object which holds information regarding the titles to compare
	 * @throws NeptuneException
	 */
	public UITitleCompareData compareTitles(TitleCompareRequest request) throws NeptuneException;

	/**
	 * Generates the previews for the titles being compared, and returns the preview keys.
	 * 
	 * @param title1 The first title to compare
	 * @param title2 The second title to compare
	 * @param specificationName The name of the specification for preview
	 * @throws NeptuneException
	 */
	public UITitleCompareXmlData compareTitlePreviews(UIActionTitle uiTitle1,
	        UIActionTitle uiTitle2, String specificationName, boolean useHtml) throws NeptuneException;

	/**
	 * Verify if the current user has permission to view the given title
	 * 
	 * @param titleId the title id
	 * @return if the current user has permission or not
	 * @throws NeptuneException
	 */
	public Boolean hasViewPermission( Long titleId ) throws NeptuneException;

	/**
	 * Get basic information (including permission and tabs to exclude) required to view the given title
	 * 
	 * @param titleId the title id
	 * @return UIViewInfo
	 * @throws NeptuneException
	 */
	public UIViewInfo getViewInfo( Long titleId ) throws NeptuneException;
	
	
	/**
	 * Gets all info required to display title metadata tab 
	 * @param titleId
	 * @param contentClassId
	 * @return
	 * @throws NeptuneException
	 */
	public UITitleInfo getTitleMetadataInfo(Long titleId, Long contentClassId, Long partnerId) throws NeptuneException;
	
	/**
	 * Check if the given <code>titleId</code> represents a MediaPath Distribution Site Title.
	 * 
	 * @param titleId
	 * @return
	 * @throws NeptuneException
	 */
	public Boolean isMediaPathSiteTitle(Long titleId) throws NeptuneException;

	public UITitleInfo getTitleMetadataInfoForBatch(Long contentClassId)
			throws NeptuneException;
	
	/**
	 * Search for metadata enhancements from 3rd party data sources.
	 * 
	 * @param title The title to create or update
	 * @return UIMetadataEnhInfo 
	 * @throws NeptuneException
	 */
	public UIMetadataEnhInfo searchMetadataEnhancement(UITitle title) throws NeptuneException;
	
	/**
	 * Delete Metadata Enhancement cache.
	 * 
	 * @param Metadata Enhancement request key
	 * @throws NeptuneException
	 */
	public void cleanUpMetadataEnhancementRequest(String meRequestKey) throws NeptuneException;
}
