/*
 * Created on Feb 25, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.service.asset;

import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIAsset;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIGroupAsset;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIItemAsset;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.IUIAssetDefinitionVisitor;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIAssetDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIGroupAssetDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIItemAssetDefinition;

/**
 * Service implementation for the asset builder that constructs an asset and its required children /
 * fields based on the provided asset definition.
 * 
 * @author Vijay Silva
 */
class UIAssetBuilder implements IUIAssetDefinitionVisitor {

	/* The asset built after the visitor is successfully accepted */
	private transient UIAsset builtAsset = null;

	/* Flag indicating if the required children need to be built */
	private transient boolean buildRequiredChildren = true;

	/* Flag indicating if the tree we are building is for batch edit */
	private transient boolean batchEdit = false;

	/**
	 * Constructor
	 */
	public UIAssetBuilder() {
	}

	/**
	 * Builds an asset and its required children based on the asset definition provided.
	 * 
	 * @param assetDefinition The asset definition
	 * @return The asset
	 */
	public UIAsset build(UIAssetDefinition assetDefinition) {
		return this.build(assetDefinition, true);
	}

	/**
	 * Builds an asset based on the asset definition provided. The required children of the asset
	 * are built only if the buildChildren flag is set to true.
	 * 
	 * @param assetDefinition The asset definition
	 * @param buildChildren Flag to build the tree of asset children
	 * @return The asset
	 */
	public UIAsset build(UIAssetDefinition assetDefinition, boolean buildChildren) {
		this.buildRequiredChildren = buildChildren;
		assetDefinition.accept(this);
		return this.getBuiltAsset();
	}

	/**
	 * Builds an asset based on the asset definition provided. The required children of the asset
	 * are built only if the buildChildren flag is set to true.
	 * 
	 * @param assetDefinition The asset definition
	 * @param buildChildren Flag to build the tree of asset children
	 * @return The asset
	 */
	public UIAsset build(UIAssetDefinition assetDefinition, boolean buildChildren, boolean batchEdit) {
		this.buildRequiredChildren = buildChildren;
		this.batchEdit = batchEdit;
		assetDefinition.accept(this);
		return this.getBuiltAsset();
	}

	/**
	 * Build the asset for the input group asset definition. Handles building of all the required
	 * child assets for the definition as well.
	 * 
	 * @see com.tandbergtv.cms.portal.ui.title.client.model.specification.IUIAssetDefinitionVisitor#visit
	 *      (com.tandbergtv.cms.portal.ui.title.client.model.specification.UIGroupAssetDefinition)
	 */
	public void visit(UIGroupAssetDefinition assetDefinition) {
		/* Build the Group Asset */
		UIGroupAsset asset = new UIGroupAsset();
		asset.setAssetType(assetDefinition.getAssetType());

		/* Build all the children recursively */
		if (this.buildRequiredChildren) {
			for (UIAssetDefinition childDefinition : assetDefinition.getChildren()) {
				
				int requiredCount = 
					batchEdit?childDefinition.getBatchMinimumCount():
						childDefinition.getMinimumCount();
				for (int i = 0; i < requiredCount; i++) {
					UIAsset childAsset = this.build(childDefinition);
					asset.addChild(childAsset);
				}
			}
		}

		/* Store the built asset */
		this.builtAsset = asset;
	}

	/**
	 * Build the asset for the input item asset definition.
	 * 
	 * @see com.tandbergtv.cms.portal.ui.title.client.model.specification.IUIAssetDefinitionVisitor#visit
	 *      (com.tandbergtv.cms.portal.ui.title.client.model.specification.UIItemAssetDefinition)
	 */
	public void visit(UIItemAssetDefinition assetDefinition) {
		/* Build the Item Asset */
		UIItemAsset asset = new UIItemAsset();
		asset.setAssetType(assetDefinition.getAssetType());

		/* Store the built asset */
		this.builtAsset = asset;
	}

	/*
	 * Get the built asset and reset the state
	 */
	private UIAsset getBuiltAsset() {
		UIAsset asset = this.builtAsset;
		this.builtAsset = null;
		this.buildRequiredChildren = true;
		return asset;
	}
}
