/*
 * Created on Apr 24, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.view;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.logical.shared.BeforeSelectionEvent;
import com.google.gwt.event.logical.shared.BeforeSelectionHandler;
import com.google.gwt.event.logical.shared.SelectionEvent;
import com.google.gwt.event.logical.shared.SelectionHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.Window.ClosingEvent;
import com.google.gwt.user.client.Window.ClosingHandler;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.ContentComponent;
import com.tandbergtv.cms.portal.content.client.title.model.UITitle;
import com.tandbergtv.cms.portal.content.client.title.model.UITitleOverview;
import com.tandbergtv.cms.portal.content.client.title.model.UIViewInfo;
import com.tandbergtv.cms.portal.content.client.title.service.ITitleViewService;
import com.tandbergtv.cms.portal.content.client.title.service.ITitleViewServiceAsync;
import com.tandbergtv.cms.portal.content.client.title.view.activity.TitleActivityTab;
import com.tandbergtv.cms.portal.content.client.title.view.history.TitleHistoryTab;
import com.tandbergtv.cms.portal.content.client.title.view.metadata.TitleMetadataTab;
import com.tandbergtv.cms.portal.content.client.title.view.metadata.TitleValidationView;
import com.tandbergtv.cms.portal.content.client.title.view.metadata.TitleValidationViewInput;
import com.tandbergtv.cms.portal.content.client.title.view.msooverrideshistory.MSOOverridesHistoryTab;
import com.tandbergtv.cms.portal.content.client.title.view.props.TitlePropertiesTab;
import com.tandbergtv.cms.portal.content.client.title.view.rules.RulesPreviewTab;
import com.tandbergtv.cms.portal.content.client.title.view.series.TitleSeriesTab;
import com.tandbergtv.cms.portal.content.client.title.view.sites.TitleSitesTab;
import com.tandbergtv.cms.portal.ui.title.client.view.bundle.TitleClientBundle;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.application.event.BeforeContentChangeEvent;
import com.tandbergtv.neptune.widgettoolkit.client.application.event.BeforeContentChangeHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.TabContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.HasAnchor;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.HasViewAnchorChangeHandlers;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.HasViewCancelHandlers;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.HasViewCommitHandlers;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewAnchorChangeEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewAnchorChangeHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCommitEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCommitHandler;

/**
 * The Title View represents the complete view for a Title.
 * 
 * @author Vijay Silva
 */
public class TitleView extends Composite implements HasAnchor, HasViewAnchorChangeHandlers,
        HasViewCommitHandlers<TitleViewInput>, HasViewCancelHandlers 
{
	private TitleMetadataTab metadataTab;
	private final SimpleContainer viewContainer;
	private VerticalContainer contentContainer;
	private TitleViewHeaderWidget headerPanel;
	private TitleValidationView messageView;
	
	private TabContainer tabContainer;
	private BusyIndicator busyIndicator;

	/* Services */
	private TitleViewMessages messages;
	private ITitleViewServiceAsync viewService;
	private TitleClientBundle bundle;

	/* View Input */
	private TitleViewInput input;

	/* Internal state */
	private boolean ignoreTabEvents = false;
	private HandlerRegistration windowEventRegistration, contentChangeRegistration;

	/* Constants */
	private static final String TITLE_VIEW_STYLE = "content-TitleView";
	private static final String TITLE_VIEW_TABCONTAINER = "content-TitleView-tabContainer";
	private static final String TITLE_HEADER_PANEL_STYLE = "content-TitleView-headerPanel";
	private static final String STYLE_SEVERE_ERROR = "content-TitleView-severeErrorLabel";
	private static int CMS_HEADER_MENU_SIZE_PX = 115;
	private static int METADATA_TAB_INDEX = 0;

	/**
	 * Constructs a new Title View
	 */
	public TitleView() {
		viewContainer = new SimpleContainer();
		this.initWidget(viewContainer);
		this.initialize();
	}

	/**
	 * Constructs a new Title View
	 * 
	 * @param input The view input
	 */
	public TitleView(TitleViewInput input) {
		this();
		this.setInput(input);
		this.refresh();
	}

	public void showMessage(TitleValidationViewInput data)
	{
		messageView.refresh(data);
	}
	
	public Widget getMessageViewWidget()
	{
		return messageView;
	}
	
	
	public TitleMetadataTab getMetadataTab()
	{
		return metadataTab;
	}
	
	// ========================================================================
	// ===================== HAS ANCHOR AND INPUT MANAGEMENT
	// ========================================================================

	@Override
	public String getAnchor() {
		// TODO: handle titleIds
		return (getInput() != null) ? getInput().buildAnchor() : null;
	}

	@Override
	public void setAnchor(String anchor) {
		TitleViewInput input = new TitleViewInput(anchor);
		setInput(input);
		refresh();
	}

	/**
	 * Get the input rendered by this title view
	 * 
	 * @return The title view input
	 */
	public TitleViewInput getInput() {
		return this.input;
	}

	/**
	 * Set the input for the view, causing the view to be refreshed
	 * 
	 * @param input The tab input
	 */
	public void setInput(TitleViewInput input) {
		this.input = input;
	}

	// ========================================================================
	// ===================== WIDGET OVERRIDES
	// ========================================================================

	@Override
	protected void onLoad() {
		super.onLoad();

		/* Add window close handler for any unsaved changes on current tab when window is closed */
		this.windowEventRegistration = Window.addWindowClosingHandler(new ClosingHandler() {
			@Override
			public void onWindowClosing(ClosingEvent event) {
				TitleViewTab tab = getSelectedTab();
				if (tab != null && tab.isDirty()) {
					event.setMessage(getViewMessages().tabConfirmIgnoreChangesMessage());
				}
			}
		});

		NeptuneApplication app = NeptuneApplication.getApplication();
		this.contentChangeRegistration = app.addBeforeContentChangeHandler(new BeforeContentChangeHandler() {
			@Override
			public void onBeforeContentChange(BeforeContentChangeEvent event) {
				TitleViewTab tab = getSelectedTab();
				if (!ignoreChangesOnTab(tab)) {
					event.cancel();
				}
			}
		});
	}

	/**
	 * When unloading the widget, ensure that the feedback popup is hidden
	 * 
	 * @see com.google.gwt.user.client.ui.Widget#onUnload()
	 */
	@Override
	protected void onUnload() 
	{
		/* Hide the busy indicator */
		if (busyIndicator != null) {
			busyIndicator.hide();
		}

		/* Remove the window closing listener */
		if (windowEventRegistration != null) {
			windowEventRegistration.removeHandler();
			windowEventRegistration = null;
		}

		/* Remove the content change listener */
		if (contentChangeRegistration != null) {
			contentChangeRegistration.removeHandler();
			contentChangeRegistration = null;
		}

		super.onUnload();
	}

	// ========================================================================
	// ===================== EVENT MANAGEMENT
	// ========================================================================

	@Override
	public HandlerRegistration addViewCancelHandler(ViewCancelHandler handler) {
		return addHandler(handler, ViewCancelEvent.getType());
	}

	@Override
	public HandlerRegistration addViewAnchorChangeHandler(ViewAnchorChangeHandler handler) {
		return addHandler(handler, ViewAnchorChangeEvent.getType());
	}

	@Override
	public HandlerRegistration addViewCommitHandler(ViewCommitHandler<TitleViewInput> handler) {
		return addHandler(handler, ViewCommitEvent.getType());
	}

	/*
	 * Fire an anchor change event
	 */
	public void fireViewAnchorChange() {
		String anchor = getAnchor();
		fireEvent(new ViewAnchorChangeEvent(anchor));
	}

	/*
	 * Fire a view commit event
	 */
	protected void fireViewCommit() {
		fireEvent(new ViewCommitEvent<TitleViewInput>(getInput()));
	}

	/*
	 * Fire the cancel action
	 */
	void fireCancelled(TitleViewTab source) {
		fireEvent(new ViewCancelEvent());
	}

	/*
	 * Fire the title created action
	 */
	void fireTitleCreated(TitleViewTab source, UITitle title) {
		/* Update the title maintained by the input */
		this.input.updateTitle(title);

		/* Update the bookmark */
		fireViewAnchorChange();

		/* Update the view */
		this.markTabsStale(source);
		this.enableTabSelection();

		/* Fire the commit action */
		fireViewCommit();
	}

	/*
	 * Fire the title updated action
	 */
	void fireTitleUpdated(TitleViewTab source, UITitle title) {
		/* Update the title maintained by the input */
		this.input.updateTitle(title);

		/* Update the view */
		this.markTabsStale(source);

		/* Fire the commit action */
		fireViewCommit();
	}

	/*
	 * Fire the batch title update action
	 */
	void fireBatchTitleUpdated(TitleViewTab source) {

		/* Update the view */
		this.markTabsStale(source);

		/* Fire the commit action */
		fireViewCommit();
	}

	/*
	 * Handle tab event that causes a change in the dirty flag of the tab.
	 */
	void handleTabDirtyChange(TitleViewTab source) {
		String tabDisplayName = source.getTabDisplayName();
		if (source.isDirty()) {
			tabDisplayName += " " + getViewMessages().tabDirtyMarker();
		}

		int index = getTabIndex(source.getTabName());
		tabContainer.getTabBar().setTabText(index, tabDisplayName);
	}

	// ========================================================================
	// ===================== SHARED VIEW TAB STATE
	// ========================================================================

	/**
	 * Get the view service used by the different components of this view
	 * 
	 * @return The View Service
	 */
	ITitleViewServiceAsync getViewService() {
		return this.viewService;
	}

	/**
	 * Get the internationalized constants for this view
	 * 
	 * @return The View Constants
	 */
	TitleViewMessages getViewMessages() {
		return this.messages;
	}

	/**
	 * Get the title client bundle for image icons
	 * 
	 * @return the bundle
	 */
	TitleClientBundle getBundle() {
		return bundle;
	}

	// ========================================================================
	// ===================== INITIALIZATION
	// ========================================================================

	/*
	 * Initialize the widgets and the state displayed by this view
	 */
	private void initialize() {
		this.initializeServices();
		this.initializeWidgets();
	}

	/*
	 * Initialize the view service for making calls to the server
	 */
	private void initializeServices() {
		this.viewService = GWT.create(ITitleViewService.class);
		this.messages = GWT.create(TitleViewMessages.class);
		this.bundle = new TitleClientBundle();
	}

	/*
	 * Initialize the widgets displayed by this view
	 */
	private void initializeWidgets() 
	{
		/* Build the busy indicator */
		this.busyIndicator = new BusyIndicator();

		/* Style the main view container */
		addStyleName(TITLE_VIEW_STYLE);

		/* Build the content container */
		contentContainer = new VerticalContainer();

		/* Build the header panel */
		headerPanel = new TitleViewHeaderWidget(this);
		headerPanel.addStyleName(TITLE_HEADER_PANEL_STYLE);
		
		
		messageView = new TitleValidationView();
		

		/* Create the tab container and the tabs */
		this.tabContainer = new TabContainer();
		tabContainer.addStyleName(TITLE_VIEW_TABCONTAINER);
		addTabs(tabContainer);
		
		/* Select the first tab, and disable tab selection */
		tabContainer.selectTab(0);
		disableTabSelection();
		TitleTabEventHandler handler = new TitleTabEventHandler();
		tabContainer.addSelectionHandler(handler);
		tabContainer.addBeforeSelectionHandler(handler);

		/* Add the widgets to the view container */
		contentContainer.add(headerPanel);
		contentContainer.add(messageView);
		contentContainer.add(tabContainer);
	}

	/**
	 * Create the tabs to display and add to provided tab container
	 * 
	 * @param tabContainer The tab container widget to add tabs to
	 */
	protected void addTabs(TabContainer tabContainer) 
	{
		TitleViewTab tab = null;

		/* Metadata Tab */
		metadataTab = new TitleMetadataTab(this);
		tabContainer.add(metadataTab, metadataTab.getTabDisplayName());

		/* Properties Tab */
		tab = new TitlePropertiesTab(this);
		tabContainer.add(tab, tab.getTabDisplayName());

		/* History Tab */
		tab = new TitleHistoryTab(this);
		tabContainer.add(tab, tab.getTabDisplayName());

		/* Sites Tab */
		boolean isSitesFeatureActive = isSitesFeatureActive();
		if (isSitesFeatureActive == true) {
			tab = new TitleSitesTab(this);
			tabContainer.add(tab, tab.getTabDisplayName());
		}

		/* Activity Tab */
		tab = new TitleActivityTab(this);
		tabContainer.add(tab, tab.getTabDisplayName());

		/* Series Tab */
		tab = new TitleSeriesTab(this);
		tabContainer.add(tab, tab.getTabDisplayName());
		
		// Rules Preview Tab
		tab = new RulesPreviewTab(this);
		tabContainer.add(tab, tab.getTabDisplayName());
		
		/* MSO Overrides History Tab */
		/* Should be shown only if this is a view for a MediaPath Site Title */
		/* Add to the tabContainer initially. */ 
		/* Will be removed from tabContainer (in refreshInput() below) if title being displayed is not a MediaPath Site Title. */
		tab = new MSOOverridesHistoryTab(this);
		tabContainer.add(tab, tab.getTabDisplayName());
	}
	
	protected void renderActionsListWidgets(UIViewInfo viewInfo) {
		if (getInput() != null && getInput().getTitleId() != null) {
			//If it is a site title, remove Series and Rule Preview tabs
			if (viewInfo.getIsSiteTitle()) {
				removeTabByName(TitleSeriesTab.TAB_NAME);
				getMetadataTab().buildActionsListWidgets(true, false);
			}
			else {
				//Master
				getMetadataTab().buildActionsListWidgets(false, false);
			}
		}
		else {
			// Initial Create
			getMetadataTab().setWidgetsVisibility(false);
		}
	}

	private boolean isSitesFeatureActive() {
		NeptuneApplication application = NeptuneApplication.getApplication();
		ContentComponent component = application.getComponent(ContentComponent.class);
		return component.isSitesFeatureActive();
	}

	// ========================================================================
	// ===================== REFRESHING VIEW
	// ========================================================================

	/**
	 * Refresh the state displayed on this view. Do not call this method until after a valid input
	 * is set.
	 */
	public void refresh() {
		this.markTabsStale(null);
		this.refreshInput();
	}

	/*
	 * Mark all tabs (except the source) as stale
	 */
	private void markTabsStale(TitleViewTab source) {
		/* Mark the state on all tabs as stale except the source tab */
		for (int tabIndex = 0; tabIndex < tabContainer.getTabBar().getTabCount(); tabIndex++) {
			Widget widget = tabContainer.getWidget(tabIndex);
			if ((widget instanceof TitleViewTab) && !widget.equals(source)) {
				((TitleViewTab) widget).markStale();
			}
		}
	}

	/*
	 * Update the input displayed by this view by fetching from the server
	 */
	private void refreshInput() {
		/* Verify that the input is set */
		viewContainer.clear();
		if (getInput() == null) {
			return;
		}

		/* Verify that the anchor set for the widget is valid */
		if (getInput().isInvalidAnchor()) {
			String invalidField = getInput().getInvalidAnchorField();
			String message = getViewMessages().viewAnchorInvalid(invalidField);
			showSevereError(message);
		}
		/* Verify that the user has permission to view the title */
		else if (!getInput().hasTitleReadPermission()) {
			String message = getViewMessages().viewNotAuthorized();
			showSevereError(message);
		}
		/* Verify that the user has permission to create the title */
		else if (!getInput().isBulkEdit() && !getInput().hasTitleCreatePermission() && getInput().getTitleId() == null) {
			String message = getViewMessages().createNotAuthorized();
			showSevereError(message);
		}
		/* Show the title view if the title belongs to authorized partners */
		else {
			showBusyIndicator();
			/* Verify that the user has permission to view the title */
			viewService.getViewInfo( getInput().getTitleId(), 
				new AsyncCallback<UIViewInfo>() {

    				@Override
    				public void onFailure( Throwable caught ) {
    					String error = (caught != null) ? caught.getLocalizedMessage() : "";
    					String name = getInput().getSpecificationName();
    					String message = (name != null) ? getViewMessages().specificationFetchFailure(name, error)
    					        : getViewMessages().defaultSpecificationFetchFailure(error);

    					/* Show the error panel and hide busy indicator */
    					showSevereError(message);
    					hideBusyIndicator();
    				}
    
    				@Override
        			public void onSuccess(UIViewInfo viewInfo) {
        				if (viewInfo.getViewPermission()) {	
	    					/* Show the title view */					
    					
        					/* Ensure that the content container is showing */
        					viewContainer.setWidget(contentContainer);
        					ignoreTabEvents = true;
        					try {
        						/* Disable tab selection and go to tab specified as selected in input */
        						disableTabSelection();
        						tabContainer.selectTab(getSelectedTabIndex());
        					} finally {
        						ignoreTabEvents = false;
        					}
        					
        					renderActionsListWidgets(viewInfo);

        					// If it is a media path site title, remove mso override history tab
        					if(viewInfo.getIsMediaPathSiteTitle() == false) 
        					{
        						removeTabByName(MSOOverridesHistoryTab.TAB_NAME);
        					}
        					
        					/* Update the tab being displayed and enable the required tabs */
        					updateTab(tabContainer.getTabBar().getSelectedTab());
        					enableTabSelection();
        					
    					} else {
    						String message = getViewMessages().viewNotAuthorized();
        					showSevereError(message);
    					}
    					hideBusyIndicator();
    				}
    			} 
			);
		}
	}

	
	private void removeTabByName(String tabName)
	{
		int tabIndex = getTabIndex(tabName);
		if(tabIndex >= 0) 
		{
			Widget w = tabContainer.getWidget(tabIndex);
			if (w != null) 
			{
				tabContainer.remove(w);
			}
		}
	}
	
	
	/*
	 * Get the tab index specified in the title view input
	 */
	private int getSelectedTabIndex() {
		int index = METADATA_TAB_INDEX;

		/* If not a new title, show appropriate index */
		if (getInput().isExistingTitle()) {
			String tabName = getInput().getTabName();
			index = getTabIndex(tabName);
			if (index < 0)
				index = METADATA_TAB_INDEX;
		}

		return index;
	}

	/*
	 * Get the title view tab that is current selected, or null if the selected tab is not a title
	 * view tab widget
	 */
	private TitleViewTab getSelectedTab() {
		Widget widget = tabContainer.getWidget(getSelectedTabIndex());
		if (widget instanceof TitleViewTab) {
			return (TitleViewTab) widget;
		}

		return null;
	}

	/*
	 * Update the display for the tab if the specified tab state is stale
	 */
	private void updateTab(int tabIndex) {
		Widget widget = tabContainer.getWidget(tabIndex);
		if (widget instanceof TitleViewTab) {
			TitleViewTab tab = (TitleViewTab) widget;
			if (tab.isStale()) {
				tab.refresh();
			}
		}
	}

	/*
	 * Enabled selection of the appropriate tabs in the UI
	 */
	private void enableTabSelection() {
		boolean existingTitle = (getInput().getTitleId() != null);
		int tabCount = tabContainer.getTabBar().getTabCount();
		for (int tabIndex = 0; tabIndex < tabCount; tabIndex++) {
			Widget widget = tabContainer.getWidget(tabIndex);
			boolean enabled = existingTitle && !input.isBulkEdit();
			if (widget instanceof TitleMetadataTab) {
				enabled = true;
			}
			tabContainer.getTabBar().setTabEnabled(tabIndex, enabled);
		}
	}

	/*
	 * Disable selection of any of the tabs from the UI
	 */
	private void disableTabSelection() {
		int tabCount = tabContainer.getTabBar().getTabCount();
		for (int tabIndex = 0; tabIndex < tabCount; tabIndex++) {
			tabContainer.getTabBar().setTabEnabled(tabIndex, false);
		}
	}

	/**
	 * Hides the title view and shows the severe error message
	 */
	void showSevereError(String message) {
		hideBusyIndicator();
		LabelWidget label = new LabelWidget(message);
		label.addStyleName(STYLE_SEVERE_ERROR);
		viewContainer.setWidget(label);
	}

	/*
	 * Check if the user wants to ignore changes on the current tab before navigating away
	 */
	private boolean ignoreChangesOnTab(TitleViewTab tab) {
		if (tab != null && tab.isDirty()) {
			return Window.confirm(getViewMessages().tabConfirmIgnoreChangesMessage());
		}

		return true;
	}

	// ========================================================================
	// ===================== TAB SELECTION
	// ========================================================================

	/*
	 * The tab listener that updates the tab state before displaying the tab if required
	 */
	private class TitleTabEventHandler implements SelectionHandler<Integer>,
	        BeforeSelectionHandler<Integer> {

		/*
		 * The tab has been selected, update the tab if the state is stale
		 */
		@Override
		public void onSelection(SelectionEvent<Integer> event) 
		{
			if (ignoreTabEvents)
				return;

			/* Notify the tab of selection and update the input */
			Widget widget = tabContainer.getWidget(event.getSelectedItem());
			if (widget instanceof TitleViewTab) {
				TitleViewTab viewTab = (TitleViewTab) widget;
				getInput().setTabName(viewTab.getTabName());
				viewTab.onTabSelected();
				fireViewAnchorChange();
			}

			updateTab(event.getSelectedItem());
		}

		/*
		 * The tab is about to be selected
		 */
		@Override
		public void onBeforeSelection(BeforeSelectionEvent<Integer> event) {
			if (ignoreTabEvents)
				return;

			int currentIndex = tabContainer.getTabBar().getSelectedTab();
			if (currentIndex != -1) {
				Widget widget = tabContainer.getWidget(currentIndex);
				if (widget instanceof TitleViewTab) {
					TitleViewTab tab = (TitleViewTab) widget;
					if (tab.isDirty()) {
						/* Tab is dirty, issue warning! */
						if (!ignoreChangesOnTab(tab)) {
							event.cancel();
						} else {
							tab.refresh();
							tab.onTabUnselected();
						}
					}
					else
						tab.onTabUnselected();
				}
			}
		}
	}

	/**
	 * Switch to tab with given tab name (if not already displayed)
	 * 
	 * @param tabName The tab name
	 */
	public void showTab(String tabName) {
		int index = getEnabledTabIndex(tabName);
		if (index >= 0 && tabContainer.getTabBar().getSelectedTab() != index) {
			tabContainer.selectTab(index);
		}
	}

	/*
	 * Get the tab index for the tab name if the tab is enabled
	 */
	private int getEnabledTabIndex(String tabName) {
		int index = getTabIndex(tabName);
		if (index >= 0 && !tabContainer.getTabBar().isTabEnabled(index)) {
			index = -1;
		}

		return index;
	}

	/*
	 * Get the tab index for the given tab name
	 */
	private int getTabIndex(String tabName) {
		int tabCount = tabContainer.getTabBar().getTabCount();
		for (int tabIndex = 0; tabIndex < tabCount; tabIndex++) {
			Widget widget = tabContainer.getWidget(tabIndex);
			if (widget instanceof TitleViewTab) {
				if (((TitleViewTab) widget).getTabName().equals(tabName)) {
					return tabIndex;
				}
			}
		}

		return -1;
	}

	// ========================================================================
	// ===================== TITLE HEADING MANAGEMENT
	// ========================================================================

	/**
	 * Update the title header given the title name and status
	 */
	void updateTitleHeader(UITitleOverview overview) {
		this.headerPanel.setInput(overview);
		this.headerPanel.refresh();
	}

	/**
	 * Show the master title comparison, triggered from the header widget
	 */
	void showMasterTitleComparison(Long masterTitleId, String targetedVersion) {
		int index = getEnabledTabIndex(TitleSitesTab.TAB_NAME);
		if (index >= 0) {
			TitleSitesTab tab = (TitleSitesTab) tabContainer.getWidget(index);
			tab.showMasterTitleComparison(masterTitleId, targetedVersion);
			showTab(TitleSitesTab.TAB_NAME);
		}
	}

	/**
	 * Show the distributed title comparison, triggered from the header widget
	 */
	void showDistributedTitleComparison(Long titleId, String distributedVersion) {
		int index = getEnabledTabIndex(TitleSitesTab.TAB_NAME);
		if (index >= 0) {
			TitleSitesTab tab = (TitleSitesTab) tabContainer.getWidget(index);
			tab.showDistributedTitleComparison(titleId, distributedVersion);
			showTab(TitleSitesTab.TAB_NAME);
		}
	}

	// ========================================================================
	// ===================== CONTENT AREA CALCULATIONS
	// ========================================================================

	/**
	 * @return the tabHeight
	 */
	int getTabHeight() {
		return Window.getClientHeight() - getUsedHeight();
	}

	/**
	 * @return the tabWidth
	 */
	int getTabWidth() {
		return Window.getClientWidth() - getUsedWidth();
	}

	/**
	 * @return the used height
	 */
	int getUsedHeight() {
		/* application header offset */
		int usedHeight = CMS_HEADER_MENU_SIZE_PX;

		/* header panel and the tab bar offsets */
		usedHeight += headerPanel.getOffsetHeight() + tabContainer.getTabBar().getOffsetHeight();

		/* 10px - tab bottom panel padding and margin */
		usedHeight += 10;

		return usedHeight;
	}

	/**
	 * @return the used width
	 */
	int getUsedWidth() {
		/* 20px - application padding, 10px - title view padding, 10px - tab container padding */
		return 40;
	}

	// ========================================================================
	// ===================== BUSY INDICATOR
	// ========================================================================

	/**
	 * @return the busy indicator
	 */
	BusyIndicator getBusyIndicator() {
		return busyIndicator;
	}

	/**
	 * Show the busy indicator in the center of the browser
	 */
	void showBusyIndicator() {
		busyIndicator.center();
	}

	/**
	 * Hide the busy indicator, if showing
	 */
	void hideBusyIndicator() {
		busyIndicator.hide();
	}

}
