/*
 * Created on Apr 24, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.view;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.tandbergtv.cms.portal.content.client.ContentComponent;
import com.tandbergtv.cms.portal.content.client.Permissions;
import com.tandbergtv.cms.portal.content.client.create.model.UIContentClass;
import com.tandbergtv.cms.portal.content.client.title.model.UITitle;
import com.tandbergtv.cms.portal.content.client.title.model.UITitleOverview;
import com.tandbergtv.cms.portal.content.client.title.model.history.UITitleHistory;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.UITitleMetadata;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIAsset;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIAssetFilePath;
import com.tandbergtv.cms.portal.content.client.title.service.ITitleViewService;
import com.tandbergtv.cms.portal.content.client.title.service.ITitleViewServiceAsync;
import com.tandbergtv.cms.portal.content.client.title.service.asset.UIAssetFactory;
import com.tandbergtv.cms.portal.content.client.title.view.activity.TitleActivityTab;
import com.tandbergtv.cms.portal.content.client.title.view.history.TitleHistoryTab;
import com.tandbergtv.cms.portal.content.client.title.view.metadata.TitleMetadataTab;
import com.tandbergtv.cms.portal.content.client.title.view.metadata.UITitleInfo;
import com.tandbergtv.cms.portal.content.client.title.view.sites.TitleSitesTab;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIAssetSpecification;
import com.tandbergtv.cms.portal.ui.title.client.model.title.UITitleStatus;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.AnchorTokenizer;

/**
 * The input to the Title View that contains the Specification and the Title being viewed. The model
 * for this view is large, and hence is fetched in a lazy manner where possible.
 * 
 * @author Vijay Silva
 */
public class TitleViewInput {

	/* Properties */
	private Long requestKey;
	private String action;
	private String listOfAssetIds;
	private String tabName;
	private boolean invalidAnchor;
	private String invalidAnchorField;
	private Set<Long> bulkEditTitleIds = null;
	private Long selectedAssetId;
	private String from;	// Anchor token From. At the moment the only value supported is From=Template

	/* The title view service */
	private ITitleViewServiceAsync viewService = GWT.create(ITitleViewService.class);

	/* The specification fetched lazily */
	private UIAssetSpecification specification = null;

	/* The title model with properties fetched lazily */
	private UITitle title = null;

	/* The list of file paths / URI values that can be used as suggestions */
	private List<UIAssetFilePath> filePaths;

	/** Anchor Token: Title Id */
	public static final String TITLE_ID_ANCHOR_KEY = "Id";

	/** Anchor Token: Request Key */
	public static final String REQUESTKEY_ANCHOR_KEY = "RequestKey";

	/** Anchor Token: Action Name */
	public static final String ACTION_ANCHOR_KEY = "Action";

	/** Anchor Token: Asset ID list */
	public static final String ASSET_IDS_ANCHOR_KEY = "AssetIDs";

	/** Anchor Token: Selected Asset ID, id of the  selected node(asset) in the asset tree*/
	public static final String SELECTED_ASSET_ID_ANCHOR_KEY = "SelectedAsset.Id";
	
	/** Anchor Token: Tab Name */
	public static final String TAB_ANCHOR_KEY = "Tab";

	/** Action Value: Select Action */
	public static final String ACTION_SELECT = "SELECT";

	/** Action Value: Map Action */
	public static final String ACTION_MAP = "MAP_ASSET";

	/** Action Value: Correct Action */
	public static final String ACTION_CORRECT = "CORRECT";

	/** Action Value: Deconflict Action */
	public static final String ACTION_DECONFLICT = "DECONFLICT";
	
	/** The Metadata Tab Name */
	public static final String METADATA_TAB_NAME = TitleMetadataTab.TAB_NAME;

	/** The History Tab Name */
	public static final String HISTORY_TAB_NAME = TitleHistoryTab.TAB_NAME;

	/** The Activity Tab Name */
	public static final String ACTIVITY_TAB_NAME = TitleActivityTab.TAB_NAME;

	/** The Sites Tab Name */
	public static final String SITES_TAB_NAME = TitleSitesTab.TAB_NAME;

	/**
	 * Construct a new input from the anchor
	 * 
	 * @param anchor the anchor
	 */
	public TitleViewInput(String anchor) {
		parseAnchor(anchor);
	}

	/**
	 * Create an input for a new title
	 * 
	 * @param partnerId The partner ID
	 * @param contentClassId The content class ID
	 * @param contentClassMetadata The content class metadata
	 */
	public TitleViewInput(long partnerId, long contentClassId) {
		this.title = new UITitle();
		this.title.setPartnerId(partnerId);
		this.title.setContentClassId(contentClassId);
	}

	
	/**
	 * Create a Title View Input for an existing title using the current specification with request
	 * key information
	 * 
	 * @param titleId The title ID
	 * @param editable flag indicating if the input is editable
	 * @param requestKey The request key for the order that asked for manual mapping
	 */
	public TitleViewInput(Long titleId, Long requestKey, String action, String listOfAssetIds) {
		this.title = new UITitle();
		this.title.setId(titleId);
		this.requestKey = requestKey;
		this.action = action;
		this.listOfAssetIds = listOfAssetIds;
	}

	/**
	 * Create a Title View Input for bulk edit of titles
	 * 
	 * @param titleIds The list of titles for bulk edit
	 * @param editable flag indicating if the input is editable
	 * @param requestKey The request key for the order that asked for manual mapping
	 */
	public TitleViewInput(Collection<Long> titleIds, long contentClassId) {
		this.title = new UITitle();
		this.title.setContentClassId(contentClassId);
		
		/* Verify the input */
		if (titleIds == null || titleIds.size() == 0) {
			throw new IllegalArgumentException("Cannot perform bulk edit of 0 titles.");
		}

		this.bulkEditTitleIds = new HashSet<Long>();
		this.bulkEditTitleIds.addAll(titleIds);
	}

	
	public UITitle getTitle()
	{
		return title;
	}

	public String getFrom() {
		return from;
	}

	
	// ========================================================================
	// ===================== SECURITY
	// ========================================================================

	/**
	 * Determine if the user has permission to read the title
	 * 
	 * @return true if permission exists, false otherwise
	 */
	public boolean hasTitleReadPermission() {
		return ClientAuthorizationManager.isAuthorized(Permissions.TITLE_VIEW);
	}

	/**
	 * Determine if the user has permission to modify the title
	 * 
	 * @return true if permission exists, false otherwise
	 */
	public boolean hasTitleModifyPermission() {
		return ClientAuthorizationManager.isAuthorized(Permissions.TITLE_MODIFY);
	}

	/**
	 * Determine if the user has permission to create a title
	 * 
	 * @return true if permission exists, false otherwise
	 */
	public boolean hasTitleCreatePermission() {
		return ClientAuthorizationManager.isAuthorized(Permissions.TITLE_CREATE);
	}

	// ========================================================================
	// ===================== SPECIFICATION MANAGEMENT
	// ========================================================================

	/**
	 * Get the Specification Name
	 * 
	 * @return The specification name
	 */
	public String getSpecificationName() {
		return getDefaultSpecificationName();
	}


	// ========================================================================
	// ===================== TITLE MANAGEMENT
	// ========================================================================

	/**
	 * Get the title ID for this input
	 * 
	 * @return the title ID
	 */
	public Long getTitleId() {
		return title.getId();
	}

	/**
	 * Determine if the input title is an existing / persisted title, or is a new title
	 * 
	 * @return true if an existing title, false otherwise
	 */
	public boolean isExistingTitle() {
		return (title.getId() != null);
	}

	/**
	 * Determine if the view is for bulk edit of titles
	 * 
	 * @return True if for bulk edit of titles, false otherwise
	 */
	public boolean isBulkEdit() {
		return (this.bulkEditTitleIds != null);
	}

	/**
	 * Call this method when the title is created / updated. This will invalidate the existing
	 * cached state of the title. The state provided by the new title is maintained as the new
	 * cached state. Any parts of the state that are null / missing are fetched from the server on
	 * the next call.
	 * 
	 * @param title The updated title
	 */
	void updateTitle(UITitle title) {
		/* Only update the Title ID if the current ID is null. */
		if (getTitleId() == null) {
			this.title.setId(title.getId());
		}

		this.invalidateTitle();
		this.title.setMetadata(title.getMetadata());
		this.title.setHistory(title.getHistory());
	}

	/**
	 * Invalidate the entire title, clearing any cached state of the title
	 */
	public void invalidateTitle() {
		this.invalidateMetadata();
		this.invalidateHistory();
		this.filePaths = null;
	}

	// ========================================================================
	// ===================== TITLE METADATA MANAGEMENT
	// ========================================================================

	/**
	 * Get the cached metadata, or null if no cache is present.
	 * 
	 * @return the metadata
	 */
	public UITitleMetadata getMetadata() {
		return title.getMetadata();
	}
	private void onGetMetadataSuccess(final AsyncCallback<UITitleInfo> callback, UITitleInfo result) {
	
		TitleViewInput.this.specification = result.getSpecification();
		TitleViewInput.this.filePaths = result.getFilePaths();
		
		// Existing title
		if(title.getId() != null) {
			UITitle resTitle = result.getTitle();

			// Overview
			UITitleOverview overview = resTitle.getOverview(); 
			if(overview == null) overview = new UITitleOverview();
			overview.setContentClassName(specification.getContentClassName());
			title.setOverview(overview);

			title.setMetadata(resTitle.getMetadata());
			// Set content class
			title.setContentClassId(resTitle.getContentClassId());
			//title.setContentClassMetadata(resTitle.getContentClassMetadata());
			callback.onSuccess(result);
		}
		// New title
		else {
			// Build new root asset using factory and cache the root asset
			UITitleOverview overview = new UITitleOverview();
			overview.setStatus(UITitleStatus.DRAFT);
			overview.setContentClassName(specification.getContentClassName());
			title.setOverview(overview);

			UITitleMetadata metadata = new UITitleMetadata();
			UIAssetFactory assetFactory = new UIAssetFactory();
			UIAsset rootAsset = assetFactory.createAsset(specification, isBulkEdit());
			metadata.setRootAsset(rootAsset);
			metadata.setTitleStatus(UITitleStatus.DRAFT);
			title.setMetadata(metadata);
					
			if(bulkEditTitleIds != null && bulkEditTitleIds.size() >= 1) {
				Long bulkTitleId = bulkEditTitleIds.iterator().next();
				viewService.getTitle(bulkTitleId, new NeptuneAsyncCallback<UITitle>() {
					@Override
					public void onNeptuneFailure(Throwable caught) {
						callback.onFailure(caught);
					}

					@Override
					public void onNeptuneSuccess(UITitle result) {
						title.setContentClassId(result.getContentClassId());
						//title.setContentClassMetadata(result.getContentClassMetadata());
						UITitleInfo info = new UITitleInfo();
						info.setTitle(result);
						info.setSpecification(specification);
						info.setFilePaths(filePaths);
						callback.onSuccess(info);
					}
				});
			} 
			else {
				result.setTitle(title);
				callback.onSuccess(result);
			}
		}				
		
	}
	/**
	 * Get the metadata from the cache, or go to the server if not present.
	 * 
	 * @param callback The call back to call after getting the root asset
	 */
	public void getMetadata(final AsyncCallback<UITitleInfo> callback) {
		// Get title metadata from the server
		
		if(bulkEditTitleIds != null) {
			//call batch method
			viewService.getTitleMetadataInfoForBatch(title.getContentClassId(),new NeptuneAsyncCallback<UITitleInfo>() {

				@Override
				public void onNeptuneFailure(Throwable caught) {
					callback.onFailure(caught);
					
				}

				@Override
				public void onNeptuneSuccess(UITitleInfo result) {
					// TODO Auto-generated method stub
					onGetMetadataSuccess(callback, result);
				}
				
			});
		}
		else {
		viewService.getTitleMetadataInfo(title.getId(), title.getContentClassId(), 
				title.getPartnerId(), new NeptuneAsyncCallback<UITitleInfo>() 
		{
			@Override
			public void onNeptuneFailure(Throwable caught) {
				callback.onFailure(caught);
			}

			@Override
			public void onNeptuneSuccess(UITitleInfo result) {
				onGetMetadataSuccess(callback, result);
			}
		});
		}
	}

	/**
	 * Invalidate the title metadata, forcing the title metadata to be read from the server
	 */
	public void invalidateMetadata() {
		title.setMetadata(null);
	}

	/**
	 * Get the title IDs for bulk edit
	 */
	public Set<Long> getBulkEditTitleIds() {
		return bulkEditTitleIds;
	}

	/**
	 * Get the ordered list of title IDs for bulk edit
	 */
	public List<Long> getOrderedBulkEditTitleIds() {
		if (bulkEditTitleIds == null)
			return null;
		
		List<Long> titleIds = new ArrayList<Long>(bulkEditTitleIds);
		Collections.sort(titleIds);
		return titleIds;
	}

	// ========================================================================
	// ===================== TITLE HISTORY MANAGEMENT
	// ========================================================================

	/**
	 * Get the cached history for this title. Returns null if no cache is present.
	 * 
	 * @return The cached history
	 */
	public UITitleHistory getHistory() {
		return title.getHistory();
	}

	/**
	 * Get the title history from cache, or go to the server if not present.
	 * 
	 * @param callback The call back to call after getting the title history
	 */
	public void getHistory(final AsyncCallback<UITitle> callback) {
		Long titleId = title.getId();
		UITitleHistory history = title.getHistory();
		if (history != null) {
			callback.onSuccess(title);
		} 
		// Existing title
		else if (titleId != null) {
			viewService.getTitleHistory(titleId, new NeptuneAsyncCallback<UITitle>() {
				@Override
				public void onNeptuneFailure(Throwable caught) {
					callback.onFailure(caught);
				}

				@Override
				public void onNeptuneSuccess(UITitle result) {
					title.setOverview(result.getOverview());
					title.setHistory(result.getHistory());
					callback.onSuccess(result);
				}
			});
		}
		// New title
		else {
			// Create blank overview and history for new title
			UITitleOverview overview = new UITitleOverview();
			overview.setStatus(UITitleStatus.DRAFT);
			title.setOverview(overview);

			history = new UITitleHistory();
			title.setHistory(history);
			callback.onSuccess(title);
		}
	}

	/**
	 * Invalidate the title history, forcing the title history to be read from the server
	 */
	public void invalidateHistory() {
		title.setHistory(null);
	}

	/**
	 * @return the requestKey
	 */
	public Long getRequestKey() {
		return requestKey;
	}

	/**
	 * @return the action
	 */
	public String getAction() {
		return action;
	}

	/**
	 * @return the listOfAssetIds
	 */
	public String getListOfAssetIds() {
		return listOfAssetIds;
	}

	/**
	 * @return the selectButtonVisible
	 */
	public boolean isSelectButtonVisible() {
		return ACTION_SELECT.equals(getAction());
	}

	/**
	 * @return the tabName
	 */
	public String getTabName() {
		return tabName;
	}

	/**
	 * @param tabName the tabName to set
	 */
	public void setTabName(String tabName) {
		this.tabName = tabName;
	}

	/**
	 * Determine if the anchor used to build the input could not be parsed
	 * 
	 * @return True if the anchor could not be
	 */
	public boolean isInvalidAnchor() {
		return invalidAnchor;
	}

	/**
	 * @return the unparsableField
	 */
	public String getInvalidAnchorField() {
		return invalidAnchorField;
	}

	public Long getSelectedAssetId() {
		return selectedAssetId;
	}
	
	public void setSelectedAssetId(Long selectedAssetId) {
		this.selectedAssetId = selectedAssetId;
	}
	
	// ========================================================================
	// ===================== TITLE FILE MANAGEMENT
	// ========================================================================

	/**
	 * Get the cached list of unmapped file paths that can be used when mapping an asset
	 */
	public List<UIAssetFilePath> getFilePaths() {
		return this.filePaths;
	}

	/**
	 * Get the title history from cache, or go to the server if not present.
	 * 
	 * @param callback The call back to call after getting the title history
	 */
	public void getFilePaths(final AsyncCallback<List<UIAssetFilePath>> callback) {
		if (this.filePaths != null) {
			callback.onSuccess(this.filePaths);
		} else {
			viewService.getAvailableFilePaths(new NeptuneAsyncCallback<List<UIAssetFilePath>>() {
				@Override
				public void onNeptuneFailure(Throwable caught) {
					callback.onFailure(caught);
				}

				@Override
				public void onNeptuneSuccess(List<UIAssetFilePath> result) {
					filePaths = result;
					callback.onSuccess(result);
				}
			});
		}
	}


	// ========================================================================
	// ===================== ANCHOR MANAGEMENT
	// ========================================================================

	public String buildAnchor() {
		AnchorTokenizer tokenizer = new AnchorTokenizer();
		Map<String, String> anchorTokens = new LinkedHashMap<String, String>();

		/* Put the title ID */
		if (getTitleId() != null) {
			anchorTokens.put(TITLE_ID_ANCHOR_KEY, getTitleId().toString());
		} 

		/* Put the request key */
		if (getRequestKey() != null) {
			anchorTokens.put(REQUESTKEY_ANCHOR_KEY, getRequestKey().toString());
		}

		/* Put the action value */
		if (getAction() != null) {
			anchorTokens.put(ACTION_ANCHOR_KEY, getAction());
		}

		/* Put the asset IDs */
		if (getListOfAssetIds() != null) {
			anchorTokens.put(ASSET_IDS_ANCHOR_KEY, getListOfAssetIds());
		}

		/* Put the current tab name */
		if (getTabName() != null) {
			anchorTokens.put(TAB_ANCHOR_KEY, tabName);
		}

		/* Put the selected Asset ID*/
		if (getSelectedAssetId() != null) {
			anchorTokens.put(SELECTED_ASSET_ID_ANCHOR_KEY, selectedAssetId.toString());
		}
		
		return tokenizer.buildAnchor(anchorTokens);
	}

	/*
	 * Parse the anchor building the input from the anchor
	 */
	private void parseAnchor(String anchor) {
		AnchorTokenizer tokenizer = new AnchorTokenizer();
		Map<String, String> anchorTokens = tokenizer.parseAnchor(anchor);

		this.title = new UITitle();
		this.title.setId(parseLong(anchorTokens.get(TITLE_ID_ANCHOR_KEY), TITLE_ID_ANCHOR_KEY));
		this.requestKey = parseLong(anchorTokens.get(REQUESTKEY_ANCHOR_KEY), REQUESTKEY_ANCHOR_KEY);
		this.action = anchorTokens.get(ACTION_ANCHOR_KEY);
		this.listOfAssetIds = anchorTokens.get(ASSET_IDS_ANCHOR_KEY);
		this.tabName = anchorTokens.get(TAB_ANCHOR_KEY);
		this.selectedAssetId = parseLong(anchorTokens.get(SELECTED_ASSET_ID_ANCHOR_KEY), SELECTED_ASSET_ID_ANCHOR_KEY);
		this.from = anchorTokens.get("From");
	}

	/*
	 * Get the default specification name from the content component
	 */
	private String getDefaultSpecificationName() {
		NeptuneApplication application = NeptuneApplication.getApplication();
		ContentComponent component = application.getComponent(ContentComponent.class);
		return component.getSpecificationName();
	}

	/*
	 * Parse the Long value provided by the anchor keeping track if the value cannot be parsed
	 */
	private Long parseLong(String value, String fieldName) {
		if (value == null) {
			return null;
		} else {
			try {
				return Long.parseLong(value.trim());
			} catch (RuntimeException e) {
				if (!invalidAnchor) {
					this.invalidAnchor = true;
					this.invalidAnchorField = fieldName;
				}
			}
		}

		return null;
	}

	public String getContentClassName()
	{
		return (specification == null) ? "" : specification.getContentClassName();
	}

	public boolean isSeriesTitle()
	{
		return (specification != null && specification.getContentClassTypeId() == UIContentClass.TYPE_SERIES);
	}

	public boolean isCategoryTitle()
	{
		return (specification != null && specification.getContentClassTypeId() == UIContentClass.TYPE_CATEGORY);
	}
}
