/*
 * Created on Apr 24, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.view;

import com.google.gwt.user.client.ui.Composite;
import com.tandbergtv.cms.portal.content.client.title.model.UITitle;
import com.tandbergtv.cms.portal.content.client.title.model.UITitleOverview;
import com.tandbergtv.cms.portal.content.client.title.service.ITitleViewServiceAsync;
import com.tandbergtv.cms.portal.content.client.title.view.TitleErrorPanel.EventListener;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;

/**
 * The View for the section displayed in the Title panel. The section is initialized in a lazy
 * manner. State changes made to other sections cause a notification on this tab when the tab is
 * selected.
 * 
 * @author Vijay Silva
 */
public abstract class TitleViewTab extends Composite {

	/* Properties */
	private SimpleContainer mainContainer;
	protected TitleView titleView;
	private boolean stale = true;
	private boolean dirty = false;

	/* Style Names */
	private static final String STYLE_NAME = "content-TitleViewTab";

	/**
	 * Constructor
	 */
	public TitleViewTab(TitleView parent) {
		this.titleView = parent;

		this.mainContainer = new SimpleContainer();
		this.mainContainer.setStylePrimaryName(STYLE_NAME);
		this.initWidget(mainContainer);

		this.initialize();
	}

	/**
	 * Initialize the main widget for this tab adding any widgets for display as required. No input
	 * is present at this point.
	 */
	protected abstract void initialize();

	/**
	 * Get the container that holds the widgets for this tab
	 * 
	 * @return The main simple container
	 */
	protected SimpleContainer getContainer() {
		return this.mainContainer;
	}

	/**
	 * Forces the tab to clear dirty flag and re-fetch data for rendering.
	 */
	public final void refresh() 
	{
		getTitleView().showMessage(null);
		clearDirty();
		refreshTab();
	}

	/**
	 * Forces the tab to re-fetch the input displayed on that tab from the server and update the tab
	 * view once the request to the server completes.
	 */
	protected abstract void refreshTab();
	
	/**
	 * Get the parent title panel
	 * 
	 * @return The title panel
	 */
	public TitleView getTitleView() {
		return this.titleView;
	}

	/**
	 * Get the input for the view
	 * 
	 * @return The title view input
	 */
	public TitleViewInput getViewInput() {
		return this.titleView.getInput();
	}

	/**
	 * Get the service implementation for this view
	 * 
	 * @return The view service
	 */
	public ITitleViewServiceAsync getViewService() {
		return this.titleView.getViewService();
	}

	/**
	 * Get the constants defined for this view
	 * 
	 * @return The constants for this view
	 */
	public TitleViewMessages getViewMessages() {
		return this.titleView.getViewMessages();
	}

	/**
	 * Determine if the state displayed by the tab is stale
	 * 
	 * @return true if the state of the tab is stale, false otherwise
	 */
	public boolean isStale() {
		return stale;
	}

	/**
	 * Allows for this tab to be marked stale so that the state displayed on the tab can be updated
	 */
	public void markStale() {
		this.stale = true;
		
		/* Since tab is stale, it cannot be dirty */
		clearDirty();
	}

	/**
	 * Marks this tab as updated so that future requests to view this tab do not need to re-fetch
	 * the tab state
	 */
	protected void markUpdated() {
		this.stale = false;
	}

	/**
	 * Determine if this tab contains unsaved state
	 * 
	 * @return true if tab has unsaved state, false otherwise
	 */
	public boolean isDirty() {
		return this.dirty;
	}

	/**
	 * Mark this tab as dirty (containing unsaved changes) 
	 */
	protected void markDirty() {
		setDirty(true);
	}

	/**
	 * Clear the dirty flag for the tab (the tab does not contain unsaved changes).
	 */
	protected void clearDirty() {
		setDirty(false);
	}

	/*
	 * Set the dirty flag to the specified value, fire event if value has changed
	 */
	private void setDirty(boolean dirty) {
		if (this.dirty != dirty) {
			this.dirty = dirty;
			this.getTitleView().handleTabDirtyChange(this);
		}
	}

	/**
	 * The cancel button has been clicked on a tab and this action needs to trigger the view cancel
	 * event.
	 */
	protected void onCancelled() {
		this.getTitleView().fireCancelled(this);
	}

	/**
	 * Notify the Title View that the title was created
	 * 
	 * @param titleId The title ID
	 */
	protected void onTitleCreated(UITitle title) {
		this.getTitleView().fireTitleCreated(this, title);
	}

	/**
	 * Notify the Title View that the title (or some part of the title) was saved.
	 */
	protected void onTitleUpdated(UITitle title) {
		this.getTitleView().fireTitleUpdated(this, title);
	}

	/**
	 * Notify the Title View that several titles were saved.
	 */
	protected void onBatchTitleUpdated() {
		this.getTitleView().fireBatchTitleUpdated(this);
	}

	/**
	 * Update the header to show the new title name and status
	 * 
	 * @param titleName The title name
	 * @param titleStatus The title status
	 */
	protected void updateTitleHeader(UITitleOverview overview) {
		this.getTitleView().updateTitleHeader(overview);
	}

	/**
	 * Update the title view to not show anything by the severe error message
	 * 
	 * @param message The error message
	 */
	protected void showSevereError(String message) {
		this.getTitleView().showSevereError(message);
	}

	/**
	 * Show the title error panel with the specified message replacing the current tab widget for
	 * critical errors.
	 */
	protected void showErrorPanel(String message) 
	{
		TitleErrorPanel errorPanel = new TitleErrorPanel(message);

		/* Listen for the refresh event */
		errorPanel.addEventListener(new EventListener() {
			public void onRefreshClicked(TitleErrorPanel panel) {
				refresh();
			}
		});

		getContainer().setWidget(errorPanel);
	}

	/**
	 * Gives a chance to the tab to perform any initialization work it may need before the tab is
	 * displayed.
	 */
	protected void onTabSelected() {
		return;
	}

	/**
	 * Gives a chance to the tab handle loss of selection (when another tab gets selected).
	 */
	protected void onTabUnselected() {
		return;
	}

	/**
	 * Get the busy indicator
	 * 
	 * @return The busy indicator
	 */
	protected BusyIndicator getBusyIndicator() {
		return getTitleView().getBusyIndicator();
	}

	/**
	 * Shows the busy indicator in the center of the screen
	 */
	protected void showBusyIndicator() {
		getTitleView().showBusyIndicator();
	}

	/**
	 * Hides the busy indicator
	 */
	protected void hideBusyIndicator() {
		getTitleView().hideBusyIndicator();
	}

	/**
	 * Get the tab width
	 * 
	 * @return The tab width
	 */
	protected int getTabWidth() {
		return getTitleView().getTabWidth();
	}

	/**
	 * Get the tab height
	 * 
	 * @return The tab height
	 */
	protected int getTabHeight() {
		return getTitleView().getTabHeight();
	}

	/**
	 * Get the height in pixels used by the widget and its parents (not including tab area)
	 * 
	 * @return The used height
	 */
	protected int getUsedHeight() {
		return getTitleView().getUsedHeight();
	}

	/**
	 * Get the width in pixels used by the widget and its parents (not including tab area)
	 * 
	 * @return The used width
	 */
	protected int getUsedWidth() {
		return getTitleView().getUsedWidth();
	}

	/**
	 * Get the name of the tab (cannot be blank and must be unique)
	 * 
	 * @return the tab name
	 */
	protected abstract String getTabName();

	/**
	 * Get the display name for the tab (as soon on the main view)
	 * 
	 * @return The tab display name
	 */
	public abstract String getTabDisplayName();
}
