/*
 * Created on Jun 26, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.view.asset;

import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HTMLTable;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.FlexTableContainer;

/**
 * A table used in the asset content panel to render any part of the asset contents.
 * 
 * @author Vijay Silva
 */
public class AssetContentTable extends Composite {

	/* Properties */
	private FlexTableContainer table = new FlexTableContainer();

	/* Styles */
	private static final String TABLE_STYLE = "content-AssetContentTable";
	private static final String ROW_STYLE = "content-AssetContentTable-row";
	private static final String ROW_HEADER_STYLE = "content-AssetContentTable-row-header";
	private static final String ROW_BODY_STYLE = "content-AssetContentTable-row-body";
	private static final String CELL_STYLE = "content-AssetContentTable-cell";
	private static final String CELL_HEADER_STYLE = "content-AssetContentTable-cell-header";
	private static final String CELL_BODY_STYLE = "content-AssetContentTable-cell-body";
	private static final String CELL_HEADER_LABEL_STYLE = "content-AssetContentTable-header-text";
	public static final String CELL_DELETE_COLUMN_STYLE = "content-AssetContentTable-cell-body-delete";

	/**
	 * Constructor
	 */
	public AssetContentTable() {
		initialize();
	}

	/* Initialize the table */
	private void initialize() {
		table.addStyleName(TABLE_STYLE);

		/* Create the header row */
		table.insertRow(0);
		table.getRowFormatter().addStyleName(0, ROW_STYLE);
		table.getRowFormatter().addStyleName(0, ROW_HEADER_STYLE);
		this.initWidget(table);
	}

	/**
	 * Get the table used by the widget
	 * 
	 * @return The table
	 */
	protected FlexTableContainer getTable() {
		return table;
	}

	// ========================================================================
	// ===================== TABLE HEADING MANAGEMENT
	// ========================================================================

	/**
	 * Get the number of cells in the heading row
	 * 
	 * @return The number of heading cells
	 */
	public int getHeadingCellCount() {
		return table.getCellCount(0);
	}

	/**
	 * Add a new cell to the table header row.
	 * 
	 * @param heading The header text
	 */
	public void addHeading(String heading) {
		/* Add the column to the header row */
		int column = table.getCellCount(0);
		table.insertCell(0, column);

		/* Update the style of the header cells */
		table.getCellFormatter().addStyleName(0, column, CELL_STYLE);
		table.getCellFormatter().addStyleName(0, column, CELL_HEADER_STYLE);

		updateHeading(heading, column);
	}

	/**
	 * Update an existing cell content in the table header row
	 * 
	 * @param heading The header text
	 * @param column The column index
	 */
	public void updateHeading(String heading, int column) {
		/* Create new label widget or update existing widget */
		LabelWidget label = getHeadingWidget(column);
		if (label == null) {
			/* Update the widget displayed as the column header */
			label = new LabelWidget(heading);
			label.addStyleName(CELL_HEADER_LABEL_STYLE);
			table.setWidget(0, column, label);
		} else {
			label.setText(heading);
		}
	}

	/**
	 * Remove the existing cell from the table header row.
	 * 
	 * @param column The column index
	 */
	public void removeHeading(int column) {
		table.removeCell(0, column);
	}

	/**
	 * Determine if a heading cell is present for the specified column index
	 * 
	 * @param column The column index
	 * @return true if the cell is present, false otherwise
	 */
	public boolean isHeadingPresent(int column) {
		return table.isCellPresent(0, column);
	}

	/**
	 * Get the label widget used to represent the header cell
	 * 
	 * @return The header label
	 */
	public LabelWidget getHeadingWidget(int column) {
		Widget widget = table.getWidget(0, column);
		return (widget instanceof LabelWidget) ? (LabelWidget) widget : null;
	}

	// ========================================================================
	// ===================== TABLE HEADING STYLE MANAGEMENT
	// ========================================================================

	/**
	 * Override the style for a table heading row
	 * 
	 * @param styleName The style name
	 */
	public void setTableHeadingStyle(String styleName) {
		table.getRowFormatter().setStyleName(0, styleName);
	}

	/**
	 * Add an additional style for a table heading row
	 * 
	 * @param styleName The style name
	 */
	public void addTableHeadingStyle(String styleName) {
		table.getRowFormatter().addStyleName(0, styleName);
	}

	/**
	 * Remove a style for a table heading row
	 * 
	 * @param styleName The style name
	 */
	public void removeTableHeadingStyle(String styleName) {
		table.getRowFormatter().removeStyleName(0, styleName);
	}

	/**
	 * Override the style for a cell in the table heading row
	 * 
	 * @param column the column index
	 * @param styleName The style name
	 */
	public void setTableHeadingStyle(int column, String styleName) {
		table.getCellFormatter().setStyleName(0, column, styleName);
	}

	/**
	 * Add an additional style for a table heading row
	 * 
	 * @param column the column index
	 * @param styleName The style name
	 */
	public void addTableHeadingStyle(int column, String styleName) {
		table.getCellFormatter().addStyleName(0, column, styleName);
	}

	/**
	 * Remove a style for a table heading row
	 * 
	 * @param column the column index
	 * @param styleName The style name
	 */
	public void removeTableHeadingStyle(int column, String styleName) {
		table.getCellFormatter().removeStyleName(0, column, styleName);
	}

	// ========================================================================
	// ===================== TABLE BODY MANAGEMENT
	// ========================================================================

	/**
	 * Returns a count of the rows in the table body
	 * 
	 * @return The row count
	 */
	public int getRowCount() {
		return table.getRowCount() - 1;
	}

	/**
	 * Add a new row to the table. The row must be added to the table before any cells can be added
	 * to the row.
	 * 
	 * @return the index of the newly created row
	 */
	public int addRow() {
		/* Add a row to the end of the table body (do not count the header row) */
		return this.addRow(getRowCount());
	}

	/**
	 * Add a new row to the table at the specified index. The index cannot exceed the row count
	 * 
	 * @param row The row index
	 * @return The index of the newly created row
	 */
	public int addRow(int row) {
		validateBodyRow(row);

		/* Shift the row index to account for the header row */
		int tableRow = row + 1;
		table.insertRow(tableRow);
		table.getRowFormatter().addStyleName(tableRow, ROW_STYLE);
		table.getRowFormatter().addStyleName(tableRow, ROW_BODY_STYLE);

		return row;
	}

	/**
	 * Remove an existing row from the table
	 * 
	 * @param row The row index
	 */
	public void removeRow(int row) {
		validateBodyRow(row);
		table.removeRow(row + 1);
	}

	/**
	 * Remove all rows from the table without affecting the table header
	 */
	public void removeAllRows() {
		int rowCount = this.getRowCount();
		for (int i = rowCount - 1; i >= 0; i--) {
			this.removeRow(i);
		}
	}

	/**
	 * Get the cell count for a row in the table
	 * 
	 * @param row The row index
	 * @return the number of cells for the row
	 */
	public int getCellCount(int row) {
		return table.getCellCount(row + 1);
	}

	/**
	 * Add a new cell to the table body using the provided widget
	 */
	public void addCell(int row, Widget widget) {
		this.addCell(row, getCellCount(row), widget);
	}

	/**
	 * Add a new cell to the table using the provided widget at the specified column index
	 * 
	 * @param row The row index
	 * @param column The column index
	 * @param widget The widget to add
	 */
	public void addCell(int row, int column, Widget widget) {
		int tableRow = row + 1;
		table.insertCell(tableRow, column);

		/* Update the style of the body cell */
		table.getCellFormatter().addStyleName(tableRow, column, CELL_STYLE);
		table.getCellFormatter().addStyleName(tableRow, column, CELL_BODY_STYLE);

		updateCell(row, column, widget);
	}

	/**
	 * Replace the existing widget in the provided cell
	 * 
	 * @param row The
	 * @param column
	 * @param widget
	 */
	public void updateCell(int row, int column, Widget widget) {
		int tableRow = row + 1;
		table.clearCell(tableRow, column);
		table.setWidget(tableRow, column, widget);
	}

	/**
	 * Remove an existing cell from the row given the column index
	 * 
	 * @param row The row index
	 * @param column The column index
	 */
	public void removeCell(int row, int column) {
		int tableRow = row + 1;
		table.removeCell(tableRow, column);
	}

	/**
	 * Determine if a table cell is present
	 * 
	 * @param row The row index
	 * @param column The column index
	 */
	public void isCellPresent(int row, int column) {
		table.isCellPresent(row + 1, column);
	}

	/**
	 * Get the widget present in the given row and column
	 * 
	 * @param row The row index
	 * @param column The column index
	 * @return The Widget present in the row and column
	 */
	public Widget getCellWidget(int row, int column) {
		return table.getWidget(row + 1, column);
	}

	/**
	 * Get the row index in which the widget is present.
	 * 
	 * @param widget The widget to find
	 * @return The row index or -1 if no row is found
	 */
	public int getRow(Widget widget) {
		for (int row = 0; row < getRowCount(); row++) {
			for (int column = 0; column < getCellCount(row); column++) {
				if (widget.equals(getCellWidget(row, column)))
					return row;
			}
		}

		return -1;
	}

	/*
	 * Ensure that the row index is not less than 0
	 */
	private void validateBodyRow(int row) {
		if (row < 0) {
			throw new IllegalArgumentException("The row index cannot be less than 0.");
		}
	}

	// ========================================================================
	// ===================== TABLE BODY STYLE MANAGEMENT
	// ========================================================================

	/**
	 * Override the style for a table row
	 * 
	 * @param row The row index
	 * @param styleName The style name
	 */
	public void setTableRowStyle(int row, String styleName) {
		table.getRowFormatter().setStyleName(row + 1, styleName);
	}

	/**
	 * Add an additional style for a table row
	 * 
	 * @param row The row index
	 * @param styleName The style name
	 */
	public void addTableRowStyle(int row, String styleName) {
		table.getRowFormatter().addStyleName(row + 1, styleName);
	}

	/**
	 * Remove a style for a table row
	 * 
	 * @param row The row index
	 * @param styleName The style name
	 */
	public void removeTableRowStyle(int row, String styleName) {
		table.getRowFormatter().removeStyleName(row + 1, styleName);
	}

	/**
	 * Override the style for a cell in a table row
	 * 
	 * @param row The row index
	 * @param column The column index
	 * @param styleName The style name
	 */
	public void setTableCellStyle(int row, int column, String styleName) {
		table.getCellFormatter().setStyleName(row + 1, column, styleName);
	}

	/**
	 * Add an additional style for a cell in a table row
	 * 
	 * @param row The row index
	 * @param styleName The style name
	 */
	public void addTableCellStyle(int row, int column, String styleName) {
		table.getCellFormatter().addStyleName(row + 1, column, styleName);
	}

	/**
	 * Remove a style for a cell in a table row
	 * 
	 * @param row The row index
	 * @param styleName The style name
	 */
	public void removeTableCellStyle(int row, int column, String styleName) {
		table.getCellFormatter().removeStyleName(row + 1, column, styleName);
	}

	// ========================================================================
	// ===================== TABLE MANAGEMENT
	// ========================================================================

	/**
	 * @see HTMLTable#getCellPadding()
	 */
	public int getCellPadding() {
		return table.getCellPadding();
	}

	/**
	 * @see HTMLTable#setCellPadding(int)
	 */
	public void setCellPadding(int padding) {
		table.setCellPadding(padding);
	}

	/**
	 * @see HTMLTable#getCellSpacing()
	 */
	public int getCellSpacing() {
		return table.getCellSpacing();
	}

	/**
	 * @see HTMLTable#setCellSpacing(int)
	 */
	public void setCellSpacing(int spacing) {
		table.setCellSpacing(spacing);
	}

	/**
	 * @see HTMLTable#setBorderWidth(int)
	 */
	public void setBorderWidth(int width) {
		table.setBorderWidth(width);
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	/**
	 * @see HTMLTable#addClickHandler(ClickHandler)
	 */
	public HandlerRegistration addClickHandler(ClickHandler handler) {
		return this.addDomHandler(handler, ClickEvent.getType());
	}

	/**
	 * Determine if the click event happened for a cell in the table header row
	 * 
	 * @param event The click event
	 * @return true if the click was for a cell in the header row, false otherwise
	 */
	public boolean isHeadingClicked(ClickEvent event) {
		HTMLTable.Cell cell = table.getCellForEvent(event);
		return (cell != null && cell.getRowIndex() == 0);
	}

	/**
	 * Determine if the click event happened for a cell in the table body row
	 * 
	 * @param event The click event
	 * @return true if the click was for a cell in the body row, false otherwise
	 */
	public boolean isBodyClicked(ClickEvent event) {
		HTMLTable.Cell cell = table.getCellForEvent(event);
		return (cell != null && cell.getRowIndex() > 0);
	}

	/**
	 * Get the index of the row in which the cell was clicked
	 * 
	 * @param event The click event
	 * @return -1 if no cell was clicked or if a header cell was clicked, or the row index
	 */
	public int getClickedRow(ClickEvent event) {
		HTMLTable.Cell cell = table.getCellForEvent(event);
		int rowIndex = -1;
		if (cell != null && cell.getRowIndex() > 0) {
			rowIndex = cell.getRowIndex() - 1;
		}
		return rowIndex;
	}

	/**
	 * Get the index of the cell that was clicked
	 * 
	 * @param event The click event
	 * @return -1 if no cell was clicked, or the column index for the cell clicked
	 */
	public int getClickedColumn(ClickEvent event) {
		HTMLTable.Cell cell = table.getCellForEvent(event);
		return (cell != null) ? cell.getCellIndex() : -1;
	}
}
