package com.tandbergtv.cms.portal.content.client.title.view.asset;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;
import com.google.gwt.event.dom.client.ContextMenuEvent;
import com.google.gwt.event.dom.client.ContextMenuHandler;
import com.google.gwt.event.logical.shared.ResizeEvent;
import com.google.gwt.event.logical.shared.ResizeHandler;
import com.google.gwt.event.logical.shared.SelectionEvent;
import com.google.gwt.event.logical.shared.SelectionHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.MenuBar;
import com.google.gwt.user.client.ui.MenuItem;
import com.google.gwt.user.client.ui.PopupPanel;
import com.google.gwt.user.client.ui.ScrollPanel;
import com.google.gwt.user.client.ui.TreeItem;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.ContentComponent;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIAsset;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIAssetFilePath;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIGroupAsset;
import com.tandbergtv.cms.portal.content.client.title.service.asset.IUIAssetFactory;
import com.tandbergtv.cms.portal.content.client.title.service.asset.UIAssetFactory;
import com.tandbergtv.cms.portal.content.client.title.view.HasViewChangeHandlers;
import com.tandbergtv.cms.portal.content.client.title.view.TitleView;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewMessageCache;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewMessages;
import com.tandbergtv.cms.portal.content.client.title.view.ViewChangeEvent;
import com.tandbergtv.cms.portal.content.client.title.view.ViewChangeHandler;
import com.tandbergtv.cms.portal.content.client.title.view.asset.AssetTree.AssetInfo;
import com.tandbergtv.cms.portal.content.client.title.view.keyframes.KeyframesLogicalClipPanel;
import com.tandbergtv.cms.portal.content.client.title.view.keyframes.KeyframesSetImagePanel;
import com.tandbergtv.cms.portal.content.client.title.view.videoplayer.SetVideoImagePanel;
import com.tandbergtv.cms.portal.content.client.title.view.videoplayer.VideoAvailTimeCodePanel;
import com.tandbergtv.cms.portal.content.client.title.view.videoplayer.VideoLogicalClipPanel;
import com.tandbergtv.cms.portal.content.client.title.view.videoplayer.VideoSnapshotPanel;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIAssetDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIGroupAssetDefinition;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;

/**
 * The Asset Panel displays a complete asset tree structure with all metadata and files.
 * 
 * @author Tom Rybak
 * @author Vijay Silva
 */
public class AssetPanel extends Composite implements HasViewChangeHandlers 
{
	private List<Widget> offsettingWidgets = new ArrayList<Widget>();
	
	private TitleView titleView;
	private SimpleContainer container;
	private AssetTree assetTree;
	
	private ScrollPanel leftScrollPanel;
	private ScrollPanel rightScrollPanel;
	
	private SimpleContainer assetContent;

	private boolean readOnly = false;
	private AssetInfo input;
	private List<UIAssetFilePath> filePaths;
	private IUIAssetFactory assetFactory = new UIAssetFactory();
	private final Map<AssetInfo, AssetContent> assetInfoWidgets = new HashMap<AssetInfo, AssetContent>();
	private PopupPanel contextMenu;
	private TitleViewMessages constants;
	private ViewChangeHandler handler = new EventHandler();
	private boolean dirty = false;

	private static final String STYLE_NAME = "content-AssetPanel";
	private static final String STYLE_CONTEXT_MENU = "content-AssetPanel-contextMenu";
	private static final String STYLE_CONTEXT_MENU_DISABLED = "content-AssetPanel-contextMenu-disabled";

	private HandlerRegistration windowRegistration;
	
	/**
	 * The Asset Panel Constructor
	 * 
	 * @param input The Asset
	 * @param readOnly Flag indicating if the state on this panel can be edited
	 */
	public AssetPanel(TitleView titleView, AssetInfo input, List<UIAssetFilePath> filePaths, boolean readOnly) 
	{
		this.titleView = titleView;
		this.readOnly = readOnly;
		this.filePaths = filePaths;
		this.input = input;

		initialize();
	}

	public void addOffsettingWidget(Widget widget)
	{
		offsettingWidgets.add(widget);
	}
	
	/*
	 * Initialize the widgets on this asset panel
	 */
	private void initialize() 
	{
		constants = TitleViewMessageCache.getMessages();

		container = new SimpleContainer();
		container.setStyleName(STYLE_NAME);
		initWidget(container);

		/* Initialize the asset tree */
		assetTree = new AssetTree(input);
		leftScrollPanel = new ScrollPanel(assetTree);
		leftScrollPanel.addStyleName("LeftPanel");
		leftScrollPanel.setWidth("250px");
		
		/* Initialize the asset content panel */
		assetContent = new SimpleContainer();
		rightScrollPanel = new ScrollPanel(assetContent);
		rightScrollPanel.addStyleName("RightPanel");
		rightScrollPanel.setWidth("100%");
		
		/* Add the listener to the tree events to update the content panel */
		assetTree.addSelectionHandler(new SelectionHandler<TreeItem>() 
		{
			@Override
			public void onSelection(SelectionEvent<TreeItem> event) 
			{
				AssetContent widget = getAssetContent(event.getSelectedItem());
				assetContent.setWidget(widget);
			}
		});

		/* Build the pop up used for the context menu, and add the context menu listener */
		contextMenu = new PopupPanel(true);
		contextMenu.addStyleName(STYLE_CONTEXT_MENU);
		assetTree.addContextMenuHandler(new ContextMenuHandler() 
		{
			@Override
			public void onContextMenu(ContextMenuEvent event) 
			{
				showContextMenu(event);
			}
		});

		/* The container that contains the asset tree and asset content */
		HorizontalPanel splitContainer = new HorizontalPanel();
		splitContainer.setStyleName("ScrollPanelContainer");
		splitContainer.add(leftScrollPanel);
		splitContainer.add(rightScrollPanel);
		
		container.setWidget(splitContainer);

		/* Select the root node in the asset tree */
		assetTree.setSelectedAssetInfo(input);
	}

	
	/**
	 * Validate the metadata in the view
	 */
	public List<AssetValidationMessage> validate(boolean draft) {
		List<AssetValidationMessage> messages = new ArrayList<AssetValidationMessage>();

		/* Go through the tree nodes in breadth first order */
		List<TreeItem> treeItems = new ArrayList<TreeItem>();
		for (int index = 0; index < assetTree.getItemCount(); index++) {
			treeItems.add(assetTree.getItem(index));
		}

		validate(treeItems, messages, draft);

		return messages;
	}

	/*
	 * Validate each of the assets associated with each tree node in breadth first order
	 */
	private void validate(List<TreeItem> allTreeItems, List<AssetValidationMessage> messages,boolean draft) {
		while (allTreeItems.size() > 0) {
			TreeItem currentTreeItem = allTreeItems.remove(0);
			AssetContent widget = getAssetContent(currentTreeItem);
			messages.addAll(widget.validate(draft));

			for (int index = 0; index < currentTreeItem.getChildCount(); index++) {
				allTreeItems.add(currentTreeItem.getChild(index));
			}
		}
	}

	/**
	 * Set the focus on the widget that the validation message was generated for
	 */
	public void setFocus(AssetValidationMessage message) {
		AssetInfo assetInfo = message.getAsset();
		if (assetInfo != null) {
			assetTree.setSelectedAssetInfo(message.getAsset());
			final AssetContent widget = assetInfoWidgets.get(message.getAsset());
			final Widget fieldWidget = message.getFieldWidget();
			Scheduler.get().scheduleDeferred(new ScheduledCommand() {
				@Override
				public void execute() {
					widget.setFocus(fieldWidget);
				}
			});
			return;
		}
	}

	public AssetInfo getSelectedAssetInfo() {
		return assetTree.getSelectedAssetInfo();
	}

	private void showContextMenu(final ContextMenuEvent event) {
		TreeItem selectedTreeItem = assetTree.getSelectedItem();
		AssetInfo selectedAssetInfo = assetTree.getSelectedAssetInfo();

		/* No context menu if the tree is read-only */
		if (this.readOnly || selectedAssetInfo == null)
			return;

		/* Clear all existing menu items */
		contextMenu.clear();

		/* Create the Menu Items */
		boolean hasItems = false;
		final MenuBar menuBar = new MenuBar(true);
		final MenuBar addItemBar = new MenuBar(true);
		UIAssetDefinition assetDefinition = selectedAssetInfo.getDefinition();
		if (assetDefinition instanceof UIGroupAssetDefinition) {
			UIGroupAssetDefinition groupAssetDefinition = (UIGroupAssetDefinition) assetDefinition;
			UIGroupAsset asset = (UIGroupAsset) selectedAssetInfo.getAsset();
			boolean hasAddMenuItem = false;
			for (UIAssetDefinition assetDef : groupAssetDefinition.getChildren()) {
				String assetType = assetDef.getAssetType();
				int existingCount = asset.getChildItems(assetType).size();
				int maximumCount = assetDef.getMaximumCount();
				if(input.isBatchEdit()) {
					maximumCount = assetDef.getBatchMaximumCount();
				}
				int amountLeft = maximumCount - existingCount;

				// ensure that the maximum number of these assets haven't been added
				if (amountLeft <= 0)
					continue;

				hasAddMenuItem = true;
				Command addCommand = new AddAssetCommand(selectedTreeItem, asset, assetDef);
				addItemBar.addItem(assetDef.getDisplayName(), addCommand);
			}

			if (hasAddMenuItem) {
				menuBar.addItem(new MenuItem(constants.assetTreeMenuItemAddText(), addItemBar));
				hasItems = true;
			}
		}

		/* Determine if remove item should show */
		int minimumCount = assetDefinition.getMinimumCount();
		if(input.isBatchEdit()) {
			minimumCount = assetDefinition.getBatchMinimumCount();
		}
		UIAsset asset = selectedAssetInfo.getAsset();
		UIGroupAsset parent = (UIGroupAsset) asset.getParentAsset();
		if (parent != null && parent.getChildItems(asset.getAssetType()).size() > minimumCount) {
			String text = constants.assetTreeMenuItemRemoveText();
			menuBar.addItem(new MenuItem(text, new RemoveAssetCommand(selectedTreeItem)));
			hasItems = true;
		}

		/* If image asset type, show "Set Image from key frames" and "Set Image from Snapshots" menu options */
		NeptuneApplication application = NeptuneApplication.getApplication();
		ContentComponent component = application.getComponent(ContentComponent.class);
		List<String> imageAssetTypes = component.getImageAssetTypes();
		boolean isLicensed = component.isLicensedForVideoPreview(); 

		String assetType = asset.getAssetType();
		if (imageAssetTypes.contains(assetType)) {
			String text1 = constants.assetTreeMenuItemSetImageFromKeyframesText();
			menuBar.addItem(new MenuItem(text1, new SetImageFromKeyframesCommand(selectedTreeItem)));
			String text2 = constants.assetTreeMenuItemSetImageFromSnapshotsText();
			if (isLicensed) {
				menuBar.addItem(new MenuItem(text2, new SetImageFromSnapshotsCommand(selectedTreeItem)));
			}
			else {
				MenuItem disabledItem = new MenuItem(text2, new DoNothingCommand(selectedTreeItem));
				disabledItem.setStyleName(STYLE_CONTEXT_MENU_DISABLED);
				menuBar.addItem(disabledItem);				
			}
			hasItems = true;
		}

		if ("LOGICALCLIP".equals(assetType)) {
			String text1 = constants.assetTreeMenuItemSetMarkInMarkOutFromKeyframesText();
			String text2 = constants.assetTreeMenuItemSetMarkInMarkOutFromVideoText();
			menuBar.addItem(new MenuItem(text1, new SetMarkInMarkOutFromKeyframesCommand(selectedTreeItem)));
			if (isLicensed) {
				menuBar.addItem(new MenuItem(text2, new SetMarkInMarkOutFromVideoCommand(selectedTreeItem)));
			}
			else {
				MenuItem disabledItem = new MenuItem(text2, new DoNothingCommand(selectedTreeItem));
				disabledItem.setStyleName(STYLE_CONTEXT_MENU_DISABLED);
				menuBar.addItem(disabledItem);
			}
			hasItems = true;
		}

		/* If video asset type, show "Create Image from videos" menu option */
		List<String> videoAssetTypes = component.getVideoAssetTypes();
		if (videoAssetTypes.contains(assetType)) {
			String text = constants.assetTreeMenuItemCreateImageFromVideoText();
			String availText = constants.assetTreeMenuItemSetAvailTimeCodesFromVideoText();
			if (isLicensed) {
				menuBar.addItem(new MenuItem(text, new CreateImageFromVideoCommand(selectedTreeItem)));
				menuBar.addItem(new MenuItem(availText, new SetAvailTimeCodesFromVideoCommand(selectedTreeItem)));
			}
			else {
				MenuItem disabledItem = new MenuItem(text, new DoNothingCommand(selectedTreeItem));
				disabledItem.setStyleName(STYLE_CONTEXT_MENU_DISABLED);
				menuBar.addItem(disabledItem);	
				MenuItem disabledAvailItem = new MenuItem(availText, new DoNothingCommand(selectedTreeItem));
				disabledAvailItem.setStyleName(STYLE_CONTEXT_MENU_DISABLED);
				menuBar.addItem(disabledAvailItem);	
			}
			hasItems = true;
		}
		
		/* Prevent the browser context menu from showing */
		event.preventDefault();
		event.stopPropagation();

		/* Don't show context menu if no items present */
		if (hasItems) {
			contextMenu.add(menuBar);

			int x = event.getNativeEvent().getClientX() + Window.getScrollLeft();
			int y = event.getNativeEvent().getClientY() + Window.getScrollTop();

			/* For QA simulation test cases, the x, y coordinates may be less than 0 */
			if (x < 0) {
				x = 0;
			}

			if (y < 0) {
				y = 0;
			}

			contextMenu.setPopupPosition(x, y);
			contextMenu.show();
		}
	}

	/*
	 * Get the asset content widget for the given tree item
	 */
	private AssetContent getAssetContent(TreeItem treeItem) {
		return getAssetContent(assetTree.getAssetInfo(treeItem));
	}
	
	/*
	 * Get the asset content widget for the given asset info object
	 */
	private AssetContent getAssetContent(AssetInfo assetInfo) {
		AssetContent widget = assetInfoWidgets.get(assetInfo);
		if (widget == null && assetInfo != null) {
			widget = new AssetContent(titleView, assetInfo, filePaths, readOnly);
			if (this.dirty)
				widget.markDirty();
			widget.addViewChangeHandler(handler);
			assetInfoWidgets.put(assetInfo, widget);
		}

		return widget;
	}
	
	/**
	 * 
	 */
	public void redraw()
	{
		showWidgetInRightScrollPanel(assetContent);
		updateSize();
	}

	/*
	 * Set the widget to show in the scroll container
	 */
	private void showWidgetInRightScrollPanel(Widget widget) 
	{
		rightScrollPanel.setWidget(widget);
	}
	
	/**
	 * 1. Clear assetContent for the given assetInfo from assetInfoWidgets
	 * 2. update the content panel.
	 * 
	 * @param assetInfo
	 */
	public void redraw(AssetInfo assetInfo) {
		/* Discard the old cached widget for the asset info, re-create and display the new widget */
		assetInfoWidgets.put(assetInfo, null);
		assetContent.setWidget(getAssetContent(assetInfo));

		/* Redraw the view */ 
		redraw();
		
		/* Fire change in the model */
		fireViewChangeEvent();
	}


	private void fireViewChangeEvent() {
		fireEvent(new ViewChangeEvent());
		this.dirty = true;

		/* Update all child widgets as dirty */
		for (AssetContent content : assetInfoWidgets.values()) {
			if (content != null)
				content.markDirty();
		}
	}

	// ========================================================================
	// ===================== MENU COMMANDS
	// ========================================================================

	private final class AddAssetCommand implements Command {
		private final TreeItem parentTreeItem;
		private final UIGroupAsset parentAsset;
		private final UIAssetDefinition assetDef;

		public AddAssetCommand(TreeItem treeItem, UIGroupAsset parentAsset,
		        UIAssetDefinition assetDefinition) {
			this.parentTreeItem = treeItem;
			this.parentAsset = parentAsset;
			this.assetDef = assetDefinition;
		}

		public void execute() {
			UIAsset newAsset = assetFactory.createAsset(assetDef, true, input.isBatchEdit());
			parentAsset.addChild(newAsset);
			AssetInfo assetInfo = new AssetInfo(input.getInput(), newAsset, assetDef, input.isBatchEdit());
			assetTree.addTreeItem(assetInfo, parentTreeItem);
			contextMenu.hide();
			fireViewChangeEvent();
		}
	}

	private final class RemoveAssetCommand implements Command {
		private final TreeItem treeItem;

		public RemoveAssetCommand(TreeItem treeItem) {
			this.treeItem = treeItem;
		}

		public void execute() {
			AssetInfo object = assetTree.getAssetInfo(treeItem);
			AssetInfo parentObject = assetTree.getAssetInfo(treeItem.getParentItem());

			/* Remove asset from the model and view */
			((UIGroupAsset) parentObject.getAsset()).removeChild(object.getAsset());
			removeTreeItem(treeItem);

			/* Change the selection to the parent and hide context menu */
			assetTree.setSelectedAssetInfo(parentObject);
			contextMenu.hide();

			fireViewChangeEvent();
		}

		private void removeTreeItem(TreeItem treeItem) {
			for (int index = 0; index < treeItem.getChildCount(); index++) {
				removeTreeItem(treeItem.getChild(index));
			}

			treeItem.remove();
			assetInfoWidgets.remove(assetTree.getAssetInfo(treeItem));
		}
	}
	
	private final class SetImageFromKeyframesCommand implements Command {
		private final TreeItem treeItem;

		public SetImageFromKeyframesCommand(TreeItem treeItem) {
			this.treeItem = treeItem;
		}

		public void execute() {
			AssetInfo object = assetTree.getAssetInfo(treeItem);

			//do not change order
			KeyframesSetImagePanel keyframesSetImagePanel = new KeyframesSetImagePanel();
			showWidgetInRightScrollPanel(keyframesSetImagePanel);
			keyframesSetImagePanel.setAssetInfo(object);
			keyframesSetImagePanel.setAssetPanel(AssetPanel.this);
			
			contextMenu.hide();
		}

	}

	private final class SetMarkInMarkOutFromKeyframesCommand implements Command {
		private final TreeItem treeItem;

		public SetMarkInMarkOutFromKeyframesCommand(TreeItem treeItem) {
			this.treeItem = treeItem;
		}

		public void execute() {
			AssetInfo object = assetTree.getAssetInfo(treeItem);

			KeyframesLogicalClipPanel keyframesLogicalClipPanel = new KeyframesLogicalClipPanel();
			showWidgetInRightScrollPanel(keyframesLogicalClipPanel);
			keyframesLogicalClipPanel.setAssetPanel(AssetPanel.this);
			keyframesLogicalClipPanel.setAssetInfo(object);
			
			contextMenu.hide();
		}

	}

	private final class SetMarkInMarkOutFromVideoCommand implements Command {
		private final TreeItem treeItem;

		public SetMarkInMarkOutFromVideoCommand(TreeItem treeItem) {
			this.treeItem = treeItem;
		}

		public void execute() {
			AssetInfo object = assetTree.getAssetInfo(treeItem);

			VideoLogicalClipPanel videoLogicalClipPanel = new VideoLogicalClipPanel();
			showWidgetInRightScrollPanel(videoLogicalClipPanel);
			videoLogicalClipPanel.setTitleView(titleView);
			videoLogicalClipPanel.setAssetPanel(AssetPanel.this);
			videoLogicalClipPanel.setAssetInfo(object);
			
			contextMenu.hide();
		}

	}
	
	private final class CreateImageFromVideoCommand implements Command {
		private final TreeItem treeItem;

		public CreateImageFromVideoCommand(TreeItem treeItem) {
			this.treeItem = treeItem;
		}

		public void execute() {
			AssetInfo object = assetTree.getAssetInfo(treeItem);

			VideoSnapshotPanel videoSnapshotPanel = new VideoSnapshotPanel();
			showWidgetInRightScrollPanel(videoSnapshotPanel);
			videoSnapshotPanel.setTitleView(titleView);
			videoSnapshotPanel.setAssetPanel(AssetPanel.this);
			videoSnapshotPanel.setAssetInfo(object);
			
			contextMenu.hide();
		}
	}
	
	private final class SetAvailTimeCodesFromVideoCommand implements Command {
		private final TreeItem treeItem;

		public SetAvailTimeCodesFromVideoCommand(TreeItem treeItem) {
			this.treeItem = treeItem;
		}

		public void execute() {
			AssetInfo object = assetTree.getAssetInfo(treeItem);

			VideoAvailTimeCodePanel videoAvailTimeCodePanel = new VideoAvailTimeCodePanel();
			showWidgetInRightScrollPanel(videoAvailTimeCodePanel);
			videoAvailTimeCodePanel.setTitleView(titleView);
			videoAvailTimeCodePanel.setAssetPanel(AssetPanel.this);
			videoAvailTimeCodePanel.setAssetInfo(object);
			
			contextMenu.hide();
		}
	}

	private final class SetImageFromSnapshotsCommand implements Command {
		private final TreeItem treeItem;

		public SetImageFromSnapshotsCommand(TreeItem treeItem) {
			this.treeItem = treeItem;
		}

		public void execute() {
			AssetInfo object = assetTree.getAssetInfo(treeItem);

			//do not change order
			SetVideoImagePanel setImageFromSnapshotsPanel = new SetVideoImagePanel();
			showWidgetInRightScrollPanel(setImageFromSnapshotsPanel);
			setImageFromSnapshotsPanel.setAssetInfo(object);
			setImageFromSnapshotsPanel.setAssetPanel(AssetPanel.this);
				
			contextMenu.hide();
		}		
	}
	

	private final class DoNothingCommand implements Command {

		public DoNothingCommand(TreeItem treeItem) {
		}

		public void execute() {
		}
	}

	/**
	 * @return the input
	 */
	public AssetInfo getInput() {
		return input;
	}


	// ==============================================================
	// ===================== WIDGET OVERRIDES
	// ==============================================================

	@Override
	protected void onLoad() 
	{
		super.onLoad();

		windowRegistration = Window.addResizeHandler(new ResizeHandler() 
		{
			public void onResize(ResizeEvent event) 
			{
				updateSize();
			}
		});
	};

	@Override
	protected void onUnload() 
	{
		windowRegistration.removeHandler();
		windowRegistration = null;

		super.onUnload();
	}

	
	public void updateSize()
	{
		Scheduler.get().scheduleDeferred(new ScheduledCommand() 
		{
			@Override
			public void execute() 
			{
				int height = Window.getClientHeight() - 265;
				
				for(Widget w : offsettingWidgets)
				{
					height -= w.getOffsetHeight();
				}
				
				leftScrollPanel.setHeight(height + "px");
				rightScrollPanel.setHeight(height + "px");
			}
		});
	}
	
	
	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	@Override
	public HandlerRegistration addViewChangeHandler(ViewChangeHandler handler) 
	{
		return addHandler(handler, ViewChangeEvent.getType());
	}

	/*
	 * Internal event handler class for events handled by internal widgets
	 */
	private class EventHandler implements ViewChangeHandler 
	{
		@Override
		public void onViewChange(ViewChangeEvent event) 
		{
			/* Pass the event up */
			fireViewChangeEvent();
		}
	}
}
