package com.tandbergtv.cms.portal.content.client.title.view.asset;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.google.gwt.dom.client.Document;
import com.google.gwt.dom.client.NativeEvent;
import com.google.gwt.event.dom.client.ContextMenuEvent;
import com.google.gwt.event.dom.client.ContextMenuHandler;
import com.google.gwt.event.dom.client.HasContextMenuHandlers;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.ui.TreeItem;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIAsset;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewInput;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIAssetDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIGroupAssetDefinition;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TreeWidget;

public class AssetTree extends TreeWidget implements HasContextMenuHandlers 
{

	private static final String STYLE_NAME = "content-AssetTree"; 
	protected AssetTreeItemBuilder itemBuilder = new AssetTreeItemBuilder();

	/**
	 * Construct an editable asset tree
	 * 
	 * @param rootAsset The root asset in the tree
	 */
	public AssetTree(AssetInfo rootAsset) {
		addStyleName(STYLE_NAME);
		
		/* Initialize the root node and child tree node */
		TreeItem rootItem = createItem(rootAsset);
		addItem(rootItem);
	}

	/**
	 * Get the asset info object associated with this tree item.
	 * 
	 * @param treeItem The tree item
	 * @return The Asset Info object
	 */
	public AssetInfo getAssetInfo(TreeItem treeItem) {
		AssetInfo assetInfo = null;
		if (treeItem != null && (treeItem.getUserObject() instanceof AssetInfo))
			assetInfo = (AssetInfo) treeItem.getUserObject();
		return assetInfo;
	}

	/**
	 * Get the selected asset info object in the tree.
	 * 
	 * @param assetInfo The selected asset info object.
	 */
	public AssetInfo getSelectedAssetInfo() {
		return getAssetInfo(getSelectedItem());
	}

	/**
	 * Set the selected asset info object in the tree. Does nothing if the asset info object is not
	 * present in the tree.
	 * 
	 * @param assetInfo The asset info object to select.
	 */
	public void setSelectedAssetInfo(AssetInfo assetInfo) {
		if (assetInfo == null)
			return;

		TreeItem item = findItem(assetInfo);
		if (item != null) {
			setSelectedItem(item);
		}
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	// GWT bug/feature workaround. Fire left-click event after right-click, to trigger tree item select event.
	@Override
	public void onBrowserEvent(Event event) 
	{
		int evType = DOM.eventGetType(event);
		switch(evType) 
		{
		case Event.ONMOUSEDOWN:
		    if(DOM.eventGetButton(event) == Event.BUTTON_RIGHT) 
		    {
		    	NativeEvent nativeEvent = Document.get().createMouseDownEvent(
		    			-1, 
		    			event.getScreenX(),
		    		 	event.getScreenY(), 
		    		 	event.getClientX(), 
		    		 	event.getClientY(), 
		    		 	event.getCtrlKey(),
		    		 	event.getAltKey(), 
		    		 	event.getShiftKey(), 
		    		 	event.getMetaKey(), 
		    		 	NativeEvent.BUTTON_LEFT);
		    		 	
		    	DOM.eventGetTarget(event).dispatchEvent(nativeEvent);
		    }
		    else 
		    {
		    	super.onBrowserEvent(event);
		    }
		    break;
		default:
			super.onBrowserEvent(event);
			break;
		}
	}
	

	@Override
	public HandlerRegistration addContextMenuHandler(ContextMenuHandler handler) {
		return this.addDomHandler(handler, ContextMenuEvent.getType());
	}

	// ========================================================================
	// ===================== TREE ITEM MANIPULATION
	// ========================================================================

	/**
	 * Add new tree item to the tree
	 * 
	 * @param assetInfo The new asset to add to the tree
	 * @param parentAssetInfo The parent asset contained in a tree item
	 */
	public void addTreeItem(AssetInfo assetInfo, AssetInfo parentAssetInfo) {
		addTreeItem(assetInfo, findItem(parentAssetInfo));
	}

	/**
	 * Add new tree item to the tree
	 * 
	 * @param assetInfo The new asset to add to the tree
	 * @param parentAssetInfo The parent asset contained in a tree item
	 */
	public void addTreeItem(AssetInfo assetInfo, TreeItem parentTreeItem) {
		/* Remember the previous selection */
		TreeItem oldSelection = getSelectedItem();

		/* Build the new item to add */
		TreeItem newTreeItem = createItem(assetInfo);

		/* Build a list of asset type names that must precede this new asset. */
		AssetInfo parentAssetInfo = getAssetInfo(parentTreeItem);
		String assetType = assetInfo.getDefinition().getAssetType();
		UIGroupAssetDefinition definition = (UIGroupAssetDefinition) parentAssetInfo
		        .getDefinition();
		Set<String> precedingAssetTypes = new HashSet<String>();
		for (UIAssetDefinition childDefinition : definition.getChildren()) {
			precedingAssetTypes.add(childDefinition.getAssetType());
			if (childDefinition.getAssetType().equals(assetType)) {
				break;
			}
		}

		/*
		 * GWT Tree API does not support insert tree item at specified index, need to remove all
		 * tree nodes, and add back in correct order to insert. If API changes to support insert,
		 * remove this complex logic.
		 * 
		 * Find location of the asset with asset type name not in the list of preceding assets, and
		 * insert the new item at this location
		 */
		List<TreeItem> childTreeItems = new ArrayList<TreeItem>();
		boolean inserted = false;
		for (int index = 0; index < parentTreeItem.getChildCount(); index++) {
			TreeItem childTreeItem = parentTreeItem.getChild(index);
			String currentAssetType = getAssetInfo(childTreeItem).getDefinition().getAssetType();
			if (!inserted && !precedingAssetTypes.contains(currentAssetType)) {
				/* Found a name not in 'preceding asset type' set. Insert new node here */
				childTreeItems.add(newTreeItem);
				inserted = true;
			}

			/* Keep adding all current tree items to the list */
			childTreeItems.add(childTreeItem);
		}

		/* Check if the new item is not inserted in the list, and insert */
		if (!inserted)
			childTreeItems.add(newTreeItem);

		/* Remove all items, and re-add with new item at the correct index */
		parentTreeItem.removeItems();
		for (TreeItem childTreeItem : childTreeItems)
			parentTreeItem.addItem(childTreeItem);

		/* Ensure that the selected item stays selected */
		if (oldSelection != null && oldSelection != getSelectedItem())
			setSelectedItem(oldSelection);
	}

	/*
	 * Creates a new tree item that can be added to the tree
	 */
	protected TreeItem createItem(AssetInfo assetInfo) {
		return itemBuilder.buildTreeItem(assetInfo);
	}

	// ========================================================================
	// ===================== INTERNAL METHODS
	// ========================================================================

	/*
	 * Finds a tree item that matches the asset info object anywhere in the tree. Returns null if no
	 * match is found
	 */
	private TreeItem findItem(AssetInfo assetInfo) {
		for (int i = 0; i < getItemCount(); i++) {
			TreeItem item = findItem(assetInfo, getItem(i));
			if (item != null) {
				return item;
			}
		}

		return null;
	}

	/*
	 * Finds a tree item that matches the asset info object in or under the current tree item
	 */
	private TreeItem findItem(AssetInfo assetInfo, TreeItem item) {
		TreeItem matchedItem = null;

		AssetInfo object = getAssetInfo(item);
		if (assetInfo.equals(object)) {
			matchedItem = item;
		} else {
			for (int i = 0; i < item.getChildCount(); i++) {
				matchedItem = findItem(assetInfo, item.getChild(i));
				if (matchedItem != null)
					break;
			}
		}

		return matchedItem;
	}

	// ========================================================================
	// ===================== INTERNAL CLASSES
	// ========================================================================

	public final static class AssetInfo {
		private final TitleViewInput input;
		private final UIAsset asset;
		private final UIAssetDefinition assetDefinition;
		private final boolean isBatchEdit;

		/**
		 * Constructor
		 * @param asset
		 * @param assetDefinition
		 * @param contentClassMetadata
		 */
		public AssetInfo(TitleViewInput input, UIAsset asset, UIAssetDefinition assetDefinition) {
			this(input, asset, assetDefinition, false);
		}

		/**
		 * Constructor
		 * @param asset
		 * @param assetDefinition
		 * @param contentClassMetadata
		 * @param isBatchEdit
		 */
		public AssetInfo(TitleViewInput input, UIAsset asset, UIAssetDefinition assetDefinition, boolean isBatchEdit) {
			this.input = input;
			this.asset = asset;
			this.assetDefinition = assetDefinition;
			this.isBatchEdit = isBatchEdit;
		}

		public UIAsset getAsset() {
			return asset;
		}

		public UIAssetDefinition getDefinition() {
			return assetDefinition;
		}

		public boolean isBatchEdit() {
			return isBatchEdit;
		}

		/**
		 * @return the input
		 */
		public TitleViewInput getInput() {
			return input;
		}

		@Override
		public boolean equals(Object other) {
			if (other == null || !(other instanceof AssetInfo)) return false;
			return ((AssetInfo) other).getAsset().equals(getAsset());
		}
	}
}
