/*
 * Created on Aug 24, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.view.asset.field;

import java.util.ArrayList;
import java.util.List;

import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIComplexField;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIField;
import com.tandbergtv.cms.portal.content.client.title.service.asset.IUIAssetFactory;
import com.tandbergtv.cms.portal.content.client.title.view.asset.MetadataTypeMismatchException;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIComplexFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIFieldDefinition;

/**
 * Handler that deals with the collapsing of a field given its parent complex field definition, the
 * parent field, and the definition of the field to be collapsed.
 * 
 * @author Vijay Silva
 */
public class CollapsedFieldHandler {

	private final IUIAssetFactory assetFactory;
	private final UIComplexFieldDefinition ancestorDefinition;
	private final UIFieldDefinition targetDefinition;

	/**
	 * Constructor
	 * 
	 * @param factory The asset factory to build fields
	 * @param ancestorDefinition The ancestor field definition for target field definition.
	 * @param targetDefinition The target field definition which has been collapsed.
	 */
	public CollapsedFieldHandler(IUIAssetFactory factory,
	        UIComplexFieldDefinition ancestorDefinition, UIFieldDefinition targetDefinition) {
		this.assetFactory = factory;
		this.ancestorDefinition = ancestorDefinition;
		this.targetDefinition = targetDefinition;
	}

	/**
	 * Get the parent of the collapsed field that matches the target definition given the ancestor
	 * field which should match the ancestor definition.
	 * 
	 * @param ancestor The ancestor field
	 * @return The parent of the child field matching the target definition
	 */
	public UIComplexField getCollapsedFieldParent(UIComplexField ancestor) {
		/* No ancestor present, return null */
		if (ancestor == null)
			return null;

		/* Build the reverse field tree path from the target to the ancestor */
		List<UIFieldDefinition> reversePath = new ArrayList<UIFieldDefinition>();
		UIFieldDefinition parentDefinition = targetDefinition.getParent();
		while (!this.ancestorDefinition.equals(parentDefinition) && parentDefinition != null) {
			reversePath.add(parentDefinition);
			parentDefinition = parentDefinition.getParent();
		}

		/* Verify that the path was built successfully */
		if (!this.ancestorDefinition.equals(parentDefinition)) {
			String message = "Failed to build the reverse path from the target field definition: "
			        + targetDefinition.getXPath() + " to ancestor field definition: "
			        + ancestorDefinition.getXPath();
			throw new RuntimeException(message);
		}

		/* Find or build the collapsed field */
		UIComplexField parent = ancestor;
		for (int index = reversePath.size() - 1; index >= 0; index--) {
			UIFieldDefinition currentDefinition = reversePath.get(index);
			UIField child = parent.getChildren().getField(currentDefinition.getName());
			if (child == null) {
				/* No child found, build the child and add to the model */
				child = assetFactory.createAssetField(currentDefinition);
				parent.addField(child);
			}

			/* Verify the type of the expected collapsed complex field */
			if (child instanceof UIComplexField) {
				parent = (UIComplexField) child;
			} else {
				String name = currentDefinition.getName();
				throw new MetadataTypeMismatchException(name, currentDefinition, child);
			}
		}

		return parent;
	}
}
