/*
 * Created on Jul 27, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.view.asset.field.complex;

import java.util.List;

import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.ui.ButtonBase;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIComplexField;
import com.tandbergtv.cms.portal.content.client.title.service.asset.IUIAssetFactory;
import com.tandbergtv.cms.portal.content.client.title.service.asset.UIAssetFactory;
import com.tandbergtv.cms.portal.content.client.title.view.asset.AssetValidationMessage;
import com.tandbergtv.cms.portal.content.client.title.view.asset.MetadataTypeMismatchException;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIComplexFieldDefinition;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.DisclosureContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.RoundedDisclosureContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

/**
 * The Widget for a group of complex fields of the same definition
 * 
 * @author Vijay Silva
 */
public class ComplexFieldGroupContainer extends ComplexFieldGroupWidget {

	/* Widgets */
	private VerticalContainer container;

	/* Model */
	private IUIAssetFactory assetFactory = new UIAssetFactory();

	/* Style Name */
	private static final String STYLE_NAME = "content-ComplexFieldGroupContainer";
	private static final String STYLE_ERROR_CONTAINER = "content-ComplexFieldGroupContainer-errorDisclosureContainer";

	/**
	 * @param fieldDefinition The complex field definition
	 * @param readOnly flag indicating if the widget is read-only or not
	 */
	public ComplexFieldGroupContainer(UIComplexFieldDefinition fieldDefinition, boolean readOnly) {
		super(fieldDefinition, readOnly);

		/* Initialize the Widget */
		container = new VerticalContainer();
		container.addStyleName(STYLE_NAME);
		initWidget(container);
	}

	// ========================================================================
	// ===================== REFRESHING VIEW
	// ========================================================================

	/**
	 * Refresh the view
	 */
	@Override
	public void refresh() {
		/* Remove any previous widgets */
		container.clear();

		/* There is no input, do nothing */
		if (getParentField() == null)
			return;

		/* Get the list of fields */
		List<UIComplexField> inputFields = null;
		try {
			inputFields = getFields();
		} catch (MetadataTypeMismatchException mtme) {
			reportMismatchError(mtme);
			return;
		}

		/* Add the widgets for each of the to render */
		for (UIComplexField field : inputFields) {
			Widget widget = buildComplexFieldWidget(field);
			container.add(widget);
		}

		/* Build the new field widget if required */
		addNewFieldWidgets();
	}

	/*
	 * Build a widget to render a single complex field
	 */
	private Widget buildComplexFieldWidget(UIComplexField field) {
		/* Build the widget for the complex field */
		final ComplexFieldWidget widget = new ComplexFieldWidget(getFieldDefinition(), isReadOnly(), getParentAsset().isBatchEdit());
		widget.addViewChangeHandler(getViewChangeEventDelegator());
		widget.setInput(getParentAsset(), field);

		/* Manipulate the buttons */
		boolean visible = (!isReadOnly() && getFieldDefinition().getMaximumCount() != 1);
		ButtonBase addButton = widget.getAddButton();
		addButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleAddButtonClick(widget);
			}
		});
		addButton.setVisible(visible);
		addButton.setEnabled(visible);

		ButtonBase removeButton = widget.getRemoveButton();
		removeButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleRemoveButtonClick(widget);
			}
		});
		removeButton.setVisible(visible);
		removeButton.setEnabled(visible);

		return widget;
	}

	/*
	 * Add a 'new field' widget if required.
	 */
	private void addNewFieldWidgets() {
		/* Check if at least one complex field widget is available */
		int fieldCount = getFieldCount();
		int minCount = getFieldDefinition().getMinimumCount();
		if (minCount <= 0)
			minCount = 1;

		for (int count = fieldCount; count < minCount; count++) {
			/* Create the new field and add to the input / model */
			UIComplexField newField = createNewField();

			/* build the widget and add to view */
			Widget widget = buildComplexFieldWidget(newField);
			container.add(widget);
		}
	}

	/*
	 * Create a new field and add to the model
	 */
	private UIComplexField createNewField() {
		UIComplexFieldDefinition definition = getFieldDefinition();
		UIComplexField newField = (UIComplexField) assetFactory.createAssetField(definition);
		getParentField().addField(newField);
		return newField;
	}

	// ========================================================================
	// ===================== BUTTON ACTIONS
	// ========================================================================

	/*
	 * The add button is clicked on a complex field widget
	 */
	private void handleAddButtonClick(ComplexFieldWidget widget) {
		if (this.isReadOnly())
			return;

		/* Must add new field after current widget */
		int index = container.getWidgetIndex(widget);
		if (index == -1)
			return;

		/* Check if the field can be added */
		int maxCount = getFieldDefinition().getMaximumCount();
		if (maxCount != -1 && maxCount <= getFieldCount())
			return;

		/* Create the new field and add to the input / model */
		UIComplexField newField = createNewField();

		/* Build the widget and add to view */
		Widget newWidget = buildComplexFieldWidget(newField);
		container.insert(newWidget, index + 1);
		fireViewChangeEvent();
	}

	/*
	 * The remove button is clicked on a complex field widget
	 */
	private void handleRemoveButtonClick(ComplexFieldWidget widget) {
		if (this.isReadOnly())
			return;

		int index = container.getWidgetIndex(widget);
		if (index == -1)
			return;

		/* Remove from the view and from model */
		widget.removeFromParent();
		getParentField().removeField(widget.getField());
		fireViewChangeEvent();

		/* Determine if new field needs to be added */
		addNewFieldWidgets();
	}

	// ========================================================================
	// ===================== METADATA VALIDATION
	// ========================================================================

	@Override
	public void validate(List<AssetValidationMessage> messages,boolean draft) {
		for (Widget widget : container) {
			/* Check if widget is complex field widget, and validate */
			if (widget instanceof ComplexFieldWidget) {
				ComplexFieldWidget complexFieldWidget = (ComplexFieldWidget) widget;
				complexFieldWidget.validate(messages,draft);
			}
		}
	}

	// ========================================================================
	// ===================== ERROR HANDLING
	// ========================================================================

	/**
	 * Report the mismatch error resulting from mismatch in definition and data
	 */
	public void reportMismatchError(MetadataTypeMismatchException mtme) {
		String heading = getFieldDefinition().getDisplayName();
		DisclosureContainer disclosure = new RoundedDisclosureContainer(heading, true);
		disclosure.addStyleName(STYLE_ERROR_CONTAINER);
		Widget errorWidget = this.buildMetadataMismatchErrorWidget(mtme);
		disclosure.setContent(errorWidget);
		container.add(disclosure);
	}
}
