/*
 * Created on Jul 27, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.view.asset.field.complex;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIComplexField;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIField;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UISimpleField;
import com.tandbergtv.cms.portal.content.client.title.view.HasViewChangeHandlers;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewMessageCache;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewMessages;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewResources;
import com.tandbergtv.cms.portal.content.client.title.view.ViewChangeEvent;
import com.tandbergtv.cms.portal.content.client.title.view.ViewChangeHandler;
import com.tandbergtv.cms.portal.content.client.title.view.asset.AssetTree.AssetInfo;
import com.tandbergtv.cms.portal.content.client.title.view.asset.AssetValidationMessage;
import com.tandbergtv.cms.portal.content.client.title.view.asset.MetadataTypeMismatchException;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIComplexFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UISimpleFieldDefinition;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;

/**
 * A widget for a collection of complex fields defined using a complex field definition
 * 
 * @author Vijay Silva
 */
public abstract class ComplexFieldGroupWidget extends Composite implements HasViewChangeHandlers {

	/* Properties */
	private UIComplexFieldDefinition fieldDefinition;
	private boolean readOnly = false;
	private AssetInfo parentAsset;
	private UIComplexField parentField;
	private TitleViewMessages messages = TitleViewMessageCache.getMessages();
	private ViewChangeHandler viewChangeEventDelegator = new ViewChangeEventHandler();

	/**
	 * Constructor
	 * 
	 * @param fieldDefinition The field definition for fields displayed in this group
	 * @param readOnly The read only flag
	 */
	public ComplexFieldGroupWidget(UIComplexFieldDefinition fieldDefinition, boolean readOnly) {
		this.fieldDefinition = fieldDefinition;
		this.readOnly = readOnly;
	}

	// ========================================================================
	// ===================== UPDATING INPUT
	// ========================================================================

	/**
	 * Set the input asset for this view
	 * 
	 * @param asset The asset for which the fields need to be displayed
	 * @param parentField The parent complex field that contains fields displayed by this group
	 */
	public void setInput(AssetInfo asset, UIComplexField parentField) {
		this.parentAsset = asset;
		this.parentField = parentField;

		refresh();
	}

	/**
	 * Determine if this widget is read-only
	 * 
	 * @return true if the widget is read-only, false otherwise
	 */
	public boolean isReadOnly() {
		return this.readOnly;
	}

	/**
	 * @return the fieldDefinition
	 */
	public UIComplexFieldDefinition getFieldDefinition() {
		return fieldDefinition;
	}

	/**
	 * @return the parent asset
	 */
	public AssetInfo getParentAsset() {
		return parentAsset;
	}

	/**
	 * @return the parent field
	 */
	public UIComplexField getParentField() {
		return parentField;
	}

	/**
	 * Get all the complex fields to render for this group
	 */
	protected List<UIComplexField> getFields() throws MetadataTypeMismatchException {
		if (parentField == null)
			return null;

		List<UIComplexField> inputFields = new ArrayList<UIComplexField>();
		String name = fieldDefinition.getName();
		for (UIField field : parentField.getChildren().getFields(name)) {
			if (field instanceof UIComplexField) {
				inputFields.add((UIComplexField) field);
			} else {
				throw new MetadataTypeMismatchException(name, fieldDefinition, field);
			}
		}

		return inputFields;
	}

	/**
	 * Get a count of the fields which match the current definition
	 * 
	 * @return the field count
	 */
	protected int getFieldCount() {
		String fieldName = getFieldDefinition().getName();
		int fieldCount = 0;
		if (getParentField() != null)
			fieldCount = getParentField().getChildren().getFieldCount(fieldName);

		return fieldCount;
	}

	/**
	 * Get the messages
	 * 
	 * @return The view messages
	 */
	protected TitleViewMessages getViewMessages() {
		return this.messages;
	}

	// ========================================================================
	// ===================== REFRESHING VIEW
	// ========================================================================

	/**
	 * Refresh the view.
	 */
	public abstract void refresh();

	// ========================================================================
	// ===================== METADATA VALIDATION
	// ========================================================================

	/**
	 * Validate the widgets, and return validation messages for each of the metadata widgets that
	 * fail validation.
	 * 
	 * @param messages the list of validation messages to populate
	 */
	public abstract void validate(List<AssetValidationMessage> messages, boolean draft);

	// ========================================================================
	// ===================== ERROR HANDLING
	// ========================================================================

	/**
	 * Report the mismatch error resulting from mismatch in definition and data
	 */
	public abstract void reportMismatchError(MetadataTypeMismatchException mtme);

	/**
	 * Builds a label widget that can be used to display a error message resulting from a data type
	 * mismatch.
	 */
	protected Widget buildMetadataMismatchErrorWidget(MetadataTypeMismatchException mtme) {
		String typeName = getFieldTypeName(mtme.getFieldDefinition());
		String receivedTypeName = getFieldTypeName(mtme.getReceivedField());
		String message = getViewMessages().metadataTypeMismatchError(typeName, receivedTypeName);
		LabelWidget label = new LabelWidget(message);
		label.addStyleName(TitleViewResources.STYLE_DATATYPE_MISMATCH_LABEL);
		return label;
	}

	/*
	 * Get the data type name to display given the field definition
	 */
	private String getFieldTypeName(UIFieldDefinition fieldDefinition) {
		String name = null;
		if (fieldDefinition instanceof UIComplexFieldDefinition) {
			name = messages.complexTypeName();
		} else if (fieldDefinition instanceof UISimpleFieldDefinition) {
			name = ((UISimpleFieldDefinition) fieldDefinition).getDatatype().getName();
		} else {
			name = messages.unknownTypeName();
		}

		return name;
	}

	/*
	 * Get the data type name to display given the field definition
	 */
	private String getFieldTypeName(UIField field) {
		String name = null;
		if (field instanceof UIComplexField) {
			name = messages.complexTypeName();
		} else if (field instanceof UISimpleField) {
			name = ((UISimpleField<?>) field).getDatatype().getName();
		} else {
			name = messages.unknownTypeName();
		}

		return name;
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	@Override
	public HandlerRegistration addViewChangeHandler(ViewChangeHandler handler) {
		return addHandler(handler, ViewChangeEvent.getType());
	}

	/**
	 * Fire a view change event
	 */
	protected void fireViewChangeEvent() {
		fireEvent(new ViewChangeEvent());
	}

	/**
	 * Get the view change handler that is used to delegate view change events fired by child
	 * widgets
	 * 
	 * @return The view change handler for event delegation
	 */
	protected ViewChangeHandler getViewChangeEventDelegator() {
		return this.viewChangeEventDelegator;
	}

	/*
	 * Internal class to handle the view change event
	 */
	private class ViewChangeEventHandler implements ViewChangeHandler {
		@Override
		public void onViewChange(ViewChangeEvent event) {
			fireEvent(event);
		}
	}
}
