/*
 * Created on Jul 27, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.view.asset.field.complex;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIComplexField;
import com.tandbergtv.cms.portal.content.client.title.service.asset.IUIAssetFactory;
import com.tandbergtv.cms.portal.content.client.title.service.asset.UIAssetFactory;
import com.tandbergtv.cms.portal.content.client.title.view.HasViewChangeHandlers;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewMessageCache;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewMessages;
import com.tandbergtv.cms.portal.content.client.title.view.ViewChangeEvent;
import com.tandbergtv.cms.portal.content.client.title.view.ViewChangeHandler;
import com.tandbergtv.cms.portal.content.client.title.view.asset.AssetTree.AssetInfo;
import com.tandbergtv.cms.portal.content.client.title.view.asset.AssetValidationMessage;
import com.tandbergtv.cms.portal.content.client.title.view.asset.MetadataTypeMismatchException;
import com.tandbergtv.cms.portal.content.client.title.view.asset.field.CollapsedFieldHandler;
import com.tandbergtv.cms.portal.content.client.title.view.asset.field.complex.table.ComplexFieldTable;
import com.tandbergtv.cms.portal.content.client.title.view.asset.field.simple.table.SimpleFieldTable;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIComplexFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIFieldDefinition;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.DisclosureContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.RoundedDisclosureContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

/**
 * Widget that represents a complex field
 * 
 * @author Vijay Silva
 */
public class ComplexFieldWidget extends Composite implements HasViewChangeHandlers {

	/* Widgets */
	private SimpleContainer container;
	private DisclosureContainer disclosureContainer;
	private ButtonWidget addButton, removeButton;
	private SimpleFieldTable simpleFieldContainer;
	private VerticalContainer complexGroupContainer;
	private EventHandler internalHandler = new EventHandler();

	/* Model */
	private UIComplexFieldDefinition fieldDefinition;
	private boolean readOnly;
	private AssetInfo parentAsset;
	private UIComplexField field;
	private TitleViewMessages messages = TitleViewMessageCache.getMessages();
	private Map<UIComplexFieldDefinition, ComplexFieldGroupWidget> complexWidgetMap;
	private IUIAssetFactory assetFactory;

	/* Style names */
	private static final String STYLE_NAME = "content-ComplexFieldWidget";
	private static final String STYLE_DISCLOSURE_CONTAINER = "content-ComplexFieldWidget-disclosureContainer";
	private static final String STYLE_DISCLOSURE_HEADER = "content-ComplexFieldWidget-disclosureHeader";
	private static final String STYLE_HEADER_LABEL = "content-ComplexFieldWidget-disclosureHeader-label";
	private static final String STYLE_HEADER_ADD_BUTTON = "content-ComplexFieldWidget-disclosureHeader-addButton";
	private static final String STYLE_HEADER_REMOVE_BUTTON = "content-ComplexFieldWidget-disclosureHeader-removeButton";
	private static final String STYLE_DISCLOSURE_CONTENTS = "content-ComplexFieldWidget-disclosureContents";
	private static final String STYLE_COMPLEX_GROUPS_TABLE = "content-ComplexFieldWidget-complexGroupsContainer";

	/**
	 * Constructor
	 * 
	 * @param definition The field definition
	 * @param readOnly flag indicating if the field is read-only or not
	 */
	public ComplexFieldWidget(UIComplexFieldDefinition definition, boolean readOnly) {
		this(definition, readOnly, true);
	}
	public ComplexFieldWidget(UIComplexFieldDefinition definition, boolean readOnly, boolean batchEdit) {
		this.fieldDefinition = definition;
		this.readOnly = readOnly;

		this.complexWidgetMap = new HashMap<UIComplexFieldDefinition, ComplexFieldGroupWidget>();
		assetFactory = new UIAssetFactory();

		container = new SimpleContainer();
		this.initWidget(container);

		initialize(batchEdit);
	}

	/*
	 * Build the contents of the widget
	 */
	private void initialize(boolean batchEdit) {
		addStyleName(STYLE_NAME);

		/* Build the disclosure container */
		Widget headingWidget = buildHeadingWidget();
		disclosureContainer = new RoundedDisclosureContainer(headingWidget, true);
		disclosureContainer.addStyleName(STYLE_DISCLOSURE_CONTAINER);
		container.add(disclosureContainer);

		/* Build the contents of the disclosure container */
		VerticalContainer contentContainer = new VerticalContainer();
		contentContainer.addStyleName(STYLE_DISCLOSURE_CONTENTS);
		disclosureContainer.setContent(contentContainer);

		/* Build the container for the simple field groups and collapsed complex groups */
		simpleFieldContainer = new SimpleFieldTable(this.fieldDefinition, readOnly);
		simpleFieldContainer.addViewChangeHandler(internalHandler);
		contentContainer.add(simpleFieldContainer);

		/* Build the container for the complex field groups */
		complexGroupContainer = new VerticalContainer();
		complexGroupContainer.addStyleName(STYLE_COMPLEX_GROUPS_TABLE);
		contentContainer.add(complexGroupContainer);
		addComplexGroupWidgets(fieldDefinition, batchEdit);
	}

	/*
	 * Build the widget that is used as the heading for the complex field disclosure panel
	 */
	private Widget buildHeadingWidget() {
		HorizontalContainer headingPanel = new HorizontalContainer();
		headingPanel.addStyleName(STYLE_DISCLOSURE_HEADER);

		/* Heading label */
		LabelWidget headingLabel = new LabelWidget(getFieldDisplayName());
		headingLabel.addStyleName(STYLE_HEADER_LABEL);
		headingPanel.add(headingLabel);

		/* The add button */
		addButton = new ButtonWidget(messages.fieldAddButton());
		addButton.addStyleName(STYLE_HEADER_ADD_BUTTON);
		addButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				event.stopPropagation();
			}
		});
		addButton.setVisible(false);
		addButton.setEnabled(false);
		headingPanel.add(addButton);

		/* The remove button */
		removeButton = new ButtonWidget(messages.fieldRemoveButton());
		removeButton.addStyleName(STYLE_HEADER_REMOVE_BUTTON);
		removeButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				event.stopPropagation();
			}
		});
		removeButton.setVisible(false);
		removeButton.setEnabled(false);
		headingPanel.add(removeButton);

		return headingPanel;
	}

	/**
	 * Get the display name to use for this complex field
	 * 
	 * @return The display name for the field
	 */
	protected String getFieldDisplayName() {
		return fieldDefinition.getDisplayName();
	}

	/*
	 * Build the widgets for the children of the provided complex field definition
	 */
	private void addComplexGroupWidgets(UIComplexFieldDefinition parent, boolean batchEdit) {
		if (batchEdit && !parent.isBatch())
			return;
		ComplexFieldGroupWidget widget = null;
		for (UIFieldDefinition child : parent.getChildren()) {
			if (!(child instanceof UIComplexFieldDefinition)
					|| (batchEdit && !child.isBatch()))
				// pass if this is not a complex field or we are not diplaying
				// a batch field
				continue;
			UIComplexFieldDefinition complexChild = (UIComplexFieldDefinition) child;
			if (!complexChild.isShowCollapsed()) {
				widget = new ComplexFieldGroupContainer(complexChild, readOnly);
				widget.addViewChangeHandler(internalHandler);
				complexGroupContainer.add(widget);
				complexWidgetMap.put(complexChild, widget);
			} else if (complexChild.getMaximumCount() != 1) {
				widget = new ComplexFieldTable(complexChild, readOnly);
				widget.addViewChangeHandler(internalHandler);
				complexGroupContainer.add(widget);
				complexWidgetMap.put(complexChild, widget);
			} else {
				addComplexGroupWidgets(complexChild, batchEdit);
			}
		}
	}

	// ========================================================================
	// ===================== BUTTON ACCESS
	// ========================================================================

	/**
	 * Get the add button widget
	 * 
	 * @return The widget for the add button
	 */
	public ButtonWidget getAddButton() {
		return this.addButton;
	}

	/**
	 * Get the remove button widget
	 * 
	 * @return The widget for the remove button
	 */
	public ButtonWidget getRemoveButton() {
		return this.removeButton;
	}

	// ========================================================================
	// ===================== UPDATING INPUT
	// ========================================================================

	/**
	 * Set the input asset for this view
	 * 
	 * @param asset The asset for which the files need to be displayed
	 */
	public void setInput(AssetInfo asset, UIComplexField complexField) {
		this.parentAsset = asset;
		this.field = complexField;

		/* Update the input of the contained widgets */
		simpleFieldContainer.setInput(parentAsset, field);
		for (Widget widget : complexGroupContainer) {
			if (widget instanceof ComplexFieldGroupWidget) {
				ComplexFieldGroupWidget complexGroupWidget = (ComplexFieldGroupWidget) widget;
				try {
					setInput(complexGroupWidget);
				} catch (MetadataTypeMismatchException mtme) {
					complexGroupWidget.setInput(parentAsset, null);
					complexGroupWidget.reportMismatchError(mtme);
				}
			}
		}
	}

	/*
	 * Set the input for the field group handling the case of collapsed complex fields
	 */
	private void setInput(ComplexFieldGroupWidget complexGroupWidget)
	        throws MetadataTypeMismatchException {
		UIComplexFieldDefinition targetDefinition = complexGroupWidget.getFieldDefinition();
		CollapsedFieldHandler handler = new CollapsedFieldHandler(assetFactory, fieldDefinition,
		        targetDefinition);
		UIComplexField parent = handler.getCollapsedFieldParent(field);
		complexGroupWidget.setInput(parentAsset, parent);
	}

	/**
	 * Determine if this widget is read-only
	 * 
	 * @return true if the widget is read-only, false otherwise
	 */
	public boolean isReadOnly() {
		return this.readOnly;
	}

	/**
	 * @return the complex field definition
	 */
	public UIComplexFieldDefinition getFieldDefinition() {
		return fieldDefinition;
	}

	/**
	 * @return the parent asset
	 */
	public AssetInfo getParentAsset() {
		return parentAsset;
	}

	/**
	 * @return the complex field
	 */
	public UIComplexField getField() {
		return field;
	}

	/**
	 * @return the title view messages
	 */
	protected TitleViewMessages getViewMessages() {
		return messages;
	}

	// ========================================================================
	// ===================== REFRESHING VIEW
	// ========================================================================

	/**
	 * Refresh the view.
	 */
	public void refresh() {
		/* Refresh the state of simple fields */
		simpleFieldContainer.refresh();

		/* Refresh the state of the complex fields */
		for (Widget widget : complexGroupContainer) {
			if (widget instanceof ComplexFieldGroupWidget) {
				ComplexFieldGroupWidget complexGroupWidget = (ComplexFieldGroupWidget) widget;
				complexGroupWidget.refresh();
			}
		}

		/* Ensure that the disclosure panel is 'open' */
		disclosureContainer.setOpen(true);
	}

	// ========================================================================
	// ===================== METADATA VALIDATION
	// ========================================================================

	/**
	 * Validate the widgets, and return validation messages for each of the metadata widgets that
	 * fail validation.
	 * 
	 * @param messages the list of validation messages
	 */
	public void validate(List<AssetValidationMessage> messages,boolean draft) {
		simpleFieldContainer.validate(messages,draft);
		for (Widget widget : complexGroupContainer) {
			if (widget instanceof ComplexFieldGroupWidget) {
				ComplexFieldGroupWidget complexGroupWidget = (ComplexFieldGroupWidget) widget;
				complexGroupWidget.validate(messages,draft);
			}
		}
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	@Override
	public HandlerRegistration addViewChangeHandler(ViewChangeHandler handler) {
		return addHandler(handler, ViewChangeEvent.getType());
	}

	/*
	 * Internal event handler class for events handled by internal widgets
	 */
	private class EventHandler implements ViewChangeHandler {

		@Override
		public void onViewChange(ViewChangeEvent event) {
			/* Pass the event up */
			fireEvent(event);
		}
	}
}
