/*
 * Created on Jul 23, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.view.asset.field.complex.table;

import static com.tandbergtv.cms.portal.content.client.title.view.asset.AssetContentTable.CELL_DELETE_COLUMN_STYLE;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.HasKeyUpHandlers;
import com.google.gwt.event.dom.client.KeyUpEvent;
import com.google.gwt.event.dom.client.KeyUpHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.ui.FocusWidget;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIComplexField;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIField;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UISimpleField;
import com.tandbergtv.cms.portal.content.client.title.service.asset.IUIAssetFactory;
import com.tandbergtv.cms.portal.content.client.title.service.asset.UIAssetFactory;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewResources;
import com.tandbergtv.cms.portal.content.client.title.view.asset.AssetContentTable;
import com.tandbergtv.cms.portal.content.client.title.view.asset.AssetValidationMessage;
import com.tandbergtv.cms.portal.content.client.title.view.asset.MetadataTypeMismatchException;
import com.tandbergtv.cms.portal.content.client.title.view.asset.field.CollapsedFieldHandler;
import com.tandbergtv.cms.portal.content.client.title.view.asset.field.complex.ComplexFieldGroupWidget;
import com.tandbergtv.cms.portal.content.client.title.view.asset.field.simple.SimpleFieldWidget;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIComplexFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UISimpleFieldDefinition;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.PushButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.DisclosureContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.RoundedDisclosureContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.DataTypeWidget;

/**
 * The widget used to represent a 'collapsed' complex field with cardinality greater than 1.
 * 
 * @author Vijay Silva
 */
public class ComplexFieldTable extends ComplexFieldGroupWidget {

	/* Widgets */
	private DisclosureContainer disclosureContainer;
	private SimpleContainer container;
	private AssetContentTable table;

	/* Model */
	private final List<UISimpleFieldDefinition> columnFieldDefinitions = new ArrayList<UISimpleFieldDefinition>();
	private IUIAssetFactory assetFactory = new UIAssetFactory();
	private UIComplexField newField = null;
	private List<HandlerRegistration> newFieldHandlers = new ArrayList<HandlerRegistration>();
	private Map<UIComplexField, Widget> rowIdentifier = new HashMap<UIComplexField, Widget>();

	/* Style names */
	private static final String STYLE_NAME = "content-ComplexFieldTable";
	private static final String STYLE_CONTAINER = "content-ComplexFieldTable-container";
	private static final String STYLE_ASSET_TABLE = "content-ComplexFieldTable-assetTable";

	/**
	 * The Complex Field Table Constructor
	 * 
	 * @param fieldDefinition The Complex Field Definition
	 * @param readOnly Flag indicating if the table is read-only or not
	 */
	public ComplexFieldTable(UIComplexFieldDefinition fieldDefinition, boolean readOnly) {
		super(fieldDefinition, readOnly);

		/* The widgets */
		String heading = fieldDefinition.getDisplayName();
		disclosureContainer = new RoundedDisclosureContainer(heading, true);
		disclosureContainer.addStyleName(STYLE_NAME);
		container = new SimpleContainer();
		container.addStyleName(STYLE_CONTAINER);
		disclosureContainer.setContent(container);
		initWidget(disclosureContainer);

		/* Initialize the table */
		initialize();
	}

	/*
	 * Initialize the Table and its columns
	 */
	private void initialize() {
		table = new AssetContentTable();
		table.addStyleName(STYLE_ASSET_TABLE);
		container.setWidget(table);

		/* Recursively collect the simple field definition columns */
		this.buildColumnFieldDefinitions(getFieldDefinition());

		/* Add the labels for each of the column field definitions */
		int colIndex = 0;
		for (UISimpleFieldDefinition columnFieldDefinition : columnFieldDefinitions) 
		{
			table.addHeading(columnFieldDefinition.getDisplayName());
			LabelWidget widget = table.getHeadingWidget(colIndex);
			widget.setTitle(columnFieldDefinition.getXPath());
			++colIndex;
		}
	}

	/* Build the field definitions for each column */
	private void buildColumnFieldDefinitions(UIFieldDefinition fieldDefinition) {
		if (fieldDefinition instanceof UIComplexFieldDefinition) {
			UIComplexFieldDefinition complexDefinition = (UIComplexFieldDefinition) fieldDefinition;
			for (UIFieldDefinition childDefinition : complexDefinition.getChildren()) {
				buildColumnFieldDefinitions(childDefinition);
			}
		} else if (fieldDefinition instanceof UISimpleFieldDefinition) {
			this.columnFieldDefinitions.add((UISimpleFieldDefinition) fieldDefinition);
		}
	}

	// ========================================================================
	// ===================== REFRESHING VIEW
	// ========================================================================

	/**
	 * Refresh the view, redrawing.
	 */
	@Override
	public void refresh() {
		/* Ensure that the disclosure container is 'open' */
		this.disclosureContainer.setOpen(true);

		/* Remove all existing rendered rows and ensure table is displayed */
		table.removeAllRows();
		rowIdentifier.clear();
		this.clearNewRowState();
		this.container.setWidget(table);

		/* There is no input, do nothing */
		if(getParentField() == null)
		{
			return;
		}

		// If this is new asset (ID == null), try to set default value from the content class		
		boolean applyCC = (this.getParentAsset().getAsset().getId() == null);
		if(applyCC)
		{
			boolean hasNonDefaultValue = false;
			for(UISimpleFieldDefinition fieldDefinition : this.columnFieldDefinitions)
			{
				if(fieldDefinition.getDefaultValue() != null && !fieldDefinition.getDefaultValue().isEmpty())
				{
					hasNonDefaultValue = true;
				}
			}
			
			if(hasNonDefaultValue)
			{
				UIComplexField field = (UIComplexField) assetFactory.createAssetField(getFieldDefinition());
				getParentField().addField(field);
			}
		}
		
		/* Get the list of fields */
		List<UIComplexField> inputFields = null;
		try 
		{
			inputFields = getFields();
		} 
		catch (MetadataTypeMismatchException mtme) 
		{
			reportMismatchError(mtme);
			return;
		}
		
		/* Add the rows for each of the fields to render */
		for (UIComplexField field : inputFields) 
		{
			buildRow(field, applyCC);
		}

		/* Build the new row view if the maximum count is not exceeded */
		addNewFieldRow();
	}

	/*
	 * Build a new row for the field
	 */
	private void buildRow(UIComplexField field, boolean applyContentClass) 
	{
		int row = table.addRow();

		/* Add a widget for each column */
		for(UISimpleFieldDefinition fieldDefinition : this.columnFieldDefinitions) 
		{
			Widget widget = buildMetadataWidget(field, fieldDefinition, applyContentClass);
			table.addCell(row, widget);
		}

		/* The Delete Button */
		Widget deleteButton = buildDeleteButton(field);
		table.addCell(row, deleteButton);
		table.addTableCellStyle(row, table.getCellCount(row) - 1, CELL_DELETE_COLUMN_STYLE);

		/* Use the delete button widget as a lookup for the row */
		rowIdentifier.put(field, deleteButton);
	}

	// ========================================================================
	// ===================== NEW ROW HANDLING
	// ========================================================================

	/*
	 * Build a new field
	 */
	private void addNewFieldRow() 
	{
		/* Check if maximum count is not exceeded and new field is not already present */
		int maxCount = getFieldDefinition().getMaximumCount();
		int rowCount = this.table.getRowCount();

		boolean newRowMissing = (newField == null);
		boolean underMaxLimit = (maxCount == -1 || maxCount > rowCount);
		boolean showNewRow = (!this.isReadOnly() || rowCount == 0);
		if (newRowMissing && underMaxLimit && showNewRow) 
		{
			/* Create the new field but don't add to the input / model */
			newField = (UIComplexField) assetFactory.createAssetField(getFieldDefinition());

			/* build the row / view */
			buildRow(newField, false);
		}
	}

	/*
	 * Clear the new field created previously, and remove all registered handlers
	 */
	private void clearNewRowState() {
		newField = null;

		/* Remove all handlers added only for the new field row */
		for (HandlerRegistration registration : newFieldHandlers) {
			registration.removeHandler();
		}
		newFieldHandlers.clear();
	}

	/*
	 * After widget is updated causing the model to update, check if a new row needs to be added
	 */
	private void handleFieldUpdated(UIComplexField field) {
		/* If the updated field is the new field, add to model and create new row */
		if (field.equals(newField)) {
			/* Add new field to model */
			getParentField().addField(field);
			clearNewRowState();

			/* Update the view */
			showDeleteButton(field);
			addNewFieldRow();
		}

		/* Fire change in the view */
		fireViewChangeEvent();
	}

	/*
	 * After a field has been deleted from model, update all row widgets with any changes required
	 */
	private void handleFieldDeleted(UIComplexField field) {
		/* May need to add a new field row */
		addNewFieldRow();

		/* Fire change in the view */
		fireViewChangeEvent();
	}

	// ========================================================================
	// ===================== DELETE BUTTON WIDGET
	// ========================================================================

	/*
	 * Build the widget for the delete button along with event handling
	 */
	private Widget buildDeleteButton(final UIComplexField field) {
		ImageWidget image = new ImageWidget(TitleViewResources.DELETE_IMAGE_URL);
		final PushButtonWidget button = new PushButtonWidget(image);
		button.addStyleName(TitleViewResources.STYLE_FIELD_DELETE_BUTTON);
		boolean isReadOnly = (isReadOnly() || field.equals(newField));

		button.setEnabled(!isReadOnly);
		button.setVisible(!isReadOnly);

		/* Event handler */
		button.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleDeleteButtonClicked(field, button, event);
			}
		});

		return button;
	}

	/*
	 * Show and enable the delete button for a field
	 */
	private void showDeleteButton(UIComplexField field) {
		Widget lookupWidget = rowIdentifier.get(field);
		int row = table.getRow(lookupWidget);
		int column = columnFieldDefinitions.size();

		if (row >= 0 && column >= 0) {
			FocusWidget button = (FocusWidget) table.getCellWidget(row, column);
			button.setEnabled(true);
			button.setVisible(true);
		}
	}

	/*
	 * Event handling for the delete button click
	 */
	private void handleDeleteButtonClicked(UIComplexField field, Widget button, ClickEvent event) {
		/* Validate that the field can be deleted */
		if (!validateFieldDelete(field)) {
			return;
		}

		/* Remove from the view and from the model */
		int row = table.getRow(button);
		table.removeRow(row);
		rowIdentifier.remove(field);
		getParentField().removeField(field);

		/* Handle the delete field for other widgets */
		handleFieldDeleted(field);
	}

	/*
	 * Ensure that the field can be deleted
	 */
	private boolean validateFieldDelete(UIComplexField field) {
		/* Read only, cannot delete */
		if (isReadOnly() || field.equals(newField)) {
			return false;
		}

		return true;
	}

	// ========================================================================
	// ===================== FIELD METADATA WIDGET
	// ========================================================================

	/*
	 * Build the metadata widget
	 */
	private Widget buildMetadataWidget(UIComplexField field, UISimpleFieldDefinition definition, boolean applyContentClass) 
	{
		/* Build the widget */
		ComplexTableMetadataWidgetBuilder builder = new ComplexTableMetadataWidgetBuilder(this);
		SimpleFieldWidget<?, ?, ?> fieldWidget = null;
		
		boolean fieldReadOnlyFlag = isReadOnly();
		
		try 
		{
			UIField simpleField = getField(field, definition, applyContentClass);
			
			// Try to apply lock flag from content class
			if(fieldReadOnlyFlag == false) 
			{
				fieldReadOnlyFlag = definition.isLocked();
			}
			fieldWidget = builder.build(field, definition, simpleField, fieldReadOnlyFlag);
		} 
		catch (MetadataTypeMismatchException mtme) 
		{
			return this.buildMetadataMismatchErrorWidget(mtme);
		}

		/* For certain widgets, need to listen for key up events */
		DataTypeWidget<?> widget = fieldWidget.getDataTypeWidget();
		if (fieldReadOnlyFlag == false && field.equals(newField) && widget instanceof HasKeyUpHandlers) {
			KeyUpHandler handler = new MetadataWidgetKeyEventHandler(field, fieldWidget);
			HasKeyUpHandlers keyAwareWidget = (HasKeyUpHandlers) widget;
			newFieldHandlers.add(keyAwareWidget.addKeyUpHandler(handler));
		}

		return fieldWidget;
	}

	/*
	 * Find or create the child field given the child field definition
	 */
	private UIField getField(UIComplexField field, UISimpleFieldDefinition simpleDefinition, boolean applyContentClass) 
	{
		UIComplexFieldDefinition complexDefinition = this.getFieldDefinition();
		CollapsedFieldHandler handler = new CollapsedFieldHandler(assetFactory, complexDefinition, simpleDefinition);
		UIComplexField parentField = handler.getCollapsedFieldParent(field);
		UIField childField = parentField.getChildren().getField(simpleDefinition.getName());
		
		if(childField == null) 
		{
			childField = assetFactory.createAssetField(simpleDefinition);
						
			if(applyContentClass)
			{
				if(childField instanceof UISimpleField) 
				{
					((UISimpleField<?>)childField).setStringValue(simpleDefinition.getDefaultValue());
				}
			}
			
			parentField.addField(childField);
		}

		return childField;
	}

	/*
	 * Handle an update in the metadata widget value
	 */
	void handleMetadataValueChanged(UIComplexField field, SimpleFieldWidget<?, ?, ?> widget) {
		handleFieldUpdated(field);
	}

	/*
	 * Handle the key press event for the widgets in the 'new field' row
	 */
	private void handleKeyUpEvent(UIComplexField field, SimpleFieldWidget<?, ?, ?> widget,
	        KeyUpEvent event) {
		/* Handle event only for the new field */
		if (!field.equals(newField))
			return;

		String stringValue = widget.getDataTypeWidget().getTextValue();
		if (stringValue != null && stringValue.length() > 0) {
			handleFieldUpdated(field);
		}
	}

	/*
	 * Event Handler class for handling a key press event for data type widget in the field view
	 */
	private final class MetadataWidgetKeyEventHandler implements KeyUpHandler {

		private UIComplexField field;
		private SimpleFieldWidget<?, ?, ?> widget;

		/**
		 * Constructor
		 */
		public MetadataWidgetKeyEventHandler(UIComplexField field, SimpleFieldWidget<?, ?, ?> widget) {
			this.field = field;
			this.widget = widget;
		}

		@Override
		public void onKeyUp(KeyUpEvent event) {
			handleKeyUpEvent(field, widget, event);
		}
	}

	// ========================================================================
	// ===================== METADATA VALIDATION
	// ========================================================================

	@Override
	public void validate(List<AssetValidationMessage> messages,boolean draft) {
		SimpleFieldWidget<?, ?, ?> widget = null;
		for (int row = 0; row < table.getRowCount(); row++) {
			for (int column = 0; column < columnFieldDefinitions.size(); column++) {
				widget = (SimpleFieldWidget<?, ?, ?>) table.getCellWidget(row, column);
				if (!widget.isValidValue(draft)) {
					String name = columnFieldDefinitions.get(column).getDisplayName();
					String error = widget.getCurrentToolTip();
					messages.add(new AssetValidationMessage(getParentAsset(), name, widget, error));
				}
			}
		}
	}

	// ========================================================================
	// ===================== ERROR HANDLING
	// ========================================================================

	/**
	 * Report the mismatch error resulting from mismatch in definition and data
	 */
	public void reportMismatchError(MetadataTypeMismatchException mtme) {
		Widget errorWidget = this.buildMetadataMismatchErrorWidget(mtme);
		container.setWidget(errorWidget);
	}
}
