/*
 * Created on Jul 28, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.view.asset.field.simple.table;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.HasKeyUpHandlers;
import com.google.gwt.event.dom.client.KeyUpEvent;
import com.google.gwt.event.dom.client.KeyUpHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.ui.FocusWidget;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIComplexField;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIField;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UISimpleField;
import com.tandbergtv.cms.portal.content.client.title.service.asset.IUIAssetFactory;
import com.tandbergtv.cms.portal.content.client.title.service.asset.UIAssetFactory;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewMessages;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewResources;
import com.tandbergtv.cms.portal.content.client.title.view.asset.AssetTree.AssetInfo;
import com.tandbergtv.cms.portal.content.client.title.view.asset.AssetValidationMessage;
import com.tandbergtv.cms.portal.content.client.title.view.asset.MetadataTypeMismatchException;
import com.tandbergtv.cms.portal.content.client.title.view.asset.field.simple.SimpleFieldWidget;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UISimpleFieldDefinition;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.CheckBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.PushButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.DataTypeWidget;

/**
 * Represents a grouping a simple field widgets with the same definition. This is an 'artificial
 * widget' since it is a grouping of widgets within another widget (SimpleFieldTable).
 * 
 * @author Vijay Silva
 */
class SimpleFieldGroup {

	/* Widget */
	private SimpleFieldTable fieldTable;

	/* Model */
	private UISimpleFieldDefinition fieldDefinition;
	private boolean readOnly;
	private AssetInfo parentAsset;
	private UIComplexField parentField;
	private IUIAssetFactory assetFactory = new UIAssetFactory();
	private UISimpleField<?> newField;
	private List<HandlerRegistration> newFieldHandlers = new ArrayList<HandlerRegistration>();
	private Map<UISimpleField<?>, SimpleFieldView> viewIdentifier = new HashMap<UISimpleField<?>, SimpleFieldView>();
	

	/* Style Names */
	private static final String STYLE_FIELD_WIDGET_CONTENTS = "content-SimpleFieldGroup-fieldWidget";
	static final String STYLE_FIELD_LABEL = "content-SimpleFieldGroup-fieldLabel";
	private static final String STYLE_NULLIFY_CHECKBOX = "content-SimpleFieldGroup-nullifyCheckBox";

	/**
	 * Constructor
	 * 
	 * @param fieldDefinition The simple field definition
	 * @param readOnly flag indicating if this view is read only
	 */
	public SimpleFieldGroup(SimpleFieldTable table, UISimpleFieldDefinition fieldDefinition,
	        boolean readOnly) {
		this.fieldTable = table;
		this.fieldDefinition = fieldDefinition;
		this.readOnly = readOnly;
	}

	// ========================================================================
	// ===================== INPUT
	// ========================================================================

	/**
	 * Set the input for this simple field group
	 */
	public void setInput(AssetInfo parentAsset, UIComplexField parentField) {
		this.parentAsset = parentAsset;
		this.parentField = parentField;
		refresh();
	}

	/**
	 * @return the field definition
	 */
	public UISimpleFieldDefinition getFieldDefinition() {
		return fieldDefinition;
	}

	/**
	 * @return the read only flag value
	 */
	public boolean isReadOnly() {
		return readOnly;
	}

	/**
	 * @return the parent asset
	 */
	public AssetInfo getParentAsset() {
		return parentAsset;
	}

	/**
	 * @return the parentField
	 */
	public UIComplexField getParentField() {
		return parentField;
	}

	/**
	 * Get all the simple fields to render for this group
	 */
	private List<UISimpleField<?>> getFields() throws MetadataTypeMismatchException {
		if (parentField == null)
			return null;

		List<UISimpleField<?>> inputFields = new ArrayList<UISimpleField<?>>();
		String name = fieldDefinition.getName();
		for (UIField field : parentField.getChildren().getFields(name)) {
			/* Verify the type of the simple field and field definition match */
			if (!(field instanceof UISimpleField<?>)) {
				throw new MetadataTypeMismatchException(name, fieldDefinition, field);
			}

			UISimpleField<?> simpleField = (UISimpleField<?>) field;
			String dataTypeName = fieldDefinition.getDatatype().getName();
			if (!dataTypeName.equals(simpleField.getDatatype().getName())) {
				throw new MetadataTypeMismatchException(name, fieldDefinition, simpleField);
			}

			inputFields.add(simpleField);
		}

		return inputFields;
	}

	/*
	 * Get the maximum count for the field definition
	 */
	private int getMaximumCount() {
		int minimumCount = getFieldDefinition().getMinimumCount();
		int maximumCount = getFieldDefinition().getMaximumCount();
		if (maximumCount < minimumCount) {
			maximumCount = minimumCount;
		}

		return maximumCount;
	}

	// ========================================================================
	// ===================== REFRESHING VIEW
	// ========================================================================

	/**
	 * Refresh the view.
	 */
	public void refresh() {
		/* Remove all rendered fields */
		fieldTable.removeAllRows(this);
		this.clearNewFieldState();
		this.viewIdentifier.clear();

		/* There is no input, do nothing */
		if (getParentField() == null)
			return;

		/* Get the list of fields */
		List<UISimpleField<?>> inputFields = null;
		try {
			inputFields = getFields();
		} catch (MetadataTypeMismatchException mtme) {
			fieldTable.addMismatchErrorRow(this, mtme);
			return;
		}

		/* Add the rows for each of the fields to render */
		for (UISimpleField<?> field : inputFields) {
			buildFieldView(field);
		}

		/* Build the new field if required */
		addNewField();
	}

	/*
	 * Build a new row for the field
	 */
	private void buildFieldView(UISimpleField<?> field) 
	{
		if (this.parentAsset.isBatchEdit() && !fieldDefinition.isBatch()) 
		{
			return;
		}
		
		if(fieldDefinition.hideInUI())
		{
			return;
		}
		
		Widget label = buildLabelWidget(field);
		Widget fieldWidget = buildMetadataWidget(field);
		Widget deleteButton = buildDeleteButton(field);
		Widget nullifyCheckBox = buildNullifyCheckBox(field, fieldWidget);
		Widget autoFillCheckBox = buildAutoFillCheckBox(field, fieldWidget);
		SimpleFieldView fieldView = new SimpleFieldView(label, fieldWidget, deleteButton, nullifyCheckBox, autoFillCheckBox);
		fieldTable.addRow(this, fieldView.getWidgets());

		/* Use the delete button widget as a lookup for the row */
		viewIdentifier.put(field, fieldView);
	}

	// ========================================================================
	// ===================== NEW FIELD HANDLING
	// ========================================================================

	/*
	 * Build a new field and add to the view at the end of the field group
	 */
	private void addNewField() {
		/* Check if maximum count is not exceeded and new field is not already present */
		int maxCount = getMaximumCount();
		int fieldCount = this.getFieldCount();

		boolean newFieldMissing = (newField == null);
		boolean underMaxLimit = (maxCount > fieldCount);
		boolean showNewRow = (!this.isReadOnly() || fieldCount == 0);
		if (newFieldMissing && underMaxLimit && showNewRow) {
			/* Create the new field but don't add to the model */
			newField = createField();
			parentField.addField(newField);

			/* build the row / view */
			buildFieldView(newField);
		}
	}

	/*
	 * Clear the new field created previously, and remove all registered handlers
	 */
	private void clearNewFieldState() {
		newField = null;

		/* Remove all handlers added only for the new field row */
		for (HandlerRegistration registration : newFieldHandlers) {
			registration.removeHandler();
		}
		newFieldHandlers.clear();
	}

	/*
	 * Get the number of fields present in this group
	 */
	private int getFieldCount() {
		String name = fieldDefinition.getName();
		return (parentField != null) ? parentField.getChildren().getFieldCount(name) : 0;
	}

	/*
	 * Create a new field based on the field definition
	 */
	private UISimpleField<?> createField() {
		UISimpleField<?> newField = (UISimpleField<?>)assetFactory.createAssetField(this.fieldDefinition);
		
		// If this is new asset (ID == null), try to set default value from the content class
		if(this.getParentAsset().getAsset().getId() == null) {
			newField.setStringValue(fieldDefinition.getDefaultValue());
		}

		return newField;
	}

	/*
	 * After widget is updated causing the model to update, check if a new row needs to be added
	 */
	private void handleFieldUpdated(UISimpleField<?> field, String widgetValue) {
		/* If the updated field is the new field, add to model and create new field */
		if (widgetValue != null && widgetValue.length() > 0 && field.equals(newField)) {
			clearNewFieldState();

			/* Update the view */
			tryShowDeleteButton(field);
			addNewField();
		}
		
		// set title as dirty
		fieldTable.fireViewChangeEvent();
	}

	/*
	 * After a field has been deleted from model, update all widgets with any changes required
	 */
	private void handleFieldDeleted(UISimpleField<?> field) {
		/* Update the existing delete buttons */
		for (UISimpleField<?> currentField : getFields()) {
			tryShowDeleteButton(currentField);
		}

		/* May need to add a new field row */
		addNewField();
	}

	// ========================================================================
	// ===================== LABEL WIDGET
	// ========================================================================

	private Widget buildLabelWidget(UISimpleField<?> field) 
	{
		TitleViewMessages messages = fieldTable.getViewMessages();
		String labelText = messages.simpleFieldNameLabel(fieldDefinition.getDisplayName());
		String html = null;
		if(fieldDefinition.isRequired())
		{
			html = "<span class='nwt-FormContainer-required'>* </span>" + labelText;
		}
		else
		{
			html = labelText;
		}

		// Add xpath tooltip
		html = "<span title='" + fieldDefinition.getXPath() + "'>" + html + "</span>";
		HTML widget = new HTML();
		widget.setHTML(html);

		widget.addStyleName(STYLE_FIELD_LABEL);
		
		return widget;
	}

	// ========================================================================
	// ===================== NULLIFY / AUTOFILL CHECKBOXES
	// ========================================================================

	/*
	 * Build the Nullify check box widget
	 */
	private Widget buildNullifyCheckBox(final UISimpleField<?> field, final Widget fieldWidget) {
		boolean isBatchEditable = this.parentAsset.isBatchEdit();
		if (!isBatchEditable) {
			return null;
		}

		TitleViewMessages messages = fieldTable.getViewMessages();
		String label = messages.simpleFieldNullifyCheckBox();
		final CheckBoxWidget widget = new CheckBoxWidget(label);
		widget.addStyleName(STYLE_NULLIFY_CHECKBOX);
		widget.setEnabled(!fieldDefinition.isLocked() && !this.isReadOnly());

		/* Handle the value change */
		widget.addValueChangeHandler(new ValueChangeHandler<Boolean>() {
			@Override
			public void onValueChange(ValueChangeEvent<Boolean> event) {
				handleNullifyCheckBoxValueChange(field, widget, fieldWidget, event);
			}
		});

		return widget;
	}

	/*
	 * Handle the value change
	 */
	private void handleNullifyCheckBoxValueChange(UISimpleField<?> field,
	        CheckBoxWidget nullifyWidget, Widget fieldWidget, ValueChangeEvent<Boolean> event) {
		Boolean value = event.getValue();
		field.setNullify((value != null) ? value : false);
		if (fieldWidget instanceof SimpleFieldWidget) {
			SimpleFieldWidget<?, ?, ?> simpleFieldWidget = (SimpleFieldWidget<?, ?, ?>) fieldWidget;
			if (field.isNullify()) {
				simpleFieldWidget.setValue(null, true);
			}
			simpleFieldWidget.setEnabled(!field.isNullify());
		}
	}

	/*
	 * Build the Auto-Fill check box widget
	 */
	private Widget buildAutoFillCheckBox(final UISimpleField<?> field, Widget fieldWidget) {
		return null;
	}

	// ========================================================================
	// ===================== DELETE BUTTON WIDGET
	// ========================================================================

	/*
	 * Build the widget for the delete button along with event handling
	 */
	private Widget buildDeleteButton(final UISimpleField<?> field) {
		ImageWidget image = new ImageWidget(TitleViewResources.DELETE_IMAGE_URL);
		final PushButtonWidget button = new PushButtonWidget(image);
		button.addStyleName(TitleViewResources.STYLE_FIELD_DELETE_BUTTON);
		boolean canShow = this.canShowDeleteButton(field);
		button.setEnabled(canShow);
		button.setVisible(canShow);

		/* Event handler */
		button.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleDeleteButtonClicked(field, button, event);
			}
		});

		return button;
	}

	/*
	 * Show and enable the delete button for a field, or hide if not visible
	 */
	private void tryShowDeleteButton(UISimpleField<?> field) {
		boolean visible = this.canShowDeleteButton(field);
		SimpleFieldView view = viewIdentifier.get(field);
		if (view != null && view.getDeleteButton() != null) {
			Widget button = view.getDeleteButton();
			button.setVisible(visible);
			if (button instanceof FocusWidget)
				((FocusWidget) button).setEnabled(visible);
		}
	}

	/*
	 * Determine if the delete button can be shown
	 */
	private boolean canShowDeleteButton(UISimpleField<?> field) {
		int fieldCount = getFieldCount();
		int maxCount = getMaximumCount();
		boolean hasExtraFields = (fieldCount > maxCount);
		boolean canShow = (!isReadOnly() && !field.equals(newField) && (maxCount > 1 || hasExtraFields));
		return canShow;
	}

	/*
	 * Event handling for the delete button click
	 */
	private void handleDeleteButtonClicked(UISimpleField<?> field, Widget button, ClickEvent event) {
		/* Validate that the field can be deleted */
		if (!validateFieldDelete(field)) {
			return;
		}

		/* Remove from the view and from the model */
		SimpleFieldView fieldView = viewIdentifier.get(field);
		fieldTable.removeRow(this, fieldView.getFieldLabel());
		viewIdentifier.remove(field);
		getParentField().removeField(field);
		fieldTable.fireViewChangeEvent();

		/* Handle the delete field for other widgets */
		handleFieldDeleted(field);
	}

	/*
	 * Ensure that the field can be deleted
	 */
	private boolean validateFieldDelete(UISimpleField<?> field) {
		/* Read only, cannot delete */
		if (isReadOnly() || field.equals(newField)) {
			return false;
		}

		return true;
	}

	// ========================================================================
	// ===================== FIELD METADATA WIDGET
	// ========================================================================

	/*
	 * Build the metadata widget
	 */
	private Widget buildMetadataWidget(UISimpleField<?> field) {
		/* Build the widget */
		SimpleViewMetadataWidgetBuilder builder = new SimpleViewMetadataWidgetBuilder(this);
		SimpleFieldWidget<?, ?, ?> fieldWidget = null;
		boolean fieldReadOnlyFlag = readOnly;
		
		try 
		{
			// Apply lock flag from content class
			if(readOnly == false) {
				fieldReadOnlyFlag = fieldDefinition.isLocked();
			}

			if(this.parentAsset.isBatchEdit()) 
			{ 
				field.setValue(null); 
			}

			fieldWidget = builder.build(fieldDefinition, field, fieldReadOnlyFlag);
		} 
		catch (MetadataTypeMismatchException mtme) 
		{
			return fieldTable.buildMetadataMismatchErrorWidget(mtme);
		}

		/* For certain widgets, need to listen for key up events */
		DataTypeWidget<?> widget = fieldWidget.getDataTypeWidget();
		if (fieldReadOnlyFlag == false && field.equals(newField) && widget instanceof HasKeyUpHandlers) {
			KeyUpHandler handler = new MetadataWidgetKeyEventHandler(field, fieldWidget);
			HasKeyUpHandlers keyAwareWidget = (HasKeyUpHandlers) widget;
			newFieldHandlers.add(keyAwareWidget.addKeyUpHandler(handler));
		}

		return fieldWidget;
	}

	/*
	 * Handle an update in the metadata widget value
	 */
	void handleMetadataValueChanged(UISimpleField<?> field, SimpleFieldWidget<?, ?, ?> widget) {
		handleFieldUpdated(field, widget.getTextValue());
	}

	/*
	 * Handle the key press event for the widgets in the 'new field' row
	 */
	private void handleKeyUpEvent(UISimpleField<?> field, SimpleFieldWidget<?, ?, ?> widget,
	        KeyUpEvent event) {
		/* Handle event only for the new field */
		if (!field.equals(newField))
			return;

		handleFieldUpdated(field, widget.getTextValue());
	}

	/*
	 * Event Handler class for handling a key press event for data type widget in the field view
	 */
	private final class MetadataWidgetKeyEventHandler implements KeyUpHandler {

		private UISimpleField<?> field;
		private SimpleFieldWidget<?, ?, ?> widget;

		/**
		 * Constructor
		 */
		public MetadataWidgetKeyEventHandler(UISimpleField<?> field,
		        SimpleFieldWidget<?, ?, ?> widget) {
			this.field = field;
			this.widget = widget;
		}

		@Override
		public void onKeyUp(KeyUpEvent event) {
			handleKeyUpEvent(field, widget, event);
		}
	}

	// ========================================================================
	// ===================== METADATA VALIDATION
	// ========================================================================

	/**
	 * Validate the widgets, and return validation messages for each of the metadata widgets that
	 * fail validation.
	 * 
	 * @return the list of validation messages
	 */
	public void validate(List<AssetValidationMessage> messages, boolean draft) {
		/* No state */
		if (parentField == null)
			return;

		List<UIField> fields = parentField.getChildren().getFields(fieldDefinition.getName());
		SimpleFieldWidget<?, ?, ?> fieldWidget = null;
		for (UIField field : fields) {
			SimpleFieldView view = viewIdentifier.get(field);
			if (view == null || !(view.getFieldWidget() instanceof SimpleFieldWidget<?, ?, ?>))
				continue;

			fieldWidget = (SimpleFieldWidget<?, ?, ?>) view.getFieldWidget();
			if (!fieldWidget.isValidValue(draft)) {
				String name = fieldDefinition.getDisplayName();
				String error = fieldWidget.getCurrentToolTip();
				messages.add(new AssetValidationMessage(parentAsset, name, fieldWidget, error));
			}
		}
	}

	// ========================================================================
	// ===================== SIMPLE FIELD VIEW
	// ========================================================================

	/*
	 * Maintains a collection of the widgets that make up the view for a simple field
	 */
	private static final class SimpleFieldView {
		private Widget fieldLabel;
		private Widget fieldWidget;
		private Widget deleteButton;
		private Widget nullifyCheckBox;
		private Widget autoFillCheckBox;

		/**
		 * Constructor
		 */
		public SimpleFieldView(Widget label, Widget fieldWidget, Widget deleteButton,
		        Widget nullifyCheckBox, Widget autoFillCheckBox) {
			this.fieldLabel = label;
			this.fieldWidget = fieldWidget;
			this.deleteButton = deleteButton;
			this.nullifyCheckBox = nullifyCheckBox;
			this.autoFillCheckBox = autoFillCheckBox;
		}

		/**
		 * @return the fieldLabel
		 */
		public Widget getFieldLabel() {
			return fieldLabel;
		}

		/**
		 * @return the fieldWidget
		 */
		public Widget getFieldWidget() {
			return fieldWidget;
		}

		/**
		 * @return the deleteButton
		 */
		public Widget getDeleteButton() {
			return deleteButton;
		}

		public Widget[] getWidgets() {
			HorizontalContainer content = new HorizontalContainer();
			content.addStyleName(STYLE_FIELD_WIDGET_CONTENTS);
			content.add(fieldWidget);
			content.add(deleteButton);

			if (nullifyCheckBox != null) {
				content.add(nullifyCheckBox);
			}

			if (autoFillCheckBox != null) {
				content.add(autoFillCheckBox);
			}

			return new Widget[] { fieldLabel, content };
		}
	}
}
