/*
 * Created on Jul 28, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.view.asset.field.simple.table;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIComplexField;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIField;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UISimpleField;
import com.tandbergtv.cms.portal.content.client.title.service.asset.IUIAssetFactory;
import com.tandbergtv.cms.portal.content.client.title.service.asset.UIAssetFactory;
import com.tandbergtv.cms.portal.content.client.title.view.HasViewChangeHandlers;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewMessageCache;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewMessages;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewResources;
import com.tandbergtv.cms.portal.content.client.title.view.ViewChangeEvent;
import com.tandbergtv.cms.portal.content.client.title.view.ViewChangeHandler;
import com.tandbergtv.cms.portal.content.client.title.view.asset.AssetTree.AssetInfo;
import com.tandbergtv.cms.portal.content.client.title.view.asset.AssetValidationMessage;
import com.tandbergtv.cms.portal.content.client.title.view.asset.MetadataTypeMismatchException;
import com.tandbergtv.cms.portal.content.client.title.view.asset.field.CollapsedFieldHandler;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIComplexFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UISimpleFieldDefinition;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.FlexTableContainer;

/**
 * A Widget that manages a table of all the simple fields displayed broken into partitions where
 * each partition contains a group of simple fields with the same field definition.
 * 
 * @author Vijay Silva
 */
public class SimpleFieldTable extends Composite implements HasViewChangeHandlers {

	/* Widgets */
	private final FlexTableContainer table;
	private final List<SimpleFieldGroup> simpleFieldGroups;
	private final Map<SimpleFieldGroup, Integer> sizeMap;

	/* Model */
	private UIComplexFieldDefinition parentFieldDefinition;
	private boolean readOnly;
	private AssetInfo parentAsset;
	private UIComplexField parentField;
	private IUIAssetFactory assetFactory = new UIAssetFactory();
	private TitleViewMessages messages = TitleViewMessageCache.getMessages();

	/* Style Names */
	private static final String STYLE_NAME = "content-SimpleFieldTable";
	private static final String STYLE_CELL_PREFIX = "content-SimpleFieldTable-columnCell";

	/**
	 * Constructor
	 * 
	 * @param parentFieldDefinition The parent complex field definition
	 * @param readOnly flag indicating if the widget is read only
	 */
	public SimpleFieldTable(UIComplexFieldDefinition parentFieldDefinition, boolean readOnly) {
		this.parentFieldDefinition = parentFieldDefinition;
		this.readOnly = readOnly;
		this.simpleFieldGroups = new ArrayList<SimpleFieldGroup>();
		this.sizeMap = new HashMap<SimpleFieldGroup, Integer>();

		table = new FlexTableContainer();
		table.addStyleName(STYLE_NAME);
		this.initWidget(table);

		this.initialize();
	}

	/*
	 * Initialize the widget creating the required groups within the table
	 */
	private void initialize() {
		this.buildSimpleFieldGroups(this.parentFieldDefinition);
	}

	/*
	 * Build the simple field groups given the complex field definition
	 */
	private void buildSimpleFieldGroups(UIComplexFieldDefinition definition) {
		for (UIFieldDefinition child : definition.getChildren()) {
			if (child instanceof UISimpleFieldDefinition) {
				UISimpleFieldDefinition simpleChild = (UISimpleFieldDefinition) child;
				SimpleFieldGroup group = new SimpleFieldGroup(this, simpleChild, readOnly);
				addSimpleFieldGroup(group);
			} else if (child instanceof UIComplexFieldDefinition) {
				UIComplexFieldDefinition complexChild = (UIComplexFieldDefinition) child;
				if (complexChild.isShowCollapsed() && complexChild.getMaximumCount() == 1)
					buildSimpleFieldGroups(complexChild);
			}
		}
	}

	/*
	 * Add a simple field group to the table
	 */
	private void addSimpleFieldGroup(SimpleFieldGroup group) {
		simpleFieldGroups.add(group);
		sizeMap.put(group, 0);
	}

	// ========================================================================
	// ===================== INPUT
	// ========================================================================

	/**
	 * Set the input for this simple field table
	 */
	public void setInput(AssetInfo parentAsset, UIComplexField parentField) {
		this.parentAsset = parentAsset;
		this.parentField = parentField;

		/* Update the input of all the groups contained */
		for (SimpleFieldGroup fieldGroup : simpleFieldGroups) {
			try {
				setInput(fieldGroup);
			} catch (MetadataTypeMismatchException mtme) {
				fieldGroup.setInput(parentAsset, null);
				this.addMismatchErrorRow(fieldGroup, mtme);
			}
		}
	}

	/*
	 * Set the input for the field group handling the case of collapsed complex fields
	 */
	private void setInput(SimpleFieldGroup fieldGroup) throws MetadataTypeMismatchException {
		UISimpleFieldDefinition targetDefinition = fieldGroup.getFieldDefinition();
		CollapsedFieldHandler handler = new CollapsedFieldHandler(assetFactory,
		        parentFieldDefinition, targetDefinition);
		UIComplexField immediateParentField = handler.getCollapsedFieldParent(parentField);
		fieldGroup.setInput(parentAsset, immediateParentField);
	}

	/**
	 * @return the parentFieldDefinition
	 */
	public UIComplexFieldDefinition getParentFieldDefinition() {
		return parentFieldDefinition;
	}

	/**
	 * @return the readOnly
	 */
	public boolean isReadOnly() {
		return readOnly;
	}

	/**
	 * @return the parentAsset
	 */
	public AssetInfo getParentAsset() {
		return parentAsset;
	}

	/**
	 * @return the parentField
	 */
	public UIComplexField getParentField() {
		return parentField;
	}

	/**
	 * Get the title view messages
	 * 
	 * @return The messages
	 */
	TitleViewMessages getViewMessages() {
		return this.messages;
	}

	// ========================================================================
	// ===================== REFRESHING VIEW
	// ========================================================================

	/**
	 * Refresh the view.
	 */
	public void refresh() {
		for (SimpleFieldGroup fieldGroup : simpleFieldGroups) {
			fieldGroup.refresh();
		}
	}

	// ========================================================================
	// ===================== GROUP WIDGET HANDLING
	// ========================================================================

	/**
	 * Add a row to the table for the input field group
	 */
	void addRow(SimpleFieldGroup fieldGroup, Widget[] widgets) {
		if (!this.simpleFieldGroups.contains(fieldGroup))
			return;

		int firstRow = getFirstRowIndex(fieldGroup);
		int rowCount = getRowCount(fieldGroup);
		int row = firstRow + rowCount;

		row = table.insertRow(row);
		int column = 0;
		for (Widget widget : widgets) {
			table.setWidget(row, column, widget);
			table.getCellFormatter().addStyleName(row, column, STYLE_CELL_PREFIX + column);
			column++;
		}

		sizeMap.put(fieldGroup, rowCount + 1);
	}

	/**
	 * Add a row to report the data type mismatch error for the input field group
	 */
	void addMismatchErrorRow(SimpleFieldGroup fieldGroup, MetadataTypeMismatchException error) {
		String displayName = error.getFieldDefinition().getDisplayName();
		LabelWidget label = new LabelWidget(messages.simpleFieldNameLabel(displayName));
		label.addStyleName(SimpleFieldGroup.STYLE_FIELD_LABEL);
		Widget errorWidget = this.buildMetadataMismatchErrorWidget(error);
		addRow(fieldGroup, new Widget[] { label, errorWidget });
	}

	/**
	 * Remove a row from the table for the input field group that contains the input widget
	 */
	void removeRow(SimpleFieldGroup fieldGroup, Widget widget) {
		if (!this.simpleFieldGroups.contains(fieldGroup))
			return;

		int matchingRow = -1;
		int firstRow = getFirstRowIndex(fieldGroup);
		int rowCount = getRowCount(fieldGroup);
		for (int row = firstRow; row < (firstRow + rowCount); row++) {
			for (int column = 0; column < table.getCellCount(row); column++) {
				Widget tableWidget = table.getWidget(row, column);
				if (tableWidget != null && tableWidget.equals(widget)) {
					matchingRow = row;
					break;
				}
			}
		}

		if (matchingRow != -1) {
			table.removeRow(matchingRow);
			sizeMap.put(fieldGroup, rowCount - 1);
		}
	}

	/**
	 * Remove all rows from the table for the field group
	 */
	void removeAllRows(SimpleFieldGroup fieldGroup) {
		if (!this.simpleFieldGroups.contains(fieldGroup))
			return;

		int firstRow = getFirstRowIndex(fieldGroup);
		int rowCount = getRowCount(fieldGroup);
		for (int row = rowCount - 1; row >= 0; row--) {
			table.removeRow(row + firstRow);
		}

		sizeMap.put(fieldGroup, 0);
	}

	/*
	 * Get the number of rows in a simple field group
	 */
	private int getRowCount(SimpleFieldGroup group) {
		Integer size = sizeMap.get(group);
		return ((size != null) ? size.intValue() : 0);
	}

	/*
	 * Get the index of the first row of this field group.
	 */
	private int getFirstRowIndex(SimpleFieldGroup fieldGroup) {
		if (!this.simpleFieldGroups.contains(fieldGroup))
			return -1;

		int row = 0;
		for (SimpleFieldGroup current : simpleFieldGroups) {
			if (current.equals(fieldGroup))
				break;
			row += getRowCount(current);
		}

		return row;
	}

	// ========================================================================
	// ===================== ERROR HANDLING
	// ========================================================================

	/*
	 * Builds a label widget that can be used to display a error message resulting from a data type
	 * mismatch.
	 */
	Widget buildMetadataMismatchErrorWidget(MetadataTypeMismatchException mtme) {
		String typeName = getFieldTypeName(mtme.getFieldDefinition());
		String receivedTypeName = getFieldTypeName(mtme.getReceivedField());
		String message = messages.metadataTypeMismatchError(typeName, receivedTypeName);
		LabelWidget label = new LabelWidget(message);
		label.addStyleName(TitleViewResources.STYLE_DATATYPE_MISMATCH_LABEL);
		return label;
	}

	/*
	 * Get the data type name to display given the field definition
	 */
	private String getFieldTypeName(UIFieldDefinition fieldDefinition) {
		String name = null;
		if (fieldDefinition instanceof UIComplexFieldDefinition) {
			name = messages.complexTypeName();
		} else if (fieldDefinition instanceof UISimpleFieldDefinition) {
			name = ((UISimpleFieldDefinition) fieldDefinition).getDatatype().getName();
		} else {
			name = messages.unknownTypeName();
		}

		return name;
	}

	/*
	 * Get the data type name to display given the field definition
	 */
	private String getFieldTypeName(UIField field) {
		String name = null;
		if (field instanceof UIComplexField) {
			name = messages.complexTypeName();
		} else if (field instanceof UISimpleField) {
			name = ((UISimpleField<?>) field).getDatatype().getName();
		} else {
			name = messages.unknownTypeName();
		}

		return name;
	}

	// ========================================================================
	// ===================== METADATA VALIDATION
	// ========================================================================

	/**
	 * Validate the widgets, and return validation messages for each of the metadata widgets that
	 * fail validation.
	 * 
	 * @param messages the list of validation messages
	 */
	public void validate(List<AssetValidationMessage> messages,boolean draft) {
		for (SimpleFieldGroup fieldGroup : simpleFieldGroups) {
			fieldGroup.validate(messages,draft);
		}
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	@Override
	public HandlerRegistration addViewChangeHandler(ViewChangeHandler handler) {
		return addHandler(handler, ViewChangeEvent.getType());
	}

	/*
	 * Fire the view change event
	 */
	protected void fireViewChangeEvent() {
		fireEvent(new ViewChangeEvent());
	}
}
