/*
 * Created on Jul 2, 2009
 *
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.view.asset.file;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.HasKeyUpHandlers;
import com.google.gwt.event.dom.client.KeyUpEvent;
import com.google.gwt.event.dom.client.KeyUpHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.tandbergtv.cms.portal.content.client.lookupkey.model.UiLookupKey;
import com.tandbergtv.cms.portal.content.client.rpc.lookupkey.ILookupKeyService;
import com.tandbergtv.cms.portal.content.client.rpc.lookupkey.ILookupKeyServiceAsync;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIAssetFile;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIAssetFileField;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UISimpleField;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIStringField;
import com.tandbergtv.cms.portal.content.client.title.view.asset.field.MetadataWidgetBuilder;
import com.tandbergtv.cms.portal.content.client.title.view.asset.field.simple.BooleanFieldWidget;
import com.tandbergtv.cms.portal.content.client.title.view.asset.field.simple.DateFieldWidget;
import com.tandbergtv.cms.portal.content.client.title.view.asset.field.simple.FloatFieldWidget;
import com.tandbergtv.cms.portal.content.client.title.view.asset.field.simple.IntegerFieldWidget;
import com.tandbergtv.cms.portal.content.client.title.view.asset.field.simple.SimpleFieldWidget;
import com.tandbergtv.cms.portal.content.client.title.view.asset.field.simple.StringFieldWidget;
import com.tandbergtv.cms.portal.content.client.title.view.asset.field.simple.TimeFieldWidget;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIAssetFileFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIBooleanFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIDateFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIFloatFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIIntegerFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UISimpleFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIStringFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UITimeFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.option.UIValueOption;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.DataTypeWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.TypedListBoxWidget;

/**
 * Builder that creates the widgets to use in the file table
 *
 * @author Vijay Silva
 */
class FileMetadataWidgetBuilder extends MetadataWidgetBuilder {

	/* The factory to build widgets */
	private FileView fileView;

	/* The context state required by the visitor */
	private UIAssetFile file;
	private UIAssetFileFieldDefinition fileFieldDefinition;
	private UIAssetFileField fileField;

	/* The File URI field name */
	private static final String FILE_URI_FIELD_NAME = "/Fields/Locator/Uri";
	private static final String LOOKUP_KEY_FIELD_NAME = "/Fields/CustomFields/CustomField[@name=LookupKey]/@value";

	/* Dependent style name for widgets created */
	private static final String STYLE_VIEW_SUFFIX = "fileView";
	private static final String STYLE_TYPE_SUFFIX = "fileView-";
	private static final String FILE_URI_TYPE = "fileURI";

	private ILookupKeyServiceAsync lookupKeyService = GWT.create( ILookupKeyService.class );

	/**
	 * Constructor
	 */
	public FileMetadataWidgetBuilder(FileView fileView, UIAssetFile file) {
		this.fileView = fileView;
		((FileDataTypeWidgetFactory) this.factory).setFile(file);
	}

	/*
	 * Use the file widget factory
	 */
	@Override
	protected void initializeFactory() {
		this.factory = new FileDataTypeWidgetFactory();
	}

	/**
	 * Build the widget for the given field definition
	 */
	public SimpleFieldWidget<?, ?, ?> build(UIAssetFile file, UIAssetFileFieldDefinition fieldDefinition,
			UIAssetFileField field, boolean readOnly) {
		/* Set the context state for the widget being built */
		this.file = file;
		this.fileFieldDefinition = fieldDefinition;
		this.fileField = field;

		/* Build the widget */
		UISimpleFieldDefinition simpleDefinition = fileFieldDefinition.getFieldDefinition();
		UISimpleField<?> simpleField = fileField.getField();
		return this.build(simpleDefinition, simpleField, readOnly);
	}

	/*
	 * Get the built widget
	 */
	@Override
	protected SimpleFieldWidget<?, ?, ?> getBuiltWidget() {
		this.file = null;
		this.fileFieldDefinition = null;
		this.fileField = null;

		return super.getBuiltWidget();
	}

	@Override
	public void visit(UIBooleanFieldDefinition definition) {
		super.visit(definition);

		/* Add the required event handlers */
		if (!readOnly) {
			BooleanFieldWidget widget = (BooleanFieldWidget) ((Object) builtWidget);
			widget.addValueChangeHandler(new WidgetValueChangeHandler<Boolean>(fileView, file,
			        fileField, widget));
		}

		builtWidget.addStyleDependentName(STYLE_VIEW_SUFFIX);
		builtWidget.addStyleDependentName(STYLE_TYPE_SUFFIX + definition.getDatatype().getName());
	}

	@Override
	public void visit(UIIntegerFieldDefinition definition) {
		super.visit(definition);

		/* Add the required event handlers */
		if (!readOnly) {
			IntegerFieldWidget widget = (IntegerFieldWidget) ((Object) builtWidget);
			widget.addValueChangeHandler(new WidgetValueChangeHandler<Long>(fileView, file,
			        fileField, widget));
		}

		builtWidget.addStyleDependentName(STYLE_VIEW_SUFFIX);
		builtWidget.addStyleDependentName(STYLE_TYPE_SUFFIX + definition.getDatatype().getName());
	}

	@Override
	public void visit(UIFloatFieldDefinition definition) {
		super.visit(definition);

		/* Add the required event handlers */
		if (!readOnly) {
			FloatFieldWidget widget = (FloatFieldWidget) ((Object) builtWidget);
			widget.addValueChangeHandler(new WidgetValueChangeHandler<String>(fileView, file,
			        fileField, widget));
		}

		builtWidget.addStyleDependentName(STYLE_VIEW_SUFFIX);
		builtWidget.addStyleDependentName(STYLE_TYPE_SUFFIX + definition.getDatatype().getName());
	}

	@Override
	public void visit(UIDateFieldDefinition definition) {
		super.visit(definition);

		/* Add the required event handlers */
		if (!readOnly) {
			DateFieldWidget widget = (DateFieldWidget) ((Object) builtWidget);
			widget.addValueChangeHandler(new WidgetValueChangeHandler<String>(fileView, file,
			        fileField, widget));
		}

		builtWidget.addStyleDependentName(STYLE_VIEW_SUFFIX);
		builtWidget.addStyleDependentName(STYLE_TYPE_SUFFIX + definition.getDatatype().getName());
	}

	@Override
	public void visit(UITimeFieldDefinition definition) {
		super.visit(definition);

		/* Add the required event handlers */
		if (!readOnly) {
			TimeFieldWidget widget = (TimeFieldWidget) ((Object) builtWidget);
			widget.addValueChangeHandler(new WidgetValueChangeHandler<String>(fileView, file,
			        fileField, widget));
		}

		builtWidget.addStyleDependentName(STYLE_VIEW_SUFFIX);
		builtWidget.addStyleDependentName(STYLE_TYPE_SUFFIX + definition.getDatatype().getName());
	}

	@Override
	public void visit(UIStringFieldDefinition definition) {
		boolean isFileURIField = definition.getName().equals(FILE_URI_FIELD_NAME);
		boolean isLookupKeyField = definition.getName().equals(LOOKUP_KEY_FIELD_NAME);

		if (isFileURIField) {
			buildFileURIWidget(definition);
		} else if (isLookupKeyField) {
			buildLookupKeyWidget(definition);
		} else {
			super.visit(definition);
		}

		/* Add the required event handlers */
		if (!readOnly) {
			StringFieldWidget widget = (StringFieldWidget) ((Object) builtWidget);
			String defaultToolTip = widget.getToolTip();
			widget.addValueChangeHandler(new WidgetValueChangeHandler<String>(fileView, file,
			        fileField, widget, defaultToolTip, isFileURIField));

			if (isFileURIField && (widget.getDataTypeWidget() instanceof HasKeyUpHandlers)) {
				HasKeyUpHandlers handler = (HasKeyUpHandlers) widget.getDataTypeWidget();
				handler.addKeyUpHandler(new WidgetKeyHandler(widget, defaultToolTip));
			}
		}

		builtWidget.addStyleDependentName(STYLE_VIEW_SUFFIX);

		/* Change the tool tip if file URI widget */
		if (isFileURIField) {
			String text = builtWidget.getTextValue();
			if (text != null && text.trim().length() > 0) {
				builtWidget.setToolTip(text.trim());
			}
			builtWidget.addStyleDependentName(STYLE_TYPE_SUFFIX + FILE_URI_TYPE);
		} else {
			builtWidget.addStyleDependentName(STYLE_TYPE_SUFFIX
			        + definition.getDatatype().getName());
		}
	}

	/*
	 * Build the File URI Widget
	 */
	private void buildFileURIWidget(UIStringFieldDefinition definition) {
		UIStringField field = getSimpleField(definition, this.field, UIStringField.class);
		FileURIWidgetConfiguration configuration = new FileURIWidgetConfiguration(definition);
		configuration.setFilePaths(fileView.getFilePaths());
		DataTypeWidget<String> typedWidget = factory.createStringWidget(configuration);
		this.initializeWidgetValue(typedWidget, field);
		StringFieldWidget widget = new StringFieldWidget(definition, field, typedWidget);
		widget.setReadOnly(readOnly);

		builtWidget = widget;
	}

	/**
	 * Builds lookup key widget
	 * @param definition
	 */
	private void buildLookupKeyWidget(UIStringFieldDefinition definition) {
		UIStringField field = getSimpleField(definition, this.field, UIStringField.class);

		// Configuration
		final LookupKeyWidgetConfiguration configuration = new LookupKeyWidgetConfiguration(definition);
		configuration.setOptions( new ArrayList<UIValueOption>() );

		DataTypeWidget<String> typedWidget = factory.createStringWidget(configuration);
                this.initializeWidgetValue(typedWidget, field);

                final StringFieldWidget widget = new StringFieldWidget(definition, field, typedWidget);
                widget.setReadOnly(readOnly);

                builtWidget = widget;

		// Get lookup keys for all assets from lookup key manager
		String assetType = fileFieldDefinition.getAssetFileDefinition().getParent().getAssetType();
	        lookupKeyService.getLookupKeysByAssetType( assetType, new NeptuneAsyncCallback<List<UiLookupKey>>() {

	            @Override
	            public void onNeptuneFailure( Throwable caught ) {
	                // nothing to be done, the listbox will be empty
	            }

	            @Override
	            public void onNeptuneSuccess( List<UiLookupKey> result ) {
	                TypedListBoxWidget<String> listBox = ((TypedListBoxWidget<String>) widget.getDataTypeWidget());
	                // Add the 'Options' to the listBox
	                for ( UiLookupKey key : result ) {
	                    configuration.getOptions().add( new UIValueOption( key.getName(), key.getName() ) );
	                    listBox.addItem( key.getName(), key.getName() );
	                }
	            }
	        } );

	}

	/*
	 * Event handling class to handle the value change event fired by the file field widgets
	 */
	private static final class WidgetValueChangeHandler<DataType> implements
	        ValueChangeHandler<DataType> {

		private FileView view;
		private UIAssetFile file;
		private UIAssetFileField field;
		private SimpleFieldWidget<?, ?, DataType> widget;
		private String defaultToolTip;
		private boolean showValueAsToolTip;

		/**
		 * Constructor
		 */
		public WidgetValueChangeHandler(FileView view, UIAssetFile file, UIAssetFileField field,
		        SimpleFieldWidget<?, ?, DataType> widget) {
			this(view, file, field, widget, null, false);
		}

		/**
		 * Constructor
		 */
		public WidgetValueChangeHandler(FileView view, UIAssetFile file, UIAssetFileField field,
		        SimpleFieldWidget<?, ?, DataType> widget, String defaultToolTip,
		        boolean showValueAsToolTip) {
			this.view = view;
			this.file = file;
			this.field = field;
			this.widget = widget;
			this.defaultToolTip = defaultToolTip;
			this.showValueAsToolTip = showValueAsToolTip;
		}

		@Override
		public void onValueChange(ValueChangeEvent<DataType> event) {
			/* Set the typed value, and clear the 'string' value */
			UISimpleField<DataType> simpleField = widget.getField();
			simpleField.setValue(event.getValue());

			if (showValueAsToolTip) {
				String toolTip = defaultToolTip;
				String text = widget.getTextValue();
				if (text != null && text.trim().length() > 0)
					toolTip = text.trim();
				widget.setToolTip(toolTip);
			}

			view.handleMetadataValueChanged(file, field, widget);
		}
	}

	/*
	 * Event handling class to handle updating the tool tip to the value displayed by widget
	 */
	private static final class WidgetKeyHandler implements KeyUpHandler {

		private SimpleFieldWidget<?, ?, ?> widget;
		private String defaultToolTip;

		/**
		 * Constructor
		 */
		public WidgetKeyHandler(SimpleFieldWidget<?, ?, ?> widget, String defaultToolTip) {
			this.widget = widget;
			this.defaultToolTip = defaultToolTip;
		}

		@Override
		public void onKeyUp(KeyUpEvent event) {
			String toolTip = defaultToolTip;
			String text = widget.getTextValue();
			if (text != null && text.trim().length() > 0)
				toolTip = text.trim();
			widget.setToolTip(toolTip);
		}
	}
}
