/*
 * Created on Jun 30, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.view.asset.file;

import static com.tandbergtv.cms.portal.content.client.title.view.asset.AssetContentTable.CELL_DELETE_COLUMN_STYLE;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.HasKeyUpHandlers;
import com.google.gwt.event.dom.client.KeyUpEvent;
import com.google.gwt.event.dom.client.KeyUpHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.http.client.URL;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.FocusWidget;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Panel;
import com.google.gwt.user.client.ui.PopupPanel;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.ContentComponent;
import com.tandbergtv.cms.portal.content.client.title.model.UIImageEditor;
import com.tandbergtv.cms.portal.content.client.title.model.UISftpUpload;
import com.tandbergtv.cms.portal.content.client.title.model.UIVideoPlayer;
import com.tandbergtv.cms.portal.content.client.title.model.UploadedFileTitleInfo;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.UITitleValidationMessage;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIAssetFile;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIAssetFileField;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIAssetFilePath;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIComplexField;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIField;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UISimpleField;
import com.tandbergtv.cms.portal.content.client.title.service.ISftpUploadViewService;
import com.tandbergtv.cms.portal.content.client.title.service.ISftpUploadViewServiceAsync;
import com.tandbergtv.cms.portal.content.client.title.service.IVideoPlayerService;
import com.tandbergtv.cms.portal.content.client.title.service.IVideoPlayerServiceAsync;
import com.tandbergtv.cms.portal.content.client.title.service.ImageEditorService;
import com.tandbergtv.cms.portal.content.client.title.service.ImageEditorServiceAsync;
import com.tandbergtv.cms.portal.content.client.title.service.SaveImageRequest;
import com.tandbergtv.cms.portal.content.client.title.service.asset.IUIAssetFactory;
import com.tandbergtv.cms.portal.content.client.title.service.asset.UIAssetFactory;
import com.tandbergtv.cms.portal.content.client.title.view.HasViewChangeHandlers;
import com.tandbergtv.cms.portal.content.client.title.view.TitleView;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewMessageCache;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewMessages;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewResources;
import com.tandbergtv.cms.portal.content.client.title.view.ViewChangeEvent;
import com.tandbergtv.cms.portal.content.client.title.view.ViewChangeHandler;
import com.tandbergtv.cms.portal.content.client.title.view.asset.AssetContentTable;
import com.tandbergtv.cms.portal.content.client.title.view.asset.AssetTree.AssetInfo;
import com.tandbergtv.cms.portal.content.client.title.view.asset.AssetValidationMessage;
import com.tandbergtv.cms.portal.content.client.title.view.asset.MetadataTypeMismatchException;
import com.tandbergtv.cms.portal.content.client.title.view.asset.field.simple.SimpleFieldWidget;
import com.tandbergtv.cms.portal.content.client.title.view.imageeditor.EditImageDialogBox;
import com.tandbergtv.cms.portal.content.client.title.view.imageeditor.PreviewImageDialogBox;
import com.tandbergtv.cms.portal.content.client.title.view.metadata.TitleValidationViewInput;
import com.tandbergtv.cms.portal.content.client.title.view.videoplayer.JWPlayerWidget;
import com.tandbergtv.cms.portal.content.client.title.view.videoplayer.VideoPreviewDialog;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIAssetFileDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIAssetFileFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIComplexFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIFieldDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UISimpleFieldDefinition;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ListBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.PushButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.DisclosureContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.RoundedDisclosureContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.ScrollContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.DataTypeWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

/**
 * The File View for an Asset
 * 
 * @author Vijay Silva
 */
public class FileView extends Composite implements HasViewChangeHandlers {
    private TitleView titleView;
	
	/* Widgets */
	private DisclosureContainer disclosureContainer;
	private SimpleContainer container;
	private AssetContentTable table;
	private TitleViewMessages messages;
	private ButtonWidget uploadBtn;
	private Set<Button> editImageButtons = new HashSet<Button>();
	private boolean dirty = false;
	
	/* Model */
	private final UIAssetFileDefinition fileDefinition;
	private final List<UIAssetFilePath> filePaths;
	private AssetInfo input;
	private boolean readOnly = false;
	private Set<Long> fileIDs = new HashSet<Long>();
	private long generatedFileID = -1;
	private IUIAssetFactory assetFactory = new UIAssetFactory();
	private UIAssetFile newFile = null;
	private List<HandlerRegistration> newFileHandlers = new ArrayList<HandlerRegistration>();

	/* Mapping UI with model */
	private Map<UIAssetFile, LabelWidget> rowIdentifier = new HashMap<UIAssetFile, LabelWidget>();
	private Map<String, Integer> columnIndices = new HashMap<String, Integer>();

	/* Parameters used for initializing NetImager */
	private static String address;
	private static String organization;
	private static String company;
	private static String licenseString1;
	private static String licenseString2;
	private static boolean licensedForImageEditing;

	/* Parameters used for initializing File assets */
	private static boolean licensedForVideoPreview;
	private List<String> imageAssetTypes;
	private List<String> videoAssetTypes;
	
	static TitleValidationViewInput viewInputFileUpload;
	
	/* Column Names */
	private static final String FILE_ID_COLUMN = "FileID";
	private static final String METADATA_COLUMN_PREFIX = "Metadata_";
	private static final String PARENT_FILE_ID_COLUMN = "ParentFileID";
	private static final String DELETE_BUTTON_COLUMN = "Delete";

	/* style names */
	private static final String STYLE_NAME = "content-FileView";
	private static final String STYLE_DISCLOSURE_CONTAINER = "content-FileView-disclosureContainer";
	private static final String STYLE_DISCLOSURE_CONTENTS = "content-FileView-disclosureContents";
	private static final String STYLE_SCROLL_CONTENTS = "content-FileView-scrollContents";
	private static final String STYLE_CELL_FILE_ID = "content-FileView-fileIdValueCell";
	private static final String STYLE_FILE_ID_LABEL = "content-FileView-fileIdValueLabel";
	private static final String STYLE_PARENT_FILE_WIDGET = "content-FileView-parentFileListBox";

	private static final String FILE_URI_XPATH = "/Fields/Locator/Uri";
	private static final String FILE_ENCODEFORMAT_XPATH = "/Fields/CustomFields/CustomField[@name=EncodeFormat]/@value";
	private static final String FILE_LOOKUP_KEY_XPATH = "/Fields/CustomFields/CustomField[@name=LookupKey]/@value";
	private static final String GET_IMAGE_JSP_PATH = "/portal/cms_contentmgmt_ui/jsps/get_image.jsp?fileURI=";
	private static final String FILE_URI_ROOT = "";
		
	private ISftpUploadViewServiceAsync sftpViewService = GWT.create(ISftpUploadViewService.class);	
	private IVideoPlayerServiceAsync videoPlayerService = GWT.create(IVideoPlayerService.class);	
	private ImageEditorServiceAsync imageEditorService = GWT.create(ImageEditorService.class);	
	
	/**
	 * The File View Constructor
	 * 
	 * @param fileDefinition The File Definition
	 * @param readOnly true if the view if read only, false otherwise
	 */
	public FileView(TitleView titleView, UIAssetFileDefinition fileDefinition, List<UIAssetFilePath> filePaths,
	        boolean readOnly) {
		this.titleView = titleView;
		/* The state */
		this.fileDefinition = fileDefinition;
		this.filePaths = filePaths;
		this.readOnly = readOnly;

		/* The constants */
		this.messages = TitleViewMessageCache.getMessages();

		/* The widgets */
		container = new SimpleContainer();
		container.setStylePrimaryName(STYLE_NAME);
		initWidget(container);

		/* Initialize the table */
		initialize();
	}

	/*
	 * Initialize the File View Table and its columns
	 */
	private void initialize() {
		disclosureContainer = new RoundedDisclosureContainer(messages.fileSectionHeading(), true);
		disclosureContainer.addStyleName(STYLE_DISCLOSURE_CONTAINER);
		container.setWidget(disclosureContainer);

		VerticalContainer tableWrapper = new VerticalContainer();
		tableWrapper.addStyleName(STYLE_SCROLL_CONTENTS);
		table = new AssetContentTable();
		tableWrapper.add(table);
		initUploadBtn();
		tableWrapper.add(uploadBtn);

		ScrollContainer disclosureContents = new ScrollContainer(tableWrapper);
		disclosureContents.addStyleName(STYLE_DISCLOSURE_CONTENTS);
		disclosureContainer.setContent(disclosureContents);
		int columnIndex = 0;

		/* Add the file ID column */
		table.addHeading(messages.fileIdColumn());
		columnIndices.put(FILE_ID_COLUMN, columnIndex++);

		/* Add the File metadata columns */
		for (UIAssetFileFieldDefinition fieldDefinition : fileDefinition.getFieldDefinitions()) {
			table.addHeading(fieldDefinition.getDisplayName());
			columnIndices.put(METADATA_COLUMN_PREFIX + fieldDefinition.getName(), columnIndex++);
		}

		/* Add the parent file ID column */
		table.addHeading(messages.parentFileIdColumn());
		columnIndices.put(PARENT_FILE_ID_COLUMN, columnIndex++);
		columnIndices.put(DELETE_BUTTON_COLUMN, columnIndex++);

		NeptuneApplication application = NeptuneApplication.getApplication();
		ContentComponent component = application.getComponent(ContentComponent.class);
		imageAssetTypes = component.getImageAssetTypes();
		videoAssetTypes = component.getVideoAssetTypes();
		licensedForVideoPreview = component.isLicensedForVideoPreview();
		licensedForImageEditing = component.isLicensedForImageEditing();
		
		if (JWPlayerWidget.isPlayableFormatsLoaded() == false)
			loadVideoPlayableFormats();
	}

	private void loadImageEditorParameters() {
		final BusyIndicator busyIndicator = new BusyIndicator(this);
		final FileView parentView = this;
		imageEditorService.getSettings(new NeptuneAsyncCallback<UIImageEditor>() {
			/* Success getting settings */
			public void onNeptuneSuccess(UIImageEditor confSettings) {
				busyIndicator.hide();		
				Map<String,String>parms = confSettings.getParameters();
				address = parms.get("address") == null ? "" : parms.get("address").trim();
				organization = parms.get("organization") == null ? "" : parms.get("organization").trim();
				company = parms.get("company") == null ? "" : parms.get("company").trim();
				licenseString1 = parms.get("license_1") == null ? "" : parms.get("license_1").trim();
				licenseString2 = parms.get("license_2") == null ? "" : parms.get("license_2").trim();
			};

			/* Failure getting conf settings from backend */
			public void onNeptuneFailure(Throwable caught) {
				busyIndicator.hide();
				parentView.handleGetImageEditorSettingsFailure(caught);
			};
		});			
	}

	private void loadVideoPlayableFormats() {
		final BusyIndicator busyIndicator = new BusyIndicator(this);
		final FileView parentView = this;
		videoPlayerService.getSettings(new NeptuneAsyncCallback<UIVideoPlayer>() {
			/* Success getting settings */
			public void onNeptuneSuccess(UIVideoPlayer confSettings) {
				busyIndicator.hide();				
				Map<String,String> vars = confSettings.getBasicVars();
				JWPlayerWidget.loadPlayableFormats (vars);
				loadImageEditorParameters ();
			};

			/* Failure getting conf settings from backend */
			public void onNeptuneFailure(Throwable caught) {
				busyIndicator.hide();
				parentView.handleGetVideoSettingsFailure(caught);
			};
		});			
	}
	
	
	/*
	 * Initialize upload button
	 */
	private void initUploadBtn() {
		uploadBtn = new ButtonWidget(messages.fileUploadButtonText());
		uploadBtn.setEnabled(!isReadOnly());
		uploadBtn.addStyleDependentName(StyleNames.COMMIT_BUTTON_STYLE);
		final BusyIndicator busyIndicator = new BusyIndicator(this);
		final FileView parentView = this;
		uploadBtn.addClickHandler(new ClickHandler() {
			public void onClick(ClickEvent event) {
				busyIndicator.center();

				String providerId = input.getDefinition().getSpecification().getProviderId();
				sftpViewService.getSettings(providerId, new NeptuneAsyncCallback<UISftpUpload>() {
					/* Success getting settings */
					public void onNeptuneSuccess(UISftpUpload confSettings) {
						busyIndicator.hide();
						Map<String,String> regParameters = confSettings.getAppltRegParams();
						Map<String,String> xtraParameters = confSettings.getAppltXtrParams();

						FileUploadDialog dlg = new FileUploadDialog(parentView, regParameters, xtraParameters);
						dlg.center();
					};

					/* Failure getting conf settings from backend */
					public void onNeptuneFailure(Throwable caught) {
						busyIndicator.hide();
						Window.alert(messages.fileUploadWidgetGettingConfError());
					};
				});			
			}
			
		});
		
		return;
	}	

	/*
	 * backend call to register uploaded file
	 */
	void saveUploadedFileAssets(List<String> fileUrls) {
		final BusyIndicator busyIndicator = new BusyIndicator(this);
		busyIndicator.center();
		final Long assetId = this.input.getAsset().getId();
		final Long titleId = this.input.getInput().getTitleId();
		
		sftpViewService.saveUploadedFileAssets(assetId, titleId, fileUrls,
		        new NeptuneAsyncCallback<UploadedFileTitleInfo>() {
			        @Override
			        public void onNeptuneSuccess(UploadedFileTitleInfo result) {
						busyIndicator.hide();
						if (result.getValidationStatus() == true)
							handleSaveFileAssetsSuccess(assetId);
						else
							handleSaveFileAssetsSuccessWithWarning(assetId,result.getValidationMessages());
			        }

			        @Override
			        public void onNeptuneFailure(Throwable caught) {
						busyIndicator.hide();
			        	handleSaveFileAssetsFailure(assetId, caught);
			        }
		});
		return;
	}	
	
	void handleSaveFileAssetsSuccess(Long assetId) 
	{
		titleView.getInput().invalidateMetadata(); // refresh will make rpc call to backend
		titleView.refresh();

		TitleValidationViewInput input = new TitleValidationViewInput();
		input.setInfoHeader(messages.fileUploadResgisterSuccessful(assetId.toString()));
		titleView.showMessage(input);
		titleView.getMetadataTab().updateAssetPanelHeight();
	}

	void handleSaveFileAssetsSuccessWithWarning(Long assetId, List<UITitleValidationMessage> valMsgs) 
	{
		viewInputFileUpload = new TitleValidationViewInput();
		viewInputFileUpload.setInfoHeader(messages.fileUploadWithValidationErrors());
		if(!valMsgs.isEmpty())
		{
			viewInputFileUpload.getTitleValidationMessages().addAll(valMsgs);
		}

		titleView.getInput().invalidateMetadata(); // refresh will make rpc call to backend
		titleView.refresh();
	}
	
	void handleSaveFileAssetsFailure(Long assetId, Throwable caught) 
	{
		TitleValidationViewInput input = new TitleValidationViewInput();
		input.setErrorHeader(messages.fileUploadResgisterError(assetId.toString()));
		titleView.showMessage(input);
		titleView.getMetadataTab().updateAssetPanelHeight();
	}

	void handleFileUploadCanceled() 
	{
		Long assetId = this.input.getAsset().getId();
		TitleValidationViewInput input = new TitleValidationViewInput();
		input.setInfoHeader(messages.fileUploadCanceled(assetId.toString()));
		titleView.showMessage(input);
		titleView.getMetadataTab().updateAssetPanelHeight();
	}

	void handleFileUploadError() 
	{
		Long assetId = this.input.getAsset().getId();
		TitleValidationViewInput input = new TitleValidationViewInput();
		input.setErrorHeader(messages.fileUploadError(assetId.toString()));
		titleView.showMessage(input);
		titleView.getMetadataTab().updateAssetPanelHeight();
	}
	
	void showUploadedFileMessage() {
		titleView.showMessage(viewInputFileUpload);
		titleView.getMetadataTab().updateAssetPanelHeight();	
		viewInputFileUpload = null;//clear after display
	}
	
	// ========================================================================
	// ===================== UPDATING INPUT
	// ========================================================================

	/**
	 * Set the input asset for this view
	 * 
	 * @param asset The asset for which the files need to be displayed
	 */
	public void setInput(AssetInfo asset) {
		this.input = asset;
		this.updateFileIDs();

		/* update the table view */
		refresh();
	}

	/**
	 * Determine if this widget is read-only
	 * 
	 * @return true if the widget is read-only, false otherwise
	 */
	public boolean isReadOnly() {
		return this.readOnly;
	}

	/**
	 * @return The list of permitted file paths
	 */
	List<UIAssetFilePath> getFilePaths() {
		return this.filePaths;
	}

	// ========================================================================
	// ===================== FILE ID GENERATION
	// ========================================================================

	/*
	 * Update the cached list of file IDs are generate new file IDs for the files that don't have
	 * assigned IDs.
	 */
	private void updateFileIDs() {
		this.fileIDs.clear();
		this.generatedFileID = -1;

		/* Nothing to do if no input is present */
		if (this.input == null)
			return;

		/* Build the set of unique file IDs for existing files */
		for (UIAssetFile file : input.getAsset().getFiles()) {
			if (file.getId() != null) {
				fileIDs.add(file.getId());
				long absoluteFileId = getAbsoluteValue(file.getId());
				if (getAbsoluteValue(generatedFileID) <= absoluteFileId) {
					generatedFileID = -(++absoluteFileId);
				}
			}
		}

		/* Generate the file IDs for new files */
		for (UIAssetFile file : input.getAsset().getFiles()) {
			if (file.getId() == null) {
				generateFileID(file);
			}
		}
	}

	/*
	 * Get the absolute value of a number
	 */
	private long getAbsoluteValue(long value) {
		return (value < 0) ? -value : value;
	}

	/*
	 * Generate a new unique File ID
	 */
	private void generateFileID(UIAssetFile file) {
		file.setId(generatedFileID--);
		fileIDs.add(file.getId());
	}

	// ========================================================================
	// ===================== REFRESHING VIEW
	// ========================================================================

	/**
	 * Refresh the view, redrawing.
	 */
	public void refresh() {
		/* Ensure that the disclosure panel is open */
		disclosureContainer.setOpen(true);

		/* Remove all existing rendered rows */
		table.removeAllRows();
		rowIdentifier.clear();
		editImageButtons.clear();
		this.clearNewFileState();

		/* Disable the upload button */
		uploadBtn.setEnabled(false);

		/* There is no input, do nothing */
		if (input == null)
			return;

		/* Add the rows for each of the files to render */
		for (UIAssetFile file : input.getAsset().getFiles()) {
			buildRow(file);
		}

		/* Always build the new row view */
		addNewFileRow();

		/* Update the enabled buttons */
		updateEnabledButtons();
		
		if (viewInputFileUpload != null)
			showUploadedFileMessage();
			
	}

	/*
	 * Build a new row for the file
	 */
	private void buildRow(UIAssetFile file) {
		int row = table.addRow();

		/* The File ID */
		LabelWidget fileIDLabel = new LabelWidget(Long.toString(getAbsoluteValue(file.getId())));
		fileIDLabel.addStyleName(STYLE_FILE_ID_LABEL);
		table.addCell(row, fileIDLabel);
		table.addTableCellStyle(row, table.getCellCount(row) - 1, STYLE_CELL_FILE_ID);
		rowIdentifier.put(file, fileIDLabel);

		/* File Metadata */
		for (UIAssetFileFieldDefinition fieldDefinition : fileDefinition.getFieldDefinitions()) {
			Widget widget = buildMetadataWidget(file, fieldDefinition);
			table.addCell(row, widget);
		}

		/* The File Parent ID */
		Widget parentFileWidget = buildParentFileWidget(file);
		table.addCell(row, parentFileWidget);

		/* The Delete Button */
		Widget deleteButton = buildDeleteButton(file);
		
		table.addCell(row, deleteButton);
		table.addTableCellStyle(row, table.getCellCount(row) - 1, CELL_DELETE_COLUMN_STYLE);
	}

	// ========================================================================
	// ===================== DIRTY VIEW
	// ========================================================================

	/**
	 * Mark this view as dirty
	 */
	public void markDirty() {
		this.dirty = true;
		updateEnabledButtons();
	}

	/**
	 * Check if this view has been marked as dirty
	 * 
	 * @return true if marked dirty, false otherwise
	 */
	public boolean isDirty() {
		return this.dirty;
	}
	
	/*
	 * Update the buttons that have to be enabled based on the dirty flag
	 */
	private void updateEnabledButtons() {
		boolean existingTitle = (input != null && input.getInput().isExistingTitle());
		boolean enabled = !isReadOnly() && existingTitle && !dirty;
		this.uploadBtn.setEnabled(enabled);
		if (dirty)
			uploadBtn.setTitle(messages.fileUploadDisableMessage());	
		else
			uploadBtn.setTitle("");	
		
		enabled &= licensedForImageEditing;
		for (Button button : editImageButtons) {
			button.setEnabled(enabled);
		}
	}

	// ========================================================================
	// ===================== NEW ROW HANDLING
	// ========================================================================

	/*
	 * Build a new file
	 */
	void addNewFileRow() {
		/* Create the new file but don't add to the input / model */
		newFile = assetFactory.createAssetFile(this.fileDefinition);

		/* generate a new ID */
		generateFileID(newFile);

		/* build the row / view */
		buildRow(newFile);
	}

		
	/*
	 * Clear the new file created previously, and remove all registered handlers
	 */
	private void clearNewFileState() {
		newFile = null;

		/* Remove all handlers added only for the new file row */
		for (HandlerRegistration registration : newFileHandlers) {
			registration.removeHandler();
		}
		newFileHandlers.clear();
	}

	/*
	 * After widget is updated causing the model to update, check if a new row needs to be added
	 */
	private void handleFileUpdated(UIAssetFile file) {
		/* If the file updated is the new file, add file and create new row */
		if (file.equals(newFile)) {
			/* Add new file to model */
			input.getAsset().addFile(file);
			clearNewFileState();

			/* Update the view */
			updateParentFileWidgets();
			showDeleteButton(file);
			addNewFileRow();
		}

		fireViewChangeEvent();
	}

	/*
	 * After a file has been deleted from model, update all row widgets with any changes required
	 */
	private void handleFileDeleted(UIAssetFile file) {
		/* Update the view */
		updateParentFileWidgets();
		fireViewChangeEvent();
	}

	// ========================================================================
	// ===================== PARENT FILE WIDGET
	// ========================================================================

	/* Build the list box widget for the parent file column */
	private Widget buildParentFileWidget(final UIAssetFile file) {
		final ListBoxWidget<UIAssetFile> widget = new ListBoxWidget<UIAssetFile>();
		widget.addStyleName(STYLE_PARENT_FILE_WIDGET);

		/* Update the items of the widget and select the correct item */
		updateParentFileWidgetItems(file, widget);

		/* Determine if the field can be edited */
		Long fileID = file.getId();
		boolean isReadOnly = (readOnly || (fileID != null && fileID > -1));
		widget.setEnabled(!isReadOnly);

		/* Add listener for change if not read-only */
		widget.addChangeHandler(new ChangeHandler() {

			@Override
			public void onChange(ChangeEvent event) {
				handleParentFileChanged(file, widget, event);
			}
		});

		return widget;
	}

	/*
	 * Update all parent file widgets to re-build the list of parent files
	 */
	@SuppressWarnings("unchecked")
	private void updateParentFileWidgets() {
		ListBoxWidget<UIAssetFile> widget = null;
		for (UIAssetFile file : rowIdentifier.keySet()) {
			int row = table.getRow(rowIdentifier.get(file));
			int column = columnIndices.get(PARENT_FILE_ID_COLUMN);
			widget = (ListBoxWidget<UIAssetFile>) table.getCellWidget(row, column);
			updateParentFileWidgetItems(file, widget);
		}
	}

	/*
	 * Update the items displayed in the widget, and update the selected item
	 */
	private void updateParentFileWidgetItems(UIAssetFile file, ListBoxWidget<UIAssetFile> widget) {
		widget.clear();
		widget.addItem("", null);
		for (UIAssetFile otherFile : input.getAsset().getFiles()) {
			if (!otherFile.equals(file))
				widget.addItem(Long.toString(getAbsoluteValue(otherFile.getId())), otherFile);
		}

		/* Set the selected item */
		UIAssetFile parentFile = file.getParentFile();
		int index = widget.getIndex(parentFile);
		if (index == -1) {
			widget.addItem(Long.toString(getAbsoluteValue(parentFile.getId())), parentFile);
		}
		widget.setSelectedItem(parentFile);
	}

	/*
	 * Handle a value change in parent file widget
	 */
	private void handleParentFileChanged(UIAssetFile file, ListBoxWidget<UIAssetFile> widget,
	        ChangeEvent event) {
		UIAssetFile newParent = widget.getSelectedItem();

		/* Validate parent file change */
		if (!validateParentFileUpdate(file, newParent)) {
			/* Revert the data */
			widget.setSelectedItem(file.getParentFile());
			return;
		}

		file.setParentFile(newParent);
		handleFileUpdated(file);
	}

	/*
	 * Validate that the parent file can be updated
	 */
	private boolean validateParentFileUpdate(UIAssetFile file, UIAssetFile newParent) {
		/* read only, cannot update */
		if (readOnly)
			return false;

		/* check if existing file, cannot update */
		if (file.getId() != null && file.getId() > -1) {
			Window.alert(messages.fileParentChangeError());
			return false;
		}

		/* Ensure not 'loops' in tree with this update */
		if (isDescendant(file, newParent)) {
			Window.alert(messages.fileParentDescendantError());
			return false;
		}

		return true;
	}

	/*
	 * Determine if the file is a descendant of an existing file
	 */
	private boolean isDescendant(UIAssetFile file, UIAssetFile descendant) {
		List<UIAssetFile> descendants = new ArrayList<UIAssetFile>();
		descendants.add(file);
		int index = 0;
		while (index < descendants.size()) {
			UIAssetFile current = descendants.get(index);
			for (UIAssetFile inputFile : input.getAsset().getFiles()) {
				if (current.equals(inputFile.getParentFile()) && !descendants.contains(inputFile)) {
					descendants.add(inputFile);
				}
			}
			index++;
		}

		return descendants.contains(descendant);
	}

	// ========================================================================
	// ===================== DELETE BUTTON WIDGET
	// ========================================================================

	/*
	 * Build the widget for the delete button along with event handling
	 */
	private Widget buildDeleteButton(final UIAssetFile file) {
		ImageWidget deleteImage = new ImageWidget(TitleViewResources.DELETE_IMAGE_URL);
		final PushButtonWidget button = new PushButtonWidget(deleteImage);
		button.addStyleName(TitleViewResources.STYLE_FIELD_DELETE_BUTTON);
		boolean isReadOnly = (readOnly || file.equals(newFile));

		button.setEnabled(!isReadOnly);
		button.setVisible(!isReadOnly);

		/* Event handler */
		button.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleDeleteButtonClicked(file, button, event);
			}
		});

		return button;
	}

	/*
	 * Show and enable the delete button for a file
	 */
	private void showDeleteButton(UIAssetFile file) {
		int row = table.getRow(rowIdentifier.get(file));
		int column = columnIndices.get(DELETE_BUTTON_COLUMN);
		FocusWidget button = (FocusWidget) table.getCellWidget(row, column);
		button.setEnabled(true);
		button.setVisible(true);
	}

	/*
	 * Event handling for the delete button click
	 */
	private void handleDeleteButtonClicked(UIAssetFile file, Widget button, ClickEvent event) {
		/* Validate that the file can be deleted */
		if (!validateFileDelete(file)) {
			return;
		}

		int row = table.getRow(button);
		table.removeRow(row);
		input.getAsset().getFiles().remove(file);
		rowIdentifier.remove(file);

		/* Handle the delete file for other widgets */
		handleFileDeleted(file);
	}

	/*
	 * Ensure that the file can be deleted
	 */
	private boolean validateFileDelete(UIAssetFile file) 
	{
		/* Read only, cannot delete */
		if (readOnly || file.equals(newFile)) 
		{
			return false;
		}

		/* The file must have no children */
		for (UIAssetFile inputFile : input.getAsset().getFiles()) 
		{
			if (file.equals(inputFile.getParentFile())) 
			{
				TitleValidationViewInput input = new TitleValidationViewInput();
				input.setErrorHeader(messages.fileDeleteWithChildrenError());
				titleView.showMessage(input);
				titleView.getMetadataTab().updateAssetPanelHeight();
				
				return false;
			}
		}

		return true;
	}

	// ========================================================================
	// ===================== FILE METADATA WIDGET
	// ========================================================================

	/*
	 * Build the metadata widget
	 */
	private Widget buildMetadataWidget(UIAssetFile file, UIAssetFileFieldDefinition definition) {
		
		UIAssetFileField encodeFormatField = file.getFields().get(FILE_ENCODEFORMAT_XPATH);
		UIAssetFileField lookupKeyField = file.getFields().get(FILE_LOOKUP_KEY_XPATH);
		
		/* Get the field */
		UIAssetFileField field = file.getFields().get(definition.getName());
		if (field == null) {
			field = assetFactory.createAssetFileField(definition);
			file.addField(field);
		}

		/* Build the widget */
		Panel panel = new HorizontalPanel();
		FileMetadataWidgetBuilder builder = new FileMetadataWidgetBuilder(this, file);
		SimpleFieldWidget<?, ?, ?> fieldWidget = null;
		try {
			fieldWidget = builder.build(file, definition, field, readOnly);
			panel.add(fieldWidget);
			
			String fileURI = FILE_URI_ROOT;
			final String assetType = input.getAsset().getAssetType();
			String fieldName = definition.getName();

			/* Video Asset, File URI Field - show preview button */
			if (videoAssetTypes.contains(assetType) && FILE_URI_XPATH.equals(fieldName)) {
				addVideoPreviewButton(panel, file, field,encodeFormatField);
			}
			/* Image Asset, File URI Field - show preview / edit buttons */
			else if (imageAssetTypes.contains(assetType) && FILE_URI_XPATH.equals(fieldName)) {
				Button previewButton = new Button(messages.previewImageButtontext());
				previewButton.addStyleDependentName(StyleNames.ACT_TOWARDS_SAVE_BUTTON_STYLE);
				
				Button editImageButton = new Button(messages.editImageButtonText());
				editImageButton.addStyleDependentName(StyleNames.ACT_TOWARDS_SAVE_BUTTON_STYLE);
				
				editImageButtons.add(editImageButton);
				editImageButton.setEnabled(!isReadOnly());

				/* Add buttons to panel */
				panel.add(previewButton);
				panel.add(editImageButton);
				
				String fieldValue = field.getField().getDisplayValue();
				if (fieldValue == null || fieldValue.isEmpty()) {
					previewButton.setEnabled(false);
					editImageButton.setEnabled(false);
				} else {
					if (licensedForImageEditing == false)
						editImageButton.setEnabled(false);
					
					String originalFilePathFieldValue = String.valueOf(fieldValue);
					UIAssetFile parentFile = file.getParentFile();
					if (parentFile != null) {
						UIAssetFileField parentFilefield = parentFile.getFields().get(definition.getName());
						Object parentFileFieldValue = parentFilefield.getField().getDisplayValue();
						originalFilePathFieldValue = String.valueOf(parentFileFieldValue);
					}
					
					final String originalFilePath = originalFilePathFieldValue;
					
					fileURI += String.valueOf(fieldValue);
					int lastIndexOfFileSeparator = fileURI.lastIndexOf("/");
					// This could be -1 if on windows platform
					if (lastIndexOfFileSeparator < 0) {
						lastIndexOfFileSeparator = fileURI.lastIndexOf("\\");
					}
					final String imageFolder = fileURI.substring(0, lastIndexOfFileSeparator);
					String encodedFileURI = URL.encodeQueryString(fileURI);
					final String imageUrl = Window.Location.getProtocol() + "//"
							+ Window.Location.getHost() + GET_IMAGE_JSP_PATH + encodedFileURI;

					final Long assetId = file.getParentAsset().getId();
					final Long titleId = input.getInput().getTitleId();
					String lookupKeyFieldValue = null;
					if ((lookupKeyField != null) && (lookupKeyField.getField() != null)) {
						lookupKeyFieldValue = String.valueOf(lookupKeyField.getField().getValue());
					}
					final String lookupKey = lookupKeyFieldValue;		
					
					editImageButton.addClickHandler(new ClickHandler() {
						public void onClick(ClickEvent event) {
							
							final EditImageDialogBox editImageDialogBox = new EditImageDialogBox(
									originalFilePath, assetType, lookupKey, false, true);
							editImageDialogBox.setPopupPositionAndShow(new PopupPanel.PositionCallback() {
						          public void setPosition(int offsetWidth, int offsetHeight) {
						            int left = (Window.getClientWidth() - offsetWidth) / 3;
						            int top = (Window.getClientHeight() - offsetHeight) / 3;
						            editImageDialogBox.setPopupPosition(left, top);
						          }
						        });

							SaveImageRequest saveImageRequest = new SaveImageRequest();
							saveImageRequest.setImageFolderPath(imageFolder);
							saveImageRequest.setOriginalFilePath(originalFilePath);
							saveImageRequest.setTitleId(titleId);
							saveImageRequest.setAssetId(assetId);

							editImageDialogBox.setSaveImageRequest(saveImageRequest);
							
					        loadUrl(imageUrl, address, organization, company, licenseString1, licenseString2);
						}
					});

					previewButton.addClickHandler(new ClickHandler() {
						public void onClick(ClickEvent event) {
							
							final PreviewImageDialogBox previewImageDialogBox = new PreviewImageDialogBox(false, true);
							previewImageDialogBox.setPopupPositionAndShow(new PopupPanel.PositionCallback() {
						          public void setPosition(int offsetWidth, int offsetHeight) {
						            int left = (Window.getClientWidth() - offsetWidth) / 3;
						            int top = (Window.getClientHeight() - offsetHeight) / 3;
						            previewImageDialogBox.setPopupPosition(left, top);
						          }
						        });

					        previewImage(imageUrl, address, organization, company, licenseString1, licenseString2);
						}
					});
					
				}

			}

		} catch (MetadataTypeMismatchException ex) {
			String typeName = getTypeName(ex.getFieldDefinition());
			String receivedTypeName = getTypeName(ex.getReceivedField());
			String message = messages.metadataTypeMismatchError(typeName, receivedTypeName);
			LabelWidget label = new LabelWidget(message);
			label.addStyleName(TitleViewResources.STYLE_DATATYPE_MISMATCH_LABEL);
			return label;
		}

		/* For certain widgets, need to listen for key up events */
		DataTypeWidget<?> widget = fieldWidget.getDataTypeWidget();
		if (file.equals(newFile) && widget instanceof HasKeyUpHandlers) {
			KeyUpHandler handler = new WidgetKeyEventHandler(this, file, field, fieldWidget);
			HasKeyUpHandlers keyAwareWidget = (HasKeyUpHandlers) widget;
			newFileHandlers.add(keyAwareWidget.addKeyUpHandler(handler));
		}

		return panel;
	}

	/*
	 * Add video preview button for video file
	 */
	private void addVideoPreviewButton(Panel panel, UIAssetFile file, UIAssetFileField uriField, UIAssetFileField encodeFormatField) {
		Button videoPreviewButton = new Button(messages.previewImageButtontext());
		videoPreviewButton.addStyleDependentName(StyleNames.ACT_TOWARDS_SAVE_BUTTON_STYLE);
		panel.add(videoPreviewButton);

		Object encodeFormatFieldValue = null;
		if (encodeFormatField != null && encodeFormatField.getField() != null)
			encodeFormatFieldValue = encodeFormatField.getField().getValue();
		
		Object fieldValue = uriField.getField().getValue();
		if ((fieldValue == null) || (String.valueOf(fieldValue).isEmpty())) {
			videoPreviewButton.setEnabled(false);
		}
		else if ((encodeFormatFieldValue == null) || (String.valueOf(encodeFormatFieldValue).isEmpty())) {
			videoPreviewButton.setEnabled(false);			
		}
		else {
			final String fileURI = String.valueOf(fieldValue);
			final String fileEncodeFormat = String.valueOf(encodeFormatFieldValue);
				
			if (licensedForVideoPreview == false || JWPlayerWidget.isPlayable(fileEncodeFormat) == false) {
				videoPreviewButton.setEnabled(false);
				return;
			}
			final BusyIndicator busyIndicator = new BusyIndicator(this);
			final FileView parentView = this;
			videoPreviewButton.addClickHandler(new ClickHandler() {
				public void onClick(ClickEvent event) {
					busyIndicator.center();

					videoPlayerService.getSettings(new NeptuneAsyncCallback<UIVideoPlayer>() {
						/* Success getting settings */
						public void onNeptuneSuccess(UIVideoPlayer confSettings) {
							busyIndicator.hide();
							Map<String,String> basicVars = confSettings.getBasicVars();
							Map<String,String> flashVars = confSettings.getFlashVars();
							Map<String,String> paramVars = confSettings.getParamVars();

							//todo: Validate flash variable "streamer" before bringing-up dialog
							VideoPreviewDialog dlg = new VideoPreviewDialog(fileEncodeFormat.trim() + ":" + fileURI, basicVars, flashVars,paramVars);
							dlg.center();
						};

						/* Failure getting conf settings from backend */
						public void onNeptuneFailure(Throwable caught) {
							busyIndicator.hide();
							parentView.handleGetVideoSettingsFailure(caught);
						};
					});			
				}
				
			});
			
		}	
	}

		
	/*
	 * Handle error from back-end service getting settings for video player
	 */
	void handleGetVideoSettingsFailure(Throwable caught) 
	{
		//titleView.setFeedbackMessage("Failed to get settings from backend for video player");
	}
	
	/*
	 * Handle error from back-end service getting settings for video player
	 */
	void handleGetImageEditorSettingsFailure(Throwable caught) 
	{
		//titleView.setFeedbackMessage("Failed to get settings from backend for image editor");
	}
	
	/*
	 * Get the data type name to display given the field definition
	 */
	private String getTypeName(UIFieldDefinition fieldDefinition) {
		String name = null;
		if (fieldDefinition instanceof UIComplexFieldDefinition) {
			name = messages.complexTypeName();
		} else if (fieldDefinition instanceof UISimpleFieldDefinition) {
			name = ((UISimpleFieldDefinition) fieldDefinition).getDatatype().getName();
		} else {
			name = messages.unknownTypeName();
		}

		return name;
	}

	/*
	 * Get the data type name to display given the field definition
	 */
	private String getTypeName(UIField field) {
		String name = null;
		if (field instanceof UIComplexField) {
			name = messages.complexTypeName();
		} else if (field instanceof UISimpleField) {
			name = ((UISimpleField<?>) field).getDatatype().getName();
		} else {
			name = messages.unknownTypeName();
		}

		return name;
	}

	/*
	 * Handle an update in the metadata widget value
	 */
	void handleMetadataValueChanged(UIAssetFile file, UIAssetFileField field,
	        SimpleFieldWidget<?, ?, ?> widget) {
		handleFileUpdated(file);
	}

	/*
	 * Handle the key press event for the widgets in the 'new file' row
	 */
	private void handleKeyUpEvent(UIAssetFile file, UIAssetFileField field,
	        SimpleFieldWidget<?, ?, ?> widget, KeyUpEvent event) {
		/* Handle event only for the new file */
		if (!file.equals(newFile))
			return;

		String stringValue = widget.getTextValue();
		if (stringValue != null && stringValue.length() > 0) {
			handleFileUpdated(file);
		}
	}

	/*
	 * Event Handler class for handling a key press event for data type widget in the file view
	 */
	private static final class WidgetKeyEventHandler implements KeyUpHandler {

		private FileView fileView;
		private UIAssetFile file;
		private UIAssetFileField field;
		private SimpleFieldWidget<?, ?, ?> widget;

		/**
		 * Constructor
		 */
		public WidgetKeyEventHandler(FileView view, UIAssetFile file, UIAssetFileField field,
		        SimpleFieldWidget<?, ?, ?> widget) {
			this.fileView = view;
			this.file = file;
			this.field = field;
			this.widget = widget;
		}

		@Override
		public void onKeyUp(KeyUpEvent event) {
			fileView.handleKeyUpEvent(file, field, widget, event);
		}
	}

	// ========================================================================
	// ===================== METADATA VALIDATION
	// ========================================================================

	/**
	 * Validate the File View widgets, and return validation messages for each of the metadata
	 * widgets that fail validation.
	 * 
	 * @return the list of validation messages
	 */
	public void validate(List<AssetValidationMessage> messages,boolean draft) {
		int startIndex = columnIndices.get(FILE_ID_COLUMN) + 1;
		int endIndex = columnIndices.get(PARENT_FILE_ID_COLUMN) - 1;
		SimpleFieldWidget<?, ?, ?> widget = null;
		for (int row = 0; row < table.getRowCount(); row++) {
			for (int column = startIndex; column <= endIndex; column++) {
				Widget tableCellWidget = table.getCellWidget(row, column);
				if (tableCellWidget instanceof SimpleFieldWidget<?, ?, ?>) {
					widget = (SimpleFieldWidget<?, ?, ?>) table.getCellWidget(row, column);
					if (!widget.isValidValue(draft)) {
						int index = column - startIndex;
						String name = fileDefinition.getFieldDefinitions().get(index).getDisplayName();
						String error = widget.getCurrentToolTip();
						messages.add(new AssetValidationMessage(input, name, widget, error));
					}
				}
			}
		}
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	@Override
	public HandlerRegistration addViewChangeHandler(ViewChangeHandler handler) {
		return addHandler(handler, ViewChangeEvent.getType());
	}

	/*
	 * Fire a view change event indicating that the view is modified
	 */
	private void fireViewChangeEvent() { 
		fireEvent(new ViewChangeEvent());
	}

	// =============================================================================
	// ===================== JSNI ==================================================
	// =============================================================================
	
	public static native String getAppletCode() /*-{
		var NetImager;
		NetImager = new $wnd.netimager("NetImager");
		NetImager.width = 800;
		NetImager.height = 600;
		
		return NetImager.getAppletCode();
	
	}-*/;
	
	public static native void loadUrl(String imageUrl, String address, String organization, String company, String licenseString1, String licenseString2) /*-{
		$wnd.setData(imageUrl, address, organization, company, licenseString1, licenseString2);
		$wnd.editImage();
		
	}-*/;
	
	public static native void previewImage(String imageUrl, String address, String organization, String company, String licenseString1, String licenseString2) /*-{
		$wnd.setData(imageUrl, address, organization, company, licenseString1, licenseString2);
		$wnd.previewImage();
		
	}-*/;

}
