/*
 * Created on Apr 26, 2010
 * 
 * (C) Copyright TANDBERG Television Inc.
 */

package com.tandbergtv.cms.portal.content.client.title.view.compare.table;

import java.util.List;

import com.google.gwt.user.client.ui.Composite;
import com.tandbergtv.cms.portal.content.client.title.model.compare.UITitleCompareData;
import com.tandbergtv.cms.portal.content.client.title.view.compare.TitleCompareViewInput;
import com.tandbergtv.cms.portal.content.client.title.view.compare.TitleCompareViewMessages;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Table;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;

/**
 * The Title Comparison Table
 * 
 * @author Vijay Silva
 */
public abstract class TitleCompareTable<K, R extends TitleCompareRecord<K>> extends Composite {

	/* Properties */
	private final SimpleContainer mainContainer;
	private TitleCompareDataProvider<K, R> dataProvider;
	private Table<K, R> tableWidget;
	private TitleCompareViewInput input;
	private LabelWidget blankLabel;
	private final TitleCompareViewMessages messages;
	private boolean sortingEnabled = true;

	/* Style Names */
	private static final String STYLE_NAME = "content-TitleCompareTable";
	private static final String STYLE_TABLE = "content-TitleCompareTable-table";
	private static final String STYLE_BLANK_LABEL = "content-TitleCompareTable-blankLabel";
	private static final String STYLE_ERROR_LABEL = "content-TitleCompareTable-errorLabel";

	/**
	 * Constructor
	 * 
	 * @param messages The localization messages
	 */
	public TitleCompareTable(TitleCompareViewMessages messages) {
		this.messages = messages;
		mainContainer = new SimpleContainer();
		initWidget(mainContainer);
		this.initialize();
	}

	/**
	 * @return the input the widget input
	 */
	public TitleCompareViewInput getInput() {
		return input;
	}

	/**
	 * Set the widget input and refresh the view for this widget
	 * 
	 * @param input The widget input
	 */
	public void setInput(TitleCompareViewInput input) {
		this.input = input;
	}

	/**
	 * @return the messages
	 */
	public TitleCompareViewMessages getMessages() {
		return messages;
	}

	/**
	 * The sorting enabled flag, defaults to true
	 * 
	 * @return the sorting enabled flag
	 */
	public boolean isSortingEnabled() {
		return sortingEnabled;
	}

	/**
	 * Set the sorting enabled flag to enable table column sorting. This must be set before calling
	 * refresh or setting the input on the table
	 * 
	 * @param sortingEnabled the sorting enabled flag
	 */
	public void setSortingEnabled(boolean sortingEnabled) {
		this.sortingEnabled = sortingEnabled;
	}

	/**
	 * Get the table columns
	 * 
	 * @return The table columns
	 */
	public List<Column<?, R>> getTableColumns() {
		return getDataProvider().getColumns();
	}

	// ========================================================================
	// ===================== WIDGET REFRESH
	// ========================================================================

	/**
	 * Rebuilds the view for this widget using the current user input
	 */
	public void refresh() 
	{
		getContainer().setWidget(blankLabel);

		/* Check if the input is present */
		if (getInput() == null) {
			return;
		}

		/* Build the data provider and table */
		this.dataProvider = buildDataProvider();
		this.tableWidget = buildTable();

		/* Initialize the table */
		getTable().initialize(new NeptuneAsyncCallback<Void>() {
			@Override
			public void onNeptuneSuccess(Void result) {
				handleTableInitializationSuccess();
			}

			@Override
			public void onNeptuneFailure(Throwable caught) {
				handleTableInitializationFailure(caught);
			}
		});
	}

	// ========================================================================
	// ===================== INITIALIZATION
	// ========================================================================

	/*
	 * Initialize the widget
	 */
	private void initialize() {
		/* Update style */
		this.setStylePrimaryName(STYLE_NAME);

		/* Create widgets */
		blankLabel = new LabelWidget();
		blankLabel.addStyleName(STYLE_BLANK_LABEL);
		getContainer().setWidget(blankLabel);
	}

	// ========================================================================
	// ===================== PROTECTED STATE ACCESS
	// ========================================================================

	/**
	 * Get the widget container
	 * 
	 * @return The widget container
	 */
	protected SimpleContainer getContainer() {
		return this.mainContainer;
	}

	/**
	 * @return the table widget
	 */
	protected Table<K, R> getTable() {
		return tableWidget;
	}

	/**
	 * @return the dataProvider
	 */
	protected TitleCompareDataProvider<K, R> getDataProvider() {
		return dataProvider;
	}

	// ========================================================================
	// ===================== TABLE MANAGEMENT
	// ========================================================================

	/**
	 * Build the Data Provider
	 */
	protected abstract TitleCompareDataProvider<K, R> buildDataProvider();

	/**
	 * Build the Table Widget. The data provider is already built and is accessible.
	 * 
	 * @return The table widget
	 */
	protected Table<K, R> buildTable() {
		Table<K, R> table = new Table<K, R>(getDataProvider());

		/* If sorting is enabled, add the sort feature */
		if (isSortingEnabled()) {
			table.addSortFeature(getDataProvider().getSortFeature());
		}

		table.addStyleName(STYLE_TABLE);
		return table;
	}

	/**
	 * Handle the table initialization success
	 */
	protected void handleTableInitializationSuccess() {
		/* Refresh the table after initialization and show the widget */
		getTable().refresh();
		getContainer().setWidget(getTable());
	}

	/**
	 * Handle the error from table initialization, should not happen
	 */
	protected void handleTableInitializationFailure(Throwable error) {
		/* This should never happen */
		String msg = error.getLocalizedMessage();
		LabelWidget label = new LabelWidget();
		label.addStyleName(STYLE_ERROR_LABEL);
		label.setText(messages.titleCompareTableInitializationError(msg));
		getContainer().setWidget(label);
	}

	/**
	 * Handle the records updated event
	 */
	protected void handleRecordsUpdated() {
	}
}
