/*
 * Created on Apr 24, 2010
 * 
 * (C) Copyright TANDBERG Television Inc.
 */

package com.tandbergtv.cms.portal.content.client.title.view.compare.table.asset;

import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.title.view.compare.table.TitleCompareTableColumn;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;

/**
 * The Table Column in the Asset Compare Table
 * 
 * @author Vijay Silva
 */
public class AssetCompareTableColumn extends TitleCompareTableColumn<AssetCompareRecord> implements
        Column<AssetCompareRecord, AssetCompareRecord> {

	/* Properties */
	private final AssetCompareTable table;

	/* Column Names */
	static final String ASSET_TYPE_COLUMN_NAME = "AssetType";
	static final String METADATA_COLUMN_PREFIX = "AssetMetadata-";
	static final String FILE_METADATA_COLUMN_PREFIX = "FileMetadata-";
	static final String FILE1ID_COLUMN_NAME = "File1ID";
	static final String FILE2ID_COLUMN_NAME = "File2ID";
	static final String DIFFERENCE_TYPE_COLUMN_NAME = "DifferenceType";

	/**
	 * Constructor. Do not use this constructor for metadata fields
	 * 
	 * @param table The table showing this column
	 * @param name The column name
	 * @param displayName The column display name
	 */
	public AssetCompareTableColumn(AssetCompareTable table, String name, String displayName) {
		super(name, displayName);

		this.table = table;
	}

	/**
	 * Constructor. Use this constructor for only metadata fields
	 * 
	 * @param table The table showing this column
	 * @param name The metadata field name
	 * @param displayName The column display name
	 * @param fileMetadata flag indicating if this metadata is for the asset or the file
	 */
	public AssetCompareTableColumn(AssetCompareTable table, String metadataName,
	        String displayName, boolean fileMetadata) {
		this(table, getColumnName(metadataName, fileMetadata), displayName);
	}

	/**
	 * @return the showInFieldDifference flag, defaults to true
	 */
	public boolean isShownInFieldDifference() {
		return (!DIFFERENCE_TYPE_COLUMN_NAME.equals(getName()));
	}

	/**
	 * Determine if this column is either an asset or a file metadata column
	 * 
	 * @return true if metadata column, false otherwise
	 */
	public boolean isMetadataColumn() {
		return (isAssetMetadataColumn() || isFileMetadataColumn());
	}

	/**
	 * Determine if this column is an asset metadata column
	 * 
	 * @return true if metadata column, false otherwise
	 */
	public boolean isAssetMetadataColumn() {
		return (this.getName().startsWith(METADATA_COLUMN_PREFIX));
	}

	/**
	 * Determine if this column is a file metadata column
	 * 
	 * @return true if metadata column, false otherwise
	 */
	public boolean isFileMetadataColumn() {
		return (this.getName().startsWith(FILE_METADATA_COLUMN_PREFIX));
	}

	/**
	 * Get the name of the metadata field
	 * 
	 * @return The metadata field name
	 */
	public String getMetadataFieldName() {
		if (isFileMetadataColumn()) {
			return this.getName().substring(FILE_METADATA_COLUMN_PREFIX.length());
		} else if (isAssetMetadataColumn()) {
			return this.getName().substring(METADATA_COLUMN_PREFIX.length());
		}

		return null;
	}

	/*
	 * Generate a view that contains a label
	 */
	@Override
	public Widget getWidget(AssetCompareRecord record) {
		LabelWidget label = new LabelWidget();

		/* The Asset Type column */
		if (this.getName().equals(ASSET_TYPE_COLUMN_NAME)) {
			label.setText(record.getAssetTypeName());
		}
		/* The File1 ID column */
		else if (this.getName().equals(FILE1ID_COLUMN_NAME)) {
			label.setText(record.getAsset1ID());
		}
		/* The File2 ID column */
		else if (this.getName().equals(FILE2ID_COLUMN_NAME)) {
			label.setText(record.getAsset2ID());
		}
		/* The Difference Type column */
		else if (this.getName().equals(DIFFERENCE_TYPE_COLUMN_NAME)) {
			label.setText(record.getDifferenceTypeName());
			label.addClickHandler(new ColumnClickHandler(this.table, record));
		}
		/* The asset metadata columns for file records */
		else if (isAssetMetadataColumn()) {
			String metadataName = getMetadataFieldName();
			label.setText(record.getAssetMetadataValue(metadataName));
		}
		/* All other metadata columns */
		else if (isFileMetadataColumn()) {
			String metadataName = getMetadataFieldName();
			label.setText(record.getFileMetadataValue(metadataName));
		}

		return label;
	}

	/*
	 * Build the column name given the metadata field name and type
	 */
	private static String getColumnName(String metadataName, boolean fileMetadata) {
		String name = fileMetadata ? FILE_METADATA_COLUMN_PREFIX : METADATA_COLUMN_PREFIX;
		return (name + metadataName);
	}

	/*
	 * Click handler
	 */
	private static final class ColumnClickHandler implements ClickHandler {

		private final AssetCompareTable table;
		private final AssetCompareRecord record;

		/**
		 * Constructor
		 * 
		 * @param record The record for this click handler
		 */
		public ColumnClickHandler(AssetCompareTable table, AssetCompareRecord record) {
			this.table = table;
			this.record = record;
		}

		/*
		 * Click handler
		 */
		@Override
		public void onClick(ClickEvent event) {
			table.handleDifferenceTypeColumnClicked(record);
		}
	}
}
