/*
 * Created on Apr 29, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.view.history;

import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.ResizeEvent;
import com.google.gwt.event.logical.shared.ResizeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.Window;
import com.tandbergtv.cms.portal.content.client.title.model.UIActionTitle;
import com.tandbergtv.cms.portal.content.client.title.model.compare.TitleCompareRequest;
import com.tandbergtv.cms.portal.content.client.title.model.compare.UITitleCompareData;
import com.tandbergtv.cms.portal.content.client.title.model.compare.UITitleCompareXmlData;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.UITitleMetadata;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.UITitleValidationMessage;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.asset.UIAsset;
import com.tandbergtv.cms.portal.content.client.title.service.ITitleViewService;
import com.tandbergtv.cms.portal.content.client.title.service.ITitleViewServiceAsync;
import com.tandbergtv.cms.portal.content.client.title.service.TitleValidationUIException;
import com.tandbergtv.cms.portal.content.client.title.view.HtmlorXmlDisplayHelper;
import com.tandbergtv.cms.portal.content.client.title.view.TitleView;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewMessages;
import com.tandbergtv.cms.portal.content.client.title.view.asset.AssetPanel;
import com.tandbergtv.cms.portal.content.client.title.view.asset.AssetTree.AssetInfo;
import com.tandbergtv.cms.portal.content.client.title.view.compare.ITitleCompareViewController;
import com.tandbergtv.cms.portal.content.client.title.view.compare.TitleCompareView;
import com.tandbergtv.cms.portal.content.client.title.view.compare.TitleCompareViewInput;
import com.tandbergtv.cms.portal.content.client.title.view.metadata.TitleValidationViewInput;
import com.tandbergtv.cms.portal.content.client.title.view.preview.TitlePreviewWidget;
import com.tandbergtv.cms.portal.content.client.title.view.preview.TitlePreviewWidgetInput;
import com.tandbergtv.cms.portal.content.client.title.view.series.SeriesTitleView;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIAssetDefinition;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIAssetSpecification;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIAssetSpecificationListItem;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.GridContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

/**
 * The Metadata History Panel displays the metadata for particular title revision.
 * 
 * @author Vijay Silva
 */
class MetadataHistoryPanel extends HistoryPanel implements ITitleCompareViewController 
{
	/* Properties */
	private GridContainer mainContainer;
	private LabelWidget blankPanelLabel;
	private SimpleContainer controlsContainer;
	private HorizontalContainer buttonContainer;
	private ButtonWidget saveButton, compareButton, previewButton, cancelButton, rollbackAsDraftButton;
	private AssetInfo assetInfo;
	private AssetPanel assetPanel;
	private String titleRevision;
	private HandlerRegistration windowRegistration = null;
	protected TitleView titleView;

	private TitleCompareView compareView;
	private TitleCompareRequest lastRequest;
	
	private static final String INVALID_LICENSE_DATE = "INVALID_LICENSE_DATE";
	private static final String STYLE_CONTROLS_CONTAINER = "content-MetadataHistoryPanel-controlsContainer";
	private static final String STYLE_REVISION_CELL = "content-MetadataHistoryPanel-revisionLabelCell";
	private static final String STYLE_ASSETPANEL_CELL = "content-MetadataHistoryPanel-assetPanelCell";
	private static final String STYLE_REVISION_LABEL = "content-MetadataHistoryPanel-revisionLabel";
	private static final String STYLE_ASSET_PANEL = "content-MetadataHistoryPanel-assetPanel";

	
	private ITitleViewServiceAsync viewService = GWT.create(ITitleViewService.class);
	
	/**
	 * Panel that displays the metadata for a revisioned title along with buttons to revert or
	 * cancel.
	 */
	public MetadataHistoryPanel(TitleHistoryTab parent) {
		super(parent);
	}

	/* Add widgets shown in this panel */
	protected void initialize() 
	{
		addStyleDependentName("metadata");

		/* Initialize the main container */
		mainContainer = new GridContainer(2, 1);

		/* Initialize the Buttons */
		controlsContainer = new SimpleContainer();
		controlsContainer.addStyleName(STYLE_CONTROLS_CONTAINER);
		buttonContainer = new HorizontalContainer();
		controlsContainer.setWidget(buttonContainer);

		/* Save button */
		String rollbackText = this.getTab().getViewMessages().rollbackButton();
		saveButton = new ButtonWidget(rollbackText);
		saveButton.addStyleDependentName(StyleNames.COMMIT_BUTTON_STYLE);
		saveButton.addClickHandler(new ClickHandler() {
			public void onClick(ClickEvent event) {
				rollbackClickedAction();
			}
		});
		saveButton.setVisible(false);
		saveButton.setEnabled(false);
		buttonContainer.add(saveButton);

		/* Save as Draft button */
		String rollbackAsDraftText = this.getTab().getViewMessages().rollbackAsDraftButton();
		rollbackAsDraftButton = new ButtonWidget(rollbackAsDraftText);
		rollbackAsDraftButton.addStyleDependentName(StyleNames.COMMIT_BUTTON_STYLE);
		rollbackAsDraftButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				// save title as DRAFT, go back to the history tab with a success message
				Long titleId = getTab().getViewInput().getTitleId();
				getTab().getViewService().rollbackTitleAsDraft(titleId, titleRevision,
						new NeptuneAsyncCallback<Void>() {
							public void onNeptuneSuccess(Void result) {
								// successfully rolled back to this version with DRAFT status
								handleRollbackTitleSuccess();
							}

							public void onNeptuneFailure(Throwable caught) {
								// Couldn't roll back and save it as DRAFT. Nothing much we can do.
								String msg = getTab().getViewMessages().rollbackFailureMessage(
										titleRevision);
								getTab().showProgressHistory(false, msg, true);
								clearPanel();
							}
						});
			}
		});
		rollbackAsDraftButton.setVisible(false);
		rollbackAsDraftButton.setEnabled(false);
		buttonContainer.add(rollbackAsDraftButton);

		/* Compare button */
		String compareText = this.getTab().getViewMessages().compareButton();
		compareButton = new ButtonWidget(compareText);
		compareButton.addStyleDependentName(StyleNames.ACT_TOWARDS_SAVE_BUTTON_STYLE);
		compareButton.addClickHandler(new ClickHandler() {
			public void onClick(ClickEvent event) {
				compareClickedAction();
			}
		});
		buttonContainer.add(compareButton);

		/* Preview button */
		String previewText = this.getTab().getViewMessages().previewButton();
		previewButton = new ButtonWidget(previewText);
		previewButton.addStyleDependentName(StyleNames.ACT_TOWARDS_SAVE_BUTTON_STYLE);
		previewButton.addClickHandler(new ClickHandler() {
			public void onClick(ClickEvent event) {
				previewClickedAction();
			}
		});
		buttonContainer.add(previewButton);
	
		/* Cancel button */
		String cancelText = this.getTab().getViewMessages().cancelButton();
		cancelButton = new ButtonWidget(cancelText);
		cancelButton.addStyleDependentName(StyleNames.DATALOSS_BUTTON_STYLE);
		cancelButton.addClickHandler(new ClickHandler() {
			public void onClick(ClickEvent event) {
				cancelClickedAction();
			}
		});
		buttonContainer.add(cancelButton);

		/* Add buttons to container */
		mainContainer.setWidget(1, 0, controlsContainer);

		blankPanelLabel = new LabelWidget("");
		this.setWidget(blankPanelLabel);
	}

	// ==============================================================
	// ===================== TITLE ROLLBACK
	// ==============================================================

	/*
	 * Handle the roll back click
	 */
	private void rollbackClickedAction() {
		Long titleId = this.getTab().getViewInput().getTitleId();
		this.getTab().getViewService()
				.rollbackTitle(titleId, titleRevision, new NeptuneAsyncCallback<Void>() {

					@Override
					public void onNeptuneSuccess(Void result) {
						handleRollbackTitleSuccess();
					}

					@Override
					public void onNeptuneFailure(Throwable caught) {
						handleRollbackTitleFailure(caught);
					}
				});
	}

	/* The title roll back was successful */
	private void handleRollbackTitleSuccess() {
		String message = this.getTab().getViewMessages().successfullRollbackMessage(titleRevision);
		this.getTab().showProgressHistory(true, message, true);
		clearPanel();
	}

	/* The title roll back failed */
	private void handleRollbackTitleFailure(Throwable caught) 
	{
		TitleViewMessages messages = getTab().getViewMessages();

		// Go to history view with error message for server errors other than validation
		if (!(caught instanceof TitleValidationUIException)) {
			String message = messages.rollbackFailureMessage(titleRevision);
			getTab().showProgressHistory(false, message, true);
			clearPanel();
			return;
		}

		if (!canSaveAsDraft((TitleValidationUIException) caught)) {
			String message = messages.rollbackValidationFailureMessage(titleRevision);
			getTab().showProgressHistory(false, message, true);
			clearPanel();
			return;
		}

		TitleValidationViewInput validationViewInput = new TitleValidationViewInput();  
		validationViewInput.getTitleValidationMessages().clear();
		validationViewInput.setErrorHeader(messages.rollbackSaveAsDraftMessage(titleRevision));
		if(((TitleValidationUIException) caught).getValidationMessages().size() > 0) 
		{
			validationViewInput.getTitleValidationMessages().addAll(((TitleValidationUIException) caught).getValidationMessages());
		} 
		else 
		{
			validationViewInput.setErrorHeader(this.getTab().getViewMessages().rollbackFailureMessage(titleRevision));
		}
		titleView.showMessage(validationViewInput);

		rollbackAsDraftButton.setVisible(true);
		rollbackAsDraftButton.setEnabled(true);
	}

	// ==============================================================
	// ===================== COMPARE ACTION
	// ==============================================================

	/*
	 * Handle the compare click
	 */
	private void compareClickedAction() 
	{
		compareView = new TitleCompareView(this, !isSeriesTitle());
		
		/* Build the titles to compare */
		UIActionTitle title1 = new UIActionTitle();
		title1.setTitleId(getTab().getViewInput().getTitleId());

		UIActionTitle title2 = new UIActionTitle();
		title2.setTitleId(getTab().getViewInput().getTitleId());
		title2.setVersion(titleRevision);

		// Input
		TitleCompareViewInput input = new TitleCompareViewInput();
		input.specName = getTab().getViewInput().getSpecificationName();
		
		String header = getTab().getViewMessages().titleCompareSummaryMessage(
				getTab().getViewMessages().compareTitleHeading(), 
				getTab().getViewMessages().compareRevisionHeading(titleRevision));
		
		input.header = header;
		input.displayName1 = getTab().getViewMessages().compareCurrentRevisionName();
		input.displayName2 = getTab().getViewMessages().compareRevisionName(titleRevision);
		input.restrictedHeight = getTab().getUsedHeight();
		input.restrictedWidth = getTab().getUsedWidth();
		compareView.setInput(input);

		lastRequest = new TitleCompareRequest();
		lastRequest.title1 = title1;
		lastRequest.title2 = title2;
		lastRequest.specName = input.specName;
		
		if(titleView instanceof SeriesTitleView)
			lastRequest.specificationType = "SERIES";
		else
			lastRequest.specificationType = "PACKAGE";

		compareDiff();
		setWidget(compareView);
	}

	/*
	 * Handle the title compare view cancel action
	 */
	private void handleViewCancelled() {
		setWidget(this.mainContainer);
		updateScrollContainerSize();
	}

	// ==============================================================
	// ===================== PREVIEW ACTION
	// ==============================================================

	/*
	 * Handle the preview click
	 */
	private void previewClickedAction() {
		/* Build the title to preview */
		UIActionTitle title = new UIActionTitle();
		title.setTitleId(getTab().getViewInput().getTitleId());
		title.setVersion(titleRevision);
		TitlePreviewWidgetInput input = new TitlePreviewWidgetInput(title);
		input.setSelectedSpecification(getTab().getViewInput().getSpecificationName());
		input.setRestrictedHeight(getTab().getUsedHeight());
		input.setRestrictedWidth(getTab().getUsedWidth());

		/* Build view and show */
		TitlePreviewWidget previewWidget = new TitlePreviewWidget(getTab().getViewMessages());
		previewWidget.addViewCancelHandler(new ViewCancelHandler() {
			@Override
			public void onCancel(ViewCancelEvent event) {
				handleViewCancelled();
			}
		});
		previewWidget.setInput(input);
		previewWidget.refresh();
		setWidget(previewWidget);
	}

	// ==============================================================
	// ===================== CANCEL ACTION
	// ==============================================================

	/* Handle the cancel click */
	private void cancelClickedAction() {
		this.getTab().showProgressHistory(false, null, false);
		clearPanel();
	}

	// ==============================================================
	// ===================== REFRESH
	// ==============================================================

	/**
	 * Does not do anything.
	 * 
	 * @see com.tandbergtv.cms.portal.content.client.title.view.history.HistoryPanel#refresh()
	 */
	@Override
	public void refresh() {
		clearPanel();
	}

	/* Remove the asset maintained and clear the display on this panel */
	private void clearPanel() 
	{
		/* Clear the input */
		this.titleRevision = null;
		this.assetInfo = null;
		this.assetPanel = null;

		/* Clear the widgets */
		this.setWidget(blankPanelLabel);
	}

	// ==============================================================
	// ===================== LOADING INPUT
	// ==============================================================

	/**
	 * Set the input to display on this panel
	 * 
	 * @param revision The title revision
	 * @param asset The root asset
	 */
	public void setInput(String revision, UIAssetSpecification specification,
			UITitleMetadata metadata, TitleView tView) {
		/* Store the revision */
		this.titleRevision = revision;

		this.titleView = tView;

		/* Build the 'content' section which shows the metadata */
		GridContainer contentPanel = new GridContainer(2, 1);
		contentPanel.setWidth("100%");

		rollbackAsDraftButton.setVisible(false);
		rollbackAsDraftButton.setEnabled(false);

		/* Add a message to the top indicating title version number */
		String revisionMessage = this.getTab().getViewMessages().titleRevisionMessage(revision);
		LabelWidget revisionLabel = new LabelWidget(revisionMessage);
		revisionLabel.addStyleName(STYLE_REVISION_LABEL);
		contentPanel.setWidget(0, 0, revisionLabel);
		contentPanel.getCellFormatter().setStyleName(0, 0, STYLE_REVISION_CELL);

		/* Build the asset info and asset panel */
		UIAsset asset = metadata.getRootAsset();
		UIAssetDefinition definition = specification.getAssetDefinition(asset.getAssetType());
		this.assetInfo = new AssetInfo(this.getTab().getViewInput(), asset, definition);
		this.assetPanel = new AssetPanel(tView, assetInfo, null, true);
		assetPanel.addStyleName(STYLE_ASSET_PANEL);
		contentPanel.setWidget(1, 0, assetPanel);
		contentPanel.getCellFormatter().setStyleName(1, 0, STYLE_ASSETPANEL_CELL);

		assetPanel.addOffsettingWidget(titleView.getMessageViewWidget());
		assetPanel.addOffsettingWidget(revisionLabel);
		assetPanel.addOffsettingWidget(controlsContainer);
		
		/* Update the contents of the scroll container */
		mainContainer.setWidget(0, 0, contentPanel);

		/* Ensure that the main container is displayed and resize scroll container */
		this.setWidget(mainContainer);
		updateScrollContainerSize();

		/* Update the buttons */
		boolean enabled = metadata.isActive() && getTab().getViewInput().hasTitleModifyPermission();
		saveButton.setVisible(enabled);
		saveButton.setEnabled(enabled);
		
		if(isSeriesTitle())
		{
			previewButton.setVisible(false);
		}
	}

	/**
	 * If there are any validation errors due to title license end date not matching the criteria
	 * for its lists, the title cannot be saved as draft.
	 * 
	 * @return
	 */
	private boolean canSaveAsDraft(TitleValidationUIException exception) {
		for (UITitleValidationMessage message : exception.getValidationMessages()) {
			if (message.getErrorCode().equalsIgnoreCase(INVALID_LICENSE_DATE)) {
				return false;
			}
		}
		return true;
	}

	// ========================================================================
	// ===================== WINDOW RESIZE
	// ========================================================================

	@Override
	protected void onLoad() 
	{
		super.onLoad();

		windowRegistration = Window.addResizeHandler(new ResizeHandler() 
		{
			public void onResize(ResizeEvent event) 
			{
				updateScrollContainerSize();
			}
		});

		updateScrollContainerSize();
	};

	@Override
	protected void onUnload() 
	{
		windowRegistration.removeHandler();
		windowRegistration = null;

		super.onUnload();
	}

	/*
	 * Update the size of the asset container widget
	 */
	private void updateScrollContainerSize() 
	{
		if(assetPanel != null)
		{
			assetPanel.updateSize();
		}
	}

	
	@Override
	public void onCancel() 
	{
		handleViewCancelled();
	}

	
	@Override
	public void onShowDifferences() 
	{
		compareDiff();
	}

	
	@Override
	public void onShowXml() 
	{
		compareXml();		
	}

	@Override
	public void onSpecChanged(String newSpec) 
	{
		lastRequest.specName = newSpec;
		compareView.getInput().specName = newSpec;
		compareXml();
	}

	
	public void compareXml()
	{
		compareView.showBusyIndicator();
		
		viewService.compareTitlePreviews(lastRequest.title1, lastRequest.title2, 
				lastRequest.specName, HtmlorXmlDisplayHelper.useHtml(), new NeptuneAsyncCallback<UITitleCompareXmlData>() 
		{
			@Override
			public void onNeptuneFailure(Throwable caught) 
			{
				compareView.showErrorView(caught);
				compareView.hideBusyIndicator();
			}

			@Override
			public void onNeptuneSuccess(UITitleCompareXmlData result) 
			{
				compareView.showXmlView(result);
				compareView.hideBusyIndicator();
			}
		} 
		);
	}


	public void compareDiff()
	{
		compareView.showBusyIndicator();
	
		viewService.compareTitles(lastRequest, new NeptuneAsyncCallback<UITitleCompareData>() 
		{
			@Override
			public void onNeptuneFailure(Throwable caught) 
			{
				compareView.showErrorView(caught);
				compareView.hideBusyIndicator();
			}

			@Override
			public void onNeptuneSuccess(UITitleCompareData data) 
			{
				setSpecName(data);
				compareView.getInput().compareData = data;
				compareView.showDiffView();
				compareView.hideBusyIndicator();
			}
			
		});
	}

	
	private void setSpecName(UITitleCompareData data)
	{
		// Set default spec name if it is null
		if(lastRequest != null && data != null)
		{
			if(lastRequest.specName == null)
			{
				List<UIAssetSpecificationListItem> specs = data.getSpecifications(); 
				if(specs != null && specs.size() > 0) lastRequest.specName = specs.get(0).getName();
			}
		}
	}
	
	private boolean isSeriesTitle(){
		return titleView instanceof SeriesTitleView;
	}
	
}
