/*
 * Created on Apr 29, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.view.history;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import com.google.gwt.core.client.GWT;
import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.ResizeEvent;
import com.google.gwt.event.logical.shared.ResizeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.title.model.UIActionTitle;
import com.tandbergtv.cms.portal.content.client.title.model.UITitle;
import com.tandbergtv.cms.portal.content.client.title.model.compare.TitleCompareRequest;
import com.tandbergtv.cms.portal.content.client.title.model.compare.UITitleCompareData;
import com.tandbergtv.cms.portal.content.client.title.model.compare.UITitleCompareXmlData;
import com.tandbergtv.cms.portal.content.client.title.service.ITitleViewService;
import com.tandbergtv.cms.portal.content.client.title.service.ITitleViewServiceAsync;
import com.tandbergtv.cms.portal.content.client.title.service.ReadDeletedTitleException;
import com.tandbergtv.cms.portal.content.client.title.view.HtmlorXmlDisplayHelper;
import com.tandbergtv.cms.portal.content.client.title.view.compare.ITitleCompareViewController;
import com.tandbergtv.cms.portal.content.client.title.view.compare.TitleCompareView;
import com.tandbergtv.cms.portal.content.client.title.view.compare.TitleCompareViewInput;
import com.tandbergtv.cms.portal.content.client.title.view.metadata.TitleValidationViewInput;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIAssetSpecificationListItem;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Table;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Table.RefreshListener;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.ScrollContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

/**
 * The Progress History panel displays the progress item history
 * 
 * @author Vijay Silva
 */
class ProgressHistoryPanel extends HistoryPanel  implements ITitleCompareViewController
{
	private List<Widget> offsettingWidgets = new ArrayList<Widget>();

	private ProgressItemDataProvider dataProvider;
	private Table<Long, ProgressItemRecord> table;
	private VerticalContainer mainContainer;
	private HorizontalContainer buttonContainer;
	private ScrollContainer scrollContainer;
	private ButtonWidget compareButton;
	private TitleCompareView compareView;
	private TitleCompareRequest lastRequest;
	private HandlerRegistration windowRegistration = null;

	private ITitleViewServiceAsync viewService = GWT.create(ITitleViewService.class);
	
	/* Styles */
	private static final String STYLE_LOADING_LABEL = "content-ProgressHistoryPanel-loadingMessage";
	private static final String STYLE_PROGRESS_TABLE = "content-ProgressHistoryPanel-progressTable";
	private static final String STYLE_PROGRESS = "progress";
	private static final String STYLE_HISTORY_PANEL = "content-HistoryPanel";

	private boolean isSeriesTitle = false;
	
	/**
	 * Constructor
	 */
	public ProgressHistoryPanel(TitleHistoryTab parent) {
		super(parent);
	}
	
	/**
	 * Initialize the widgets that are displayed
	 * 
	 * @see com.tandbergtv.cms.portal.content.client.title.view.history.HistoryPanel#intialize()
	 */
	protected void initialize() 
	{
		dataProvider = new ProgressItemDataProvider(this);
		table = new Table<Long, ProgressItemRecord>(dataProvider);
		table.addRefreshListener(new RefreshListener() 
		{
			@Override
			public void updated() 
			{
				resizeScrollContainer();
			}
		});
		
		table.addStyleName(STYLE_PROGRESS_TABLE);

		scrollContainer = new ScrollContainer();
		
		mainContainer = new VerticalContainer();
		mainContainer.setStylePrimaryName(STYLE_HISTORY_PANEL);
		mainContainer.addStyleDependentName(STYLE_PROGRESS);
		buttonContainer = new HorizontalContainer();

		String compareText = getTab().getViewMessages().compareButton();
		compareButton = new ButtonWidget(compareText);
		compareButton.addStyleDependentName(StyleNames.COMMIT_BUTTON_STYLE);
		compareButton.addClickHandler(new ClickHandler() 
		{
			public void onClick(ClickEvent event) 
			{
				compareClickedAction();
			}
		});
		
		buttonContainer.add(compareButton);
		scrollContainer.add(table);
		
		mainContainer.add(scrollContainer);
		mainContainer.add(buttonContainer);
		
		// Show the loading message
		LabelWidget blankLabel = new LabelWidget("");
		this.setWidget(blankLabel);
	}


	public boolean isSeriesTitle() 
	{
		return isSeriesTitle;
	}

	public void setSeriesTitle(boolean isSeriesTitle) 
	{
		this.isSeriesTitle = isSeriesTitle;
	}

	
	public void addOffsettingWidget(Widget widget)
	{
		offsettingWidgets.add(widget);
	}
	
	
	private void showLoadingMessage() {
		/* Use a message widget while loading */
		String message = this.getTab().getViewMessages().historyLoading();
		LabelWidget label = new LabelWidget(message);
		label.addStyleName(STYLE_LOADING_LABEL);
		this.setWidget(label);
	}

	/**
	 * Refresh the table view being shown
	 * 
	 * @see com.tandbergtv.cms.portal.content.client.title.view.history.HistoryPanel#refresh()
	 */
	public void refresh() 
	{
		this.showLoadingMessage();

		if(!table.isInitialized()) 
		{
			table.initialize(new NeptuneAsyncCallback<Void>() 
			{
				@Override
				public void onNeptuneFailure(Throwable caught) 
				{
				}

				@Override
				public void onNeptuneSuccess(Void result) 
				{
					table.refresh();
				}
			});
		} 
		else 
		{
			table.refresh();
		}
		
	}

	// ========================================================================
	// ===================== WINDOW RESIZE
	// ========================================================================

	@Override
	protected void onLoad() 
	{
		super.onLoad();

		// Use message view to recalculate scroll view height
		offsettingWidgets.add(getTab().getTitleView().getMessageViewWidget());
		
		windowRegistration = Window.addResizeHandler(new ResizeHandler() 
		{
			public void onResize(ResizeEvent event) 
			{
				resizeScrollContainer();
			}
		});

		resizeScrollContainer();
	};

	@Override
	protected void onUnload() 
	{
		windowRegistration.removeHandler();
		windowRegistration = null;

		super.onUnload();
	}

	
	private void resizeScrollContainer() 
	{
		/* Check if attached to DOM */
		if(scrollContainer == null) return;

		Scheduler.get().scheduleDeferred(new ScheduledCommand() 
		{
			@Override
			public void execute() 
			{
				int height = Window.getClientHeight() - 280;
				
				for(Widget w : offsettingWidgets)
				{
					height -= w.getOffsetHeight();
				}
				
				scrollContainer.setHeight(height + "px");
			}
		});
	}
	
	
	/* The history fetched with the records is maintained with the UITitle in the input */
	void handleGetRecordsSuccess(UITitle title, int recordCount) 
	{
		/* If there is no history for this title, hide the table */
		if (recordCount == 0) 
		{
			// Hide the table
			String message = getTab().getViewMessages().historyNotPresent();
			LabelWidget label = new LabelWidget(message);
			this.setWidget(label);
		} 
		else 
		{
			/* Ensure that the table widget is showing */
			this.setWidget(mainContainer);
		}

		this.getTab().updateTitleHeader(title.getOverview());

		this.getTab().hideBusyIndicator();
		this.getTab().markUpdated();
	}

	/* The failure to get the records causes the error panel to display */
	void handleGetRecordsFailure(Throwable caught) 
	{
		if(caught instanceof ReadDeletedTitleException) 
		{
			this.getTab().showSevereError(caught.getMessage());
		} 
		else 
		{
			TitleHistoryTab tab = this.getTab();
			String message = tab.getViewMessages().historyFetchFailure(caught.getMessage());
			tab.showErrorPanel(message);
		}

		this.getTab().hideBusyIndicator();
	}
	
	
	private void compareClickedAction() 
	{
		Set<String> selectedVersions = dataProvider.getselectedRevisions();
		if(selectedVersions.size() > 2) 
		{
			String message = this.getTab().getViewMessages().historyErrorMoreThanTwoRevisions();
			
			TitleValidationViewInput input = new TitleValidationViewInput();  
			input.setErrorHeader(message);
			getTab().getTitleView().showMessage(input);
			resizeScrollContainer();
			
			return;
		}	
		else if (selectedVersions.size() < 2) 
		{
			String message = this.getTab().getViewMessages().historyErrorLessThanTwoRevisions();

			TitleValidationViewInput input = new TitleValidationViewInput();  
			input.setErrorHeader(message);
			getTab().getTitleView().showMessage(input);
			resizeScrollContainer();

			return;
		}
				
		Iterator<String> itr = selectedVersions.iterator(); 
	    String revision1 = itr.next(); 
	    String revision2 = itr.next(); 
		
	    compare(revision1, revision2);
	}

	
	public void compare(String revision1, String revision2) 
	{	
		getTab().getTitleView().showMessage(null);
		resizeScrollContainer();

		compareView = new TitleCompareView(this,!isSeriesTitle);
		
		/* Build the titles to compare */
		UIActionTitle title1 = new UIActionTitle();
		title1.setTitleId(getTab().getViewInput().getTitleId());
		title1.setVersion(revision1);

		UIActionTitle title2 = new UIActionTitle();
		title2.setTitleId(getTab().getViewInput().getTitleId());
		title2.setVersion(revision2);

		// Input
		TitleCompareViewInput input = new TitleCompareViewInput();
		input.specName = getTab().getViewInput().getSpecificationName();
		
		String header = getTab().getViewMessages().titleCompareSummaryMessage(
				getTab().getViewMessages().compareRevisionHeading(revision1), 
				getTab().getViewMessages().compareRevisionHeading(revision2));
		
		input.header = header;
		input.displayName1 = getTab().getViewMessages().compareRevisionName(revision1); 
		input.displayName2 = getTab().getViewMessages().compareRevisionName(revision2);
		input.restrictedHeight = getTab().getUsedHeight();
		input.restrictedWidth = getTab().getUsedWidth();

		lastRequest = new TitleCompareRequest();
		lastRequest.title1 = title1;
		lastRequest.title2 = title2;
		lastRequest.specName = input.specName;
		if(isSeriesTitle){
			lastRequest.specificationType = "SERIES";
		}
		else
			lastRequest.specificationType = "PACKAGE";

		compareView.setInput(input);
		
		compareDiff();
		setWidget(compareView);
	}

	
	public void compareDiff()
	{
		compareView.showBusyIndicator();
		
		viewService.compareTitles(lastRequest, new NeptuneAsyncCallback<UITitleCompareData>() 
		{
			@Override
			public void onNeptuneFailure(Throwable caught) 
			{
				compareView.showErrorView(caught);
				compareView.hideBusyIndicator();
			}

			@Override
			public void onNeptuneSuccess(UITitleCompareData data) 
			{
				setSpecName(data);
				compareView.getInput().compareData = data;
				compareView.showDiffView();
				compareView.hideBusyIndicator();
			}
		});
	}
	
	public void compareXml()
	{
		compareView.showBusyIndicator();
		
		viewService.compareTitlePreviews(lastRequest.title1, lastRequest.title2, 
				lastRequest.specName, HtmlorXmlDisplayHelper.useHtml(), new NeptuneAsyncCallback<UITitleCompareXmlData>() 
		{
			@Override
			public void onNeptuneFailure(Throwable caught) 
			{
				compareView.showErrorView(caught);
				compareView.hideBusyIndicator();
			}

			@Override
			public void onNeptuneSuccess(UITitleCompareXmlData result) 
			{
				compareView.showXmlView(result);
				compareView.hideBusyIndicator();
			}
		} 
		);
	}
	
	private void setSpecName(UITitleCompareData data)
	{
		// Set default spec name if it is null
		if(lastRequest != null && data != null)
		{
			if(lastRequest.specName == null)
			{
				List<UIAssetSpecificationListItem> specs = data.getSpecifications(); 
				if(specs != null && specs.size() > 0) lastRequest.specName = specs.get(0).getName();
			}
		}
	}
	
	private void handleViewCancelled() {
		setWidget(this.mainContainer);
		//updateScrollContainerSize();
	}

	@Override
	public void onCancel() {
		handleViewCancelled();		
	}

	@Override
	public void onShowDifferences() {
		compareDiff();		
	}

	@Override
	public void onShowXml() {
		compareXml();				
	}

	@Override
	public void onSpecChanged(String newSpec) {
		lastRequest.specName = newSpec;
		compareView.getInput().specName = newSpec;
		compareXml();
	}
	
}
