/*
 * Created on Apr 28, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.view.history;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.title.model.UITitle;
import com.tandbergtv.cms.portal.content.client.title.model.history.UIProgressItem;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewMessages;
import com.tandbergtv.cms.portal.ui.title.client.model.uiservice.ITitleProgressService;
import com.tandbergtv.cms.portal.ui.title.client.model.uiservice.ITitleProgressService.Hyperlink;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.application.ServiceLoader;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.CheckBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.HyperlinkWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.InlineLabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.ColumnBase;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.View;

/**
 * Data Provider for the progress item table
 * 
 * @author Vijay Silva
 */
class ProgressItemDataProvider implements DataProvider<Long, ProgressItemRecord> {

	/* The columns for this provider */
	private TitleViewMessages messages;
	private List<Column<?, ProgressItemRecord>> columns;
	private ProgressHistoryPanel parent;
	private List<String> titleRevisions = new ArrayList<String>();
	private Set<String> selectedRevisions = new TreeSet<String>();
	private String firstRevision;
	private String normalizedEventName;
	private String updatedEventName;
	
	/* Constants */
	public static final String USER_COLUMN_NAME = "sourceComponentName";
	public static final String DATE_COLUMN_NAME = "timestamp";
	public static final String ACTION_COLUMN_NAME = "name";
	public static final String REVISION_COLUMN_NAME = "titleVersion";
	public static final String DESCRIPTION_COLUMN_NAME = "description";

	/**
	 * Constructor
	 */
	public ProgressItemDataProvider(ProgressHistoryPanel parent) {
		this.parent = parent;

		/* Initialize the internationalized messages */
		messages = GWT.create(TitleViewMessages.class);

		/* Initialize the columns */
		initializeColumns();
	}

	/* Initialize the columns */
	private void initializeColumns() {
		columns = new ArrayList<Column<?, ProgressItemRecord>>();
		String displayName = null;
		ProgressItemColumn column = null;

		/* select column */
		columns.add(prepareSelectColumn());
		
		/* User Column */
		columns.add(prepareUserColumn());

		/* Action Column */
		displayName = messages.progressTableActionColumn();
		column = new ProgressItemColumn(ACTION_COLUMN_NAME, displayName) {
			protected String getColumnText(ProgressItemRecord record) {
				return getActionValue(record.getProgressItem());
			}
		};
		column.setColumnWidth("15%");
		column.setCellStyle(TableConstants.STYLE_DATACELL_TEXT);
		columns.add(column);

		columns.add(prepareDescriptionColumn());

		/* Date Column */
		displayName = messages.progressTableDateColumn();
		column = new ProgressItemColumn(DATE_COLUMN_NAME, displayName) {
			protected String getColumnText(ProgressItemRecord record) {
				return getDateValue(record.getProgressItem());
			}
		};
		column.setColumnWidth("10%");
		column.setCellStyle(TableConstants.STYLE_DATACELL_DATE);
		columns.add(column);

		/* Revision Column */
		columns.add(prepareRevisionColumn());
	}

	private Column<String, ProgressItemRecord> prepareSelectColumn() {
		ColumnBase<String, ProgressItemRecord> column = null;

		String displayName = "";
		column = new ColumnBase<String, ProgressItemRecord>(REVISION_COLUMN_NAME, displayName) {
			@Override
			public View<String> getView(final ProgressItemRecord record) {
				final String revision = record.getProgressItem().getTitleRevision();
				final boolean isFirst = isFirstProgressTitleVersion(record);
				
				return new View<String>() {
					@Override
					public String getStyleName() {
						return isFirst ? TableConstants.STYLE_DATA_CHECKBOX : null;
					}

					@Override
					public Widget getWidget() {						
						/*
						 * if this is the first among the items with same revision , create checkbox in this column
						 */
						if (isFirst) {
							final CheckBoxWidget chkBox = new CheckBoxWidget();
							if (selectedRevisions.contains(revision))
								chkBox.setValue(true);
							chkBox.addClickHandler(new ClickHandler() {
								public void onClick(ClickEvent event) {
									boolean isCheck = chkBox.getValue();
									if (isCheck)
										selectedRevisions.add(revision);
									else
										selectedRevisions.remove(revision);
										
								}
							});
							return chkBox;
						}
						else
							return new LabelWidget("");
					}

					@Override
					public void release() {
					}
				};
			}

			private boolean isFirstProgressTitleVersion(ProgressItemRecord record) {
				boolean isFirst = false;
				String revision = record.getProgressItem().getTitleRevision();
				if (!titleRevisions.contains(revision)) {
					isFirst = true;
					titleRevisions.add(revision);
				}
				return isFirst;
			}
			
			@Override
			public String getColumnWidth() {
				return TableConstants.CHECK_BOX_COL_WIDTH;
			}
		};
	
		column.setCellStyle(TableConstants.STYLE_DATACELL);
		return column;
		
	}
	
	
	private Column<String, ProgressItemRecord> prepareRevisionColumn() {
		ColumnBase<String, ProgressItemRecord> column = null;

		String displayName = messages.progressTableRevisionColumn();
		column = new ColumnBase<String, ProgressItemRecord>(REVISION_COLUMN_NAME, displayName) {
			@Override
			public View<String> getView(final ProgressItemRecord record) {
				final boolean isLatestVersion = isProgressTitleVersionLatest(record);

				return new View<String>() {
					@Override
					public String getStyleName() {
						return !isLatestVersion ? TableConstants.STYLE_DATA_LINK : null;
					}

					@Override
					public Widget getWidget() {
						LabelWidget label = new LabelWidget(getRevisionValue(record
						        .getProgressItem()));
						/*
						 * if this is the latest version show it as a label else it can be
						 * hyperlinked.
						 */
						if (!isLatestVersion) {
							label.addClickHandler(new ClickHandler() {
								public void onClick(ClickEvent event) {
									String revision = record.getProgressItem().getTitleRevision();
									parent.getTab().showTitleRevision(revision);
								}
							});
						}
						return label;
					}

					@Override
					public void release() {
					}
				};
			}

			private boolean isProgressTitleVersionLatest(ProgressItemRecord record) {
				return record.getProgressItem().isLatestRevision();
			}
			
			@Override
			public String getColumnWidth() {
				return "5%";
			}
		};
	
		column.setCellStyle(TableConstants.STYLE_DATACELL_NUMERIC);
		return column;
		
	}

	private Column<String, ProgressItemRecord> prepareUserColumn() {
		ColumnBase<String, ProgressItemRecord> column = null;

		String displayName = messages.progressTableUserColumn();
		column = new ColumnBase<String, ProgressItemRecord>(USER_COLUMN_NAME, displayName) {
			@Override
			public View<String> getView(final ProgressItemRecord record) {
				final Hyperlink hl = getHyperlinkInfoFromAppropriateComponent(record
				        .getProgressItem());

				return new View<String>() {
					@Override
					public String getStyleName() {
						return (hl != null && hl.getTargetHistoryToken() != null) ? TableConstants.STYLE_DATA_LINK
						        : null;
					}

					@Override
					public Widget getWidget() {
						if (hl != null) {
							if (hl.getTargetHistoryToken() != null) {
								return new HyperlinkWidget(hl.getText(), hl.isAsHTML(), hl
								        .getTargetHistoryToken());
							} else {
								return new LabelWidget(hl.getText());
							}
						} else {
							return new LabelWidget(record.getProgressItem()
							        .getSourceComponentName());
						}
					}

					@Override
					public void release() {
					}
				};
			}
			
			@Override
			public String getColumnWidth() {
				return "15%";
			}
		};
		column.setCellStyle(TableConstants.STYLE_DATACELL_TEXT);
		return column;

	}

	
	private Column<String, ProgressItemRecord> prepareDescriptionColumn() {
		ColumnBase<String, ProgressItemRecord> column = null;

		String displayName = messages.progressTableDescriptionColumn();
		column = new ColumnBase<String, ProgressItemRecord>(DESCRIPTION_COLUMN_NAME, displayName) {
			@Override
			public View<String> getView(final ProgressItemRecord record) {
				final String actionValue = getActionValue(record.getProgressItem());		
				final String revision = record.getProgressItem().getTitleRevision();
				
				final String descValue = getDescriptionValue(record.getProgressItem());
				return new View<String>() {
					@Override
					public String getStyleName() {
						return TableConstants.STYLE_DATA_TEXT;
					}

					@Override
					public Widget getWidget() {
						String titleChangedMsg=null;
						
						if (actionValue.equals(normalizedEventName))
								titleChangedMsg=messages.titleNormalizedMessage();
						else if (actionValue.equals(updatedEventName))
								titleChangedMsg=messages.titleUpdatedMessage();
						
						if (!revision.equals(firstRevision) && titleChangedMsg != null) {
							InlineLabelWidget descriptionLabel;
							if (descValue != null)
								descriptionLabel = new InlineLabelWidget(descValue + " " + titleChangedMsg + " - ");
							else
								descriptionLabel = new InlineLabelWidget(titleChangedMsg + " - ");
							FlowPanel descriptionMessagePanel = new FlowPanel(); 
							InlineLabelWidget viewChangeLabel = new InlineLabelWidget(messages.viewChangesMessage());
							viewChangeLabel.addStyleName(TableConstants.STYLE_DATA_LINK);
							viewChangeLabel.addClickHandler(new ClickHandler() {
								@Override
								public void onClick(ClickEvent event) {
									handleViewChangesLabelClick(revision);
								}
							});
							descriptionMessagePanel.add(descriptionLabel);
							descriptionMessagePanel.add(viewChangeLabel);

							return descriptionMessagePanel;
						}
						else
							return new LabelWidget(descValue);
					}

					@Override
					public void release() {
					}
				};
			}

			
			@Override
			public String getColumnWidth() {
				return "54%";
			}
		};
	
		column.setCellStyle(TableConstants.STYLE_DATACELL_TEXT);
		return column;	
	}
	
	private String getPreviousRevision(String revision) {
		String prevRevision=null;
		int curIndex = titleRevisions.indexOf(revision);
		if (curIndex < titleRevisions.size()-1)
			prevRevision = titleRevisions.get(curIndex+1);
		
		return prevRevision;
	}

	private void handleViewChangesLabelClick(String revision) {
		String prevRevision = getPreviousRevision(revision);
		parent.compare(revision, prevRevision);
	}
	
	private Hyperlink getHyperlinkInfoFromAppropriateComponent(UIProgressItem item) {
		ServiceLoader serviceLoader = NeptuneApplication.getApplication().getServiceLoader();
		List<ITitleProgressService> services = serviceLoader
		        .loadServices(ITitleProgressService.class);
		for (ITitleProgressService service : services) {
			Hyperlink hl = service.getLink(item.getSourceComponentName(), item
			        .getSourceEntityName(), item.getSourceId(), item.getTitleRevision());
			if (hl != null)
				return hl;
		}

		return null;
	}

	/**
	 * Get all the columns to display
	 * 
	 * @see com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider#getColumns()
	 */
	public List<Column<?, ProgressItemRecord>> getColumns() {
		return this.columns;
	}

	/**
	 * Get the Progress Item column with matching name
	 * 
	 * @param columnName The column name
	 * @return The matching column or null if no column matches
	 */
	public ProgressItemColumn getColumn(String columnName) {
		ProgressItemColumn match = null;

		for (Column<?, ProgressItemRecord> column : this.columns) {
			if ((column instanceof ProgressItemColumn) && (columnName.equals(column.getName()))) {
				match = (ProgressItemColumn) column;
				break;
			}
		}

		return match;
	}

	/**
	 * Do nothing for initialization
	 * 
	 * @see com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider#initialize(com.google.gwt.user.client.rpc.AsyncCallback)
	 */
	public void initialize(AsyncCallback<Void> callback) {
		/* do nothing */
		callback.onSuccess(null);
	}

	/**
	 * Get a single record from the server. Should never be used.
	 * 
	 * @see com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider#getRecord(java.lang.Object,
	 *      com.google.gwt.user.client.rpc.AsyncCallback)
	 */
	public void getRecord(Long key, AsyncCallback<ProgressItemRecord> callback) {
		/* should never be used */
		UIProgressItem progressItem = new UIProgressItem();
		progressItem.setId(key);
		callback.onSuccess(new ProgressItemRecord(progressItem));
	}

	/**
	 * Get all the records from the server
	 * 
	 * @see com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider#getRecords(com.google
	 *      .gwt.user.client.rpc.AsyncCallback)
	 */
	public void getRecords(final AsyncCallback<List<ProgressItemRecord>> callback) {
		titleRevisions.clear();
		parent.getTab().getViewInput().getHistory(new NeptuneAsyncCallback<UITitle>() {
			/* Success getting title history */
			public void onNeptuneSuccess(UITitle result) {
				List<ProgressItemRecord> records = new ArrayList<ProgressItemRecord>();
				for (UIProgressItem progressItem : result.getHistory().getProgressItems()) {
					records.add(new ProgressItemRecord(progressItem));
				}

				normalizedEventName = result.getHistory().getNormalizedName();
				updatedEventName = result.getHistory().getUpdatedName();
				parent.setSeriesTitle(result.getHistory().isSeriesTitle());
				int numRecs = records.size();
				if (numRecs > 0) {
					firstRevision = records.get(numRecs-1).getProgressItem().getTitleRevision();
				}
				
				callback.onSuccess(records);
				parent.handleGetRecordsSuccess(result, records.size());
			};

			/* Failure getting the title history */
			public void onNeptuneFailure(Throwable caught) {
				callback.onSuccess(new ArrayList<ProgressItemRecord>());
				parent.handleGetRecordsFailure(caught);
			};
		});
	}

	/**
	 * No check box required for this table
	 * 
	 * @see com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider#isCheckboxEnabled()
	 */
	public boolean isCheckboxEnabled() {
		return false;
	}

	/**
	 * The total record count is displayed for the table
	 * 
	 * @see com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider#isRecordCountEnabled()
	 */
	public boolean isRecordCountEnabled() {
		return true;
	}

	public Set<String> getselectedRevisions() {
		return selectedRevisions;
	}
	
	/* Get the action column value */
	private String getActionValue(UIProgressItem progressItem) {
		return progressItem.getName();
	}

	/* Get the description column value */
	private String getDescriptionValue(UIProgressItem progressItem) {
		return progressItem.getValue();
	}

	/* Get the date column value */
	private String getDateValue(UIProgressItem progressItem) {
		return progressItem.getFormattedTimestamp();
	}

	/* Get the revision column value */
	private String getRevisionValue(UIProgressItem progressItem) {
		return progressItem.getTitleRevision();
	}
}
