/*
 * Created on Apr 28, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.view.history;

import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.ContentComponent;
import com.tandbergtv.cms.portal.content.client.title.model.UITitle;
import com.tandbergtv.cms.portal.content.client.title.model.UITitleOverview;
import com.tandbergtv.cms.portal.content.client.title.view.TitleView;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewTab;
import com.tandbergtv.cms.portal.content.client.title.view.metadata.UITitleInfo;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.RefreshWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.RefreshEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.RefreshHandler;
import com.tandbergtv.cms.portal.content.client.title.view.compare.TitleCompareView;

/**
 * The title history tab
 * 
 * @author Vijay Silva
 */
public class TitleHistoryTab extends TitleViewTab {

	/* Properties */
	private RefreshWidget<ProgressHistoryPanel> progressHistoryPanel;
	private MetadataHistoryPanel metadataHistoryPanel;

	/* The refresh period */
	private static final int DEFAULT_REFRESH_RATE = 180;
	private static final int MINIMUM_REFRESH_RATE = 30;
	
	/** Tab Name */ 
	public static final String TAB_NAME = "History";

	/**
	 * Constructor
	 * 
	 * @param parent The title view
	 * @param tabKey The key for the tab
	 */
	public TitleHistoryTab(TitleView parent) {
		super(parent);
	}

	/**
	 * Create the progress item and metadata history panels
	 * 
	 * @see com.tandbergtv.cms.portal.content.client.title.view.TitleViewTab#initialize()
	 */
	@Override
	protected void initialize() {
		this.getContainer().addStyleDependentName("history");

		final ProgressHistoryPanel panel = new ProgressHistoryPanel(this);
		progressHistoryPanel = new RefreshWidget<ProgressHistoryPanel>(panel);
		progressHistoryPanel.setRefreshOnLoad(false);
		progressHistoryPanel.setRefreshPeriod(getRefreshPeriod());
		progressHistoryPanel.addRefreshHandler(new RefreshHandler() {
			@Override
			public void onRefresh(RefreshEvent event) {
				Widget w = panel.getWidget();
				if (!( w instanceof TitleCompareView ))
					handleRefreshEvent(event);
			}
		});

		metadataHistoryPanel = new MetadataHistoryPanel(this);

		/* Display the progress item panel */
		this.getContainer().setWidget(progressHistoryPanel);
	}

	/**
	 * Refresh the state of this tab.
	 */
	public void refreshTab() {
		/* Show the busy indicator */
		this.showBusyIndicator();

		/* Change the displayed panel to the progress item panel */
		this.getContainer().setWidget(progressHistoryPanel);

		/* Refresh each of the panels */
		progressHistoryPanel.getWidget().refresh();
		metadataHistoryPanel.refresh();
	}

	/* Mark the tab as updated, required for package protected access */
	@Override
	protected void markUpdated() {
		super.markUpdated();
	}

	// ========================================================================
	// ===================== AUTO REFRESH
	// ========================================================================

	/*
	 * Determine the refresh period
	 */
	private int getRefreshPeriod() {
		/* Determine the refresh rate */
		NeptuneApplication application = NeptuneApplication.getApplication();
		ContentComponent component = application.getComponent(ContentComponent.class);
		Integer refreshRateValue = component.getTitleHistoryViewRefreshRate();
		int refreshRate = DEFAULT_REFRESH_RATE;
		try {
			if (refreshRateValue != null) {
				refreshRate = refreshRateValue;
				if (refreshRate < MINIMUM_REFRESH_RATE)
					refreshRate = MINIMUM_REFRESH_RATE;
			}
		} catch (Exception e) {
			refreshRate = DEFAULT_REFRESH_RATE;
		}

		return (refreshRate * 1000);
	}

	/*
	 * Handle the refresh event
	 */
	private void handleRefreshEvent(RefreshEvent event) {
		/* Determine if visible (the widget and all its parents must be visible) */
		boolean visible = true;
		Widget widget = progressHistoryPanel;
		while (widget != null && visible) {
			visible &= widget.isVisible();
			widget = widget.getParent();
		}

		/* Check if the widget needs to be refreshed from the server */
		if (visible) {
			/* Clear the cache and refresh */
			getViewInput().invalidateHistory();
			refresh();
		}
	}

	// ==============================================================
	// ===================== TITLE METADATA REVISION VIEW
	// ==============================================================

	/*
	 * Show the Title Revision page
	 */
	void showTitleRevision(final String titleRevision) {
		/* Check if the title is an existing title */
		Long titleId = this.getViewInput().getTitleId();
		if (titleId == null) return;

		/* Show the busy indicator */
		showBusyIndicator();

		this.getViewService().getTitleRevisionInfo(titleId, titleRevision,
		        new NeptuneAsyncCallback<UITitleInfo>() {

			        @Override
			        public void onNeptuneSuccess(UITitleInfo result) {
				        handleGetTitleMetadataSuccess(titleRevision, result);
			        }

			        public void onNeptuneFailure(Throwable caught) {
				        handleGetTitleMetadataFailure(titleRevision, caught);
			        }
		        });
	}


	/*
	 * Handle the successful fetching of the revisioned title
	 */
	private void handleGetTitleMetadataSuccess(String revision, UITitleInfo info) {
		metadataHistoryPanel.setInput(revision, info.getSpecification(), info.getTitle().getMetadata(), titleView);
		this.getContainer().setWidget(metadataHistoryPanel);
		hideBusyIndicator();
	}

	/*
	 * Handle the failure to get the revisioned title
	 */
	private void handleGetTitleMetadataFailure(String revision, Throwable caught) {
		String error = caught.getLocalizedMessage();
		String message = this.getViewMessages().titleRevisionFetchError(revision, error);

		/* Invalidate the history cache and show the error panel */
		getViewInput().invalidateHistory();

		/* Show the error panel and hide busy indicator */
		showErrorPanel(message);
		hideBusyIndicator();
	}

	// ==============================================================
	// ===================== TITLE PROGRESS HISTORY VIEW
	// ==============================================================

	/* Show the Progress History page */
	void showProgressHistory(boolean titleUpdated, String infoMessage, boolean invalidateHistory) 
	{
		//this.getTitleView().setFeedbackMessage(infoMessage);
		this.getContainer().setWidget(progressHistoryPanel);

		if (titleUpdated) {
			/* The entire title cache needs to be invalidated */
			UITitle title = new UITitle();
			title.setId(title.getId());
			this.onTitleUpdated(title);
		} else if (invalidateHistory) {
			/* only if the title is not completely invalidated, invalidate the history */
			getViewInput().invalidateHistory();
		}

		if (titleUpdated || invalidateHistory) {
			/* Refresh the table view */
			this.refresh();
		}
	}

	// ==============================================================
	// ===================== OTHER OVERRIDDEN METHODS
	// ==============================================================

	@Override
	protected void onTabUnselected() {
		this.markStale();
		this.getViewInput().invalidateHistory();
	}

	@Override
	protected String getTabName() {
	    return TAB_NAME;
	}

	@Override
	public String getTabDisplayName() {
		return getViewMessages().historyTabName();
	}

	@Override
	protected void updateTitleHeader(UITitleOverview overview) {
		super.updateTitleHeader(overview);
	}

	@Override
	protected int getTabHeight() {
		return super.getTabHeight();
	}

	@Override
	protected int getTabWidth() {
		return super.getTabWidth();
	}

	@Override
	protected int getUsedHeight() {
		return super.getUsedHeight();
	}

	@Override
	protected int getUsedWidth() {
		return super.getUsedWidth();
	}

	@Override
	protected void showBusyIndicator() {
		super.showBusyIndicator();
	}

	@Override
	protected void hideBusyIndicator() {
		super.hideBusyIndicator();
	}

	@Override
	protected void showSevereError(String message) {
		super.showSevereError(message);
	}

	@Override
	protected void showErrorPanel(String message) {
		super.showErrorPanel(message);
	}
	
}
