/*
 * Created on Jul 16, 2009
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.cms.portal.content.client.title.view.metadata;

import java.util.ArrayList;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.ScrollPanel;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.UITitleValidationMessage;
import com.tandbergtv.cms.portal.content.client.title.model.metadata.UITitleValidationMessageType;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewMessages;
import com.tandbergtv.cms.portal.content.client.title.view.asset.AssetValidationMessage;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

/**
 * The view for the Title Validation Messages
 * 
 * @author Vijay Silva
 */
public class TitleValidationView extends Composite 
{
	private SimpleContainer mainContainer;
	private LabelWidget blankLabel;
	private TitleViewMessages messages;

	/* Styles */
	private static final String WIDGET_STYLE_INFO = "content-TitleValidationViewInfo";
	private static final String WIDGET_STYLE_ERROR = "content-TitleValidationViewError";
	private static final String HEADING_MESSAGE_STYLE = "content-TitleValidationView-headingMessage";
	private static final String SECTION_HEADING_STYLE = "content-TitleValidationView-sectionHeading";
	private static final String ERROR_HEADING_STYLE = "content-TitleValidationView-sectionHeading-error";
	private static final String WARN_HEADING_STYLE = "content-TitleValidationView-sectionHeading-warning";
	private static final String INFO_HEADING_STYLE = "content-TitleValidationView-sectionHeading-information";
	private static final String MESSAGE_STYLE = "content-TitleValidationView-message";
	private static final String ERROR_MESSAGE_STYLE = "content-TitleValidationView-message-error";
	private static final String WARN_MESSAGE_STYLE = "content-TitleValidationView-message-warning";
	private static final String INFO_MESSAGE_STYLE = "content-TitleValidationView-message-information";
	private static final String FIELD_LINK_STYLE = "content-TitleValidationView-fieldLink";
	private static final String VALIDATION_WIDGET_STYLE = "content-TitleValidationView-messageWidget";

	/**
	 * Constructor
	 */
	public TitleValidationView() 
	{
		initialize();
	}

	/*
	 * Initialize the widgets to display
	 */
	private void initialize() 
	{
		messages = GWT.create(TitleViewMessages.class);

		/* Build the widgets */
		mainContainer = new SimpleContainer();
		blankLabel = new LabelWidget("");
		mainContainer.setWidget(blankLabel);
		initWidget(mainContainer);
	}

	
	/**
	 * Refresh the generated view with the current state of the input
	 */
	public void refresh(TitleValidationViewInput input) 
	{
		/* Make invisible, and set visible only if messages are present */
		this.setVisible(false);

		/* Clear out the previous widget displayed */
		mainContainer.setWidget(new LabelWidget(""));

		if(input == null || (!input.hasValidationMessages() && input.getMessageHeader() == null)) 
		{
			return;
		}

		/* Build the table that contains all the messages */
		VerticalContainer messagePanel = new VerticalContainer();
		ScrollPanel scrollPanel = new ScrollPanel(messagePanel);
		scrollPanel.setStyleName("TitleValidationViewScroll");

		LabelWidget headingMessageLabel = new LabelWidget();
		headingMessageLabel.addStyleName(HEADING_MESSAGE_STYLE);
		headingMessageLabel.setText(input.getMessageHeader());
		messagePanel.add(headingMessageLabel);
		
		/* filter messages based on message uniqueness */
		filterTitleValidationMessages(input);

		/* Display messages by type: error */
		int errorCount = input.getErrorMessageCount();
		if (errorCount > 0) {
			/* Section Header */
			String sectionHeading = messages.titleValidationErrorHeading(errorCount);
			LabelWidget sectionLabel = new LabelWidget(sectionHeading);
			sectionLabel.addStyleName(SECTION_HEADING_STYLE);
			sectionLabel.addStyleName(ERROR_HEADING_STYLE);
			messagePanel.add(sectionLabel);

			/* Display the metadata validation messages first */
			for(AssetValidationMessage message : input.getAssetValidationMessages()) 
			{
				Widget widget = buildMetadataValidationMessageWidget(message, input.getMetadataTab());
				messagePanel.add(widget);
			}

			/* Display the error title validation messages */
			for (UITitleValidationMessage message : input.getTitleValidationMessages()) {
				if (UITitleValidationMessageType.Error.equals(message.getMessageType())) {
					Widget widget = buildTitleValidationMessageWidget(message);
					messagePanel.add(widget);
				}
			}
		}

		/* Display messages by type: warning */
		int warningCount = input.getWarningMessageCount();
		if (warningCount > 0) {
			/* Section Header */
			String sectionHeading = messages.titleValidationWarningHeading(warningCount);
			LabelWidget sectionLabel = new LabelWidget(sectionHeading);
			sectionLabel.addStyleName(SECTION_HEADING_STYLE);
			sectionLabel.addStyleName(WARN_HEADING_STYLE);
			messagePanel.add(sectionLabel);

			/* Display the title validation messages */
			for (UITitleValidationMessage message : input.getTitleValidationMessages()) {
				if (UITitleValidationMessageType.Warning.equals(message.getMessageType())) {
					Widget widget = buildTitleValidationMessageWidget(message);
					messagePanel.add(widget);
				}
			}
		}

		/* Display messages by type: information */
		int infoCount = input.getInformationMessageCount();
		if (infoCount > 0) {
			/* Section Header */
			String sectionHeading = messages.titleValidationInformationHeading(infoCount);
			LabelWidget sectionLabel = new LabelWidget(sectionHeading);
			sectionLabel.addStyleName(SECTION_HEADING_STYLE);
			sectionLabel.addStyleName(INFO_HEADING_STYLE);
			messagePanel.add(sectionLabel);

			/* Display the title validation messages */
			for (UITitleValidationMessage message : input.getTitleValidationMessages()) {
				if (UITitleValidationMessageType.Information.equals(message.getMessageType())) {
					Widget widget = buildTitleValidationMessageWidget(message);
					messagePanel.add(widget);
				}
			}
		}

		/* Set the built table */
		mainContainer.setWidget(scrollPanel);
		
		if(input.isInfoType())
		{
			mainContainer.setStyleName(WIDGET_STYLE_INFO);
		}
		else
		{
			mainContainer.setStyleName(WIDGET_STYLE_ERROR);
		}
		
		this.setVisible(true);
	}

	/*
	 * Filters title validation messages in input based on the localized message string for each
	 * message.
	 */
	private void filterTitleValidationMessages(TitleValidationViewInput input) 
	{
		List<UITitleValidationMessage> messagesToDisplay = new ArrayList<UITitleValidationMessage>();
		for (UITitleValidationMessage message : input.getTitleValidationMessages()) {
			// add unique messages only.
			if (!messageExists(message, messagesToDisplay)) {
				messagesToDisplay.add(message);
			}
		}
		input.getTitleValidationMessages().clear();
		input.getTitleValidationMessages().addAll(messagesToDisplay);
	}

	/*
	 * Checks if the given message exists (based on the message text) in the list of messages
	 * passed.
	 */
	private boolean messageExists(UITitleValidationMessage message,
	        List<UITitleValidationMessage> messages) {
		for (UITitleValidationMessage m : messages) {
			if (m.getMessage().equalsIgnoreCase(message.getMessage())) {
				return true;
			}
		}
		return false;
	}

	/*
	 * Build the widget to show the metadata validation message
	 */
	private Widget buildMetadataValidationMessageWidget(final AssetValidationMessage message, final TitleMetadataTab metadataTab) 
	{
		HorizontalContainer widget = new HorizontalContainer();

		String assetType = message.getAsset().getDefinition().getDisplayName();
		String fieldName = message.getFieldName();
		String fieldText = messages.titleValidationMessageField(assetType, fieldName);

		LabelWidget fieldLabel = new LabelWidget(fieldText);
		fieldLabel.addStyleName(MESSAGE_STYLE);
		fieldLabel.addStyleName(ERROR_MESSAGE_STYLE);
		fieldLabel.addStyleName(FIELD_LINK_STYLE);
		fieldLabel.addClickHandler(new ClickHandler() 
		{
			@Override
			public void onClick(ClickEvent event) 
			{
				handleFieldLabelClicked(message, event, metadataTab);
			}
		});
		widget.add(fieldLabel);

		LabelWidget messageLabel = new LabelWidget(message.getMessage());
		messageLabel.addStyleName(MESSAGE_STYLE);
		messageLabel.addStyleName(ERROR_MESSAGE_STYLE);
		widget.add(messageLabel);

		widget.addStyleName(VALIDATION_WIDGET_STYLE);

		return widget;
	}

	/*
	 * Handle the event when the field label is clicked
	 */
	private void handleFieldLabelClicked(AssetValidationMessage message, ClickEvent event, TitleMetadataTab metadataTab) 
	{
		// metadataTab is null when this View is used from non-Metadata tab
		if(metadataTab != null) 
		{
			/* Need to direct focus to this widget */
			metadataTab.showWidget(message);
		}
	}

	/*
	 * Build the Widget for the Title Validation Message
	 */
	private Widget buildTitleValidationMessageWidget(UITitleValidationMessage message) 
	{
		LabelWidget widget = new LabelWidget(message.getMessage());
		String styleName = null;
		switch (message.getMessageType()) {
			case Error:
				styleName = ERROR_MESSAGE_STYLE;
				break;
			case Warning:
				styleName = WARN_MESSAGE_STYLE;
				break;
			case Information:
				styleName = INFO_MESSAGE_STYLE;
				break;
		}

		widget.addStyleName(MESSAGE_STYLE);
		if (styleName != null)
			widget.addStyleName(styleName);
		widget.addStyleName(VALIDATION_WIDGET_STYLE);
		return widget;
	}
}
