package com.tandbergtv.cms.portal.content.client.title.view.preview;

import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.ResizeEvent;
import com.google.gwt.event.logical.shared.ResizeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.DeferredCommand;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.DockPanel;
import com.tandbergtv.cms.portal.content.client.title.model.UIActionTitle;
import com.tandbergtv.cms.portal.content.client.title.service.ITitleViewService;
import com.tandbergtv.cms.portal.content.client.title.service.ITitleViewServiceAsync;
import com.tandbergtv.cms.portal.content.client.title.view.HtmlorXmlDisplayHelper;
import com.tandbergtv.cms.portal.content.client.title.view.TitleErrorPanel;
import com.tandbergtv.cms.portal.content.client.title.view.TitleErrorPanel.EventListener;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewMessages;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIAssetSpecificationListItem;
import com.tandbergtv.cms.portal.ui.title.client.service.specification.IUISpecificationManagerAsync;
import com.tandbergtv.cms.portal.ui.title.client.service.specification.UISpecificationClientManager;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.FrameWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ListBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.DockContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.HasViewCancelHandlers;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.ViewCancelHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

/**
 * The title preview widget
 * 
 * @author Vijay Silva
 */
public class TitlePreviewWidget extends Composite implements HasViewCancelHandlers {

	private final SimpleContainer container;
	private final TitleViewMessages messages;
	private DockContainer contents;
	private HorizontalContainer controlsContents;
	private ListBoxWidget<UIAssetSpecificationListItem> specificationBox;
	private LabelWidget descriptionLabel;
	private SimpleContainer previewResultContainer;
	private FrameWidget previewFrame;
	private SimpleContainer buttonsContents;
	private ButtonWidget cancelButton;
	private BusyIndicator busyIndicator;
	private ITitleViewServiceAsync service = null;
	private HandlerRegistration windowRegistration = null;
	private Command resizeCommand;
	private TitlePreviewWidgetInput input;
	private IUISpecificationManagerAsync specificationManager;

	/* Styles */
	private static final String STYLE_NAME = "content-TitlePreviewWidget";
	private static final String STYLE_CONTROLS_PANEL = "content-TitlePreviewWidget-controls";
	private static final String STYLE_BUTTONS_PANEL = "content-TitlePreviewWidget-buttons";
	private static final String STYLE_SPEC_LISTBOX = "content-TitlePreviewWidget-specificationListBox";
	private static final String STYLE_DESCRIPTION_LABEL = "content-TitlePreviewWidget-descriptionLabel";
	private static final String STYLE_PREVIEW_RESULT_CONTAINER = "content-TitlePreviewWidget-resultContainer";
	private static final String STYLE_PREVIEW_LOADING_LABEL = "content-TitlePreviewWidget-previewLoadingLabel";
	private static final String STYLE_PREVIEW_FRAME_HOLDER = "content-TitlePreviewWidget-frameContainer";
	private static final String STYLE_PREVIEW_FRAME = "content-TitlePreviewWidget-frame";

	/**
	 * @param parent The parent view
	 */
	public TitlePreviewWidget(TitleViewMessages messages) {
		this.messages = messages;
		this.container = new SimpleContainer();
		initWidget(container);

		initialize();
	}

	/**
	 * @return the input
	 */
	public TitlePreviewWidgetInput getInput() {
		return input;
	}

	/**
	 * @param input the input to set
	 */
	public void setInput(TitlePreviewWidgetInput input) {
		this.input = input;
	}

	// ========================================================================
	// ===================== INITIALIZATION
	// ========================================================================

	private void initialize() {
		/* Initialize service */
		service = GWT.create(ITitleViewService.class);
		specificationManager = UISpecificationClientManager.getInstance();

		/* Set style */
		addStyleName(STYLE_NAME);

		/* Build the busy indicator */
		busyIndicator = new BusyIndicator();

		/* build the contents panel */
		contents = new DockContainer();

		/* Build drop-down box for specifications */
		controlsContents = new HorizontalContainer();
		controlsContents.addStyleName(STYLE_CONTROLS_PANEL);
		contents.add(controlsContents, DockPanel.NORTH);

		specificationBox = new ListBoxWidget<UIAssetSpecificationListItem>();
		specificationBox.addStyleName(STYLE_SPEC_LISTBOX);
		specificationBox.addChangeHandler(new ChangeHandler() {
			@Override
			public void onChange(ChangeEvent event) {
				handleSpecificationChange();
			}
		});
		controlsContents.add(specificationBox);

		descriptionLabel = new LabelWidget();
		descriptionLabel.addStyleName(STYLE_DESCRIPTION_LABEL);
		controlsContents.add(descriptionLabel);

		/* Build preview contents section */
		previewResultContainer = new SimpleContainer();
		previewResultContainer.addStyleName(STYLE_PREVIEW_RESULT_CONTAINER);
		resizeCommand = new Command() {
			@Override
			public void execute() {
				resizeFrame();
			}
		};
		contents.add(previewResultContainer, DockPanel.CENTER);

		/* Cancel button */
		buttonsContents = new SimpleContainer();
		buttonsContents.addStyleName(STYLE_BUTTONS_PANEL);
		contents.add(buttonsContents, DockPanel.SOUTH);

		cancelButton = new ButtonWidget(messages.cancelButton());
		cancelButton.addStyleDependentName(StyleNames.DATALOSS_BUTTON_STYLE);
		cancelButton.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				handleCancelButtonClick();
			}
		});
		buttonsContents.setWidget(cancelButton);
	}

	// ========================================================================
	// ===================== REFRESH
	// ========================================================================

	public void refresh() {
		/* Clear previous state */
		container.clear();
		specificationBox.clear();
		descriptionLabel.setText("");
		hidePreviewFrame();

		/* Check if input is present */
		if (getInput() == null)
			return;

		if (getInput().getSpecifications() != null) {
			handleGetAllSpecificationsSuccess(getInput().getSpecifications());
		} else {
			/* Make call to server to compare */
			busyIndicator.center();
			specificationManager.getPreviewSpecs(new NeptuneAsyncCallback<List<UIAssetSpecificationListItem>>() 
				{
			        @Override
			        public void onNeptuneSuccess(List<UIAssetSpecificationListItem> result) {
				        handleGetAllSpecificationsSuccess(result);
			        };

			        @Override
			        public void onNeptuneFailure(Throwable caught) {
				        handleGetAllSpecificationsFailure(caught);
			        };
		        });
		}
	}

	/*
	 * Get all specifications from the server success
	 */
	private void handleGetAllSpecificationsSuccess(List<UIAssetSpecificationListItem> result) {
		/* Cache specifications */
		getInput().setSpecifications(result);

		/* Add specifications */
		addSpecifications();
		updateDescriptionLabel();
		container.setWidget(contents);
		busyIndicator.hide();

		performPreviewAction();
	}

	/*
	 * Add the specifications, and select appropriate default
	 */
	private void addSpecifications() {
		List<UIAssetSpecificationListItem> specs = getInput().getSpecifications(); 
		String selectedSpecificationName = getInput().getSelectedSpecification();

		UIAssetSpecificationListItem selectedSpecification = null;
		
		for(UIAssetSpecificationListItem spec : specs) {
			// Don't display internal TTV spec
			if(!"TTV".equalsIgnoreCase(spec.getName())) {
				specificationBox.addItem(spec.getDisplayName(), spec);
				if(spec.getName().equals(selectedSpecificationName)) {
					selectedSpecification = spec;
				}
			}
		}

		/* Set the selected specification */
		if (selectedSpecification != null) {
			specificationBox.setSelectedItem(selectedSpecification);
		}
	}

	/*
	 * Update the text of the description label
	 */
	private void updateDescriptionLabel() {
		String version = getInput().getTitle().getVersion();
		String description = (version == null) ? messages.previewTitleDescriptionLabel() : messages
		        .previewTitleVersionDescriptionLabel(version);
		descriptionLabel.setText(description);
	}

	/*
	 * Handle the failure
	 */
	private void handleGetAllSpecificationsFailure(Throwable error) {
		busyIndicator.hide();
		String msg = messages.specificationNamesFetchFailure(error.getLocalizedMessage());

		TitleErrorPanel errorPanel = new TitleErrorPanel(msg, true);
		errorPanel.addViewCancelHandler(new ViewCancelHandler() {
			@Override
			public void onCancel(ViewCancelEvent event) {
				handleCancelButtonClick();
			}
		});

		errorPanel.addEventListener(new TitleErrorPanel.EventListener() {
			@Override
			public void onRefreshClicked(TitleErrorPanel panel) {
				refresh();
			}
		});

		container.setWidget(errorPanel);
	}

	// ========================================================================
	// ===================== PREVIEW ACTION
	// ========================================================================

	/*
	 * Ensure that the preview screen is initialized when first rendered
	 */
	private void performPreviewAction() {
		/* Clear the previous preview contents */
		hidePreviewFrame();

		/* Check if input is set */
		if (getInput() == null) {
			return;
		}

		/* Show preview message */
		LabelWidget previewLoadingLabel = new LabelWidget();
		previewLoadingLabel.addStyleName(STYLE_PREVIEW_LOADING_LABEL);
		previewLoadingLabel.setText(messages.previewTitleLoadingMessage());
		previewResultContainer.setWidget(previewLoadingLabel);
		
		/* Build the previews */
		UIActionTitle title = getInput().getTitle();
		UIAssetSpecificationListItem specification = specificationBox.getSelectedItem();
		String name = (specification != null) ? specification.getName() : null;
		busyIndicator.center();
		service.previewTitle(title, name, HtmlorXmlDisplayHelper.useHtml(), new NeptuneAsyncCallback<String>() {
			@Override
			public void onNeptuneFailure(Throwable caught) {
				handlePreviewTitleFailure(caught);
			}

			@Override
			public void onNeptuneSuccess(String result) {
				handlePreviewTitleSuccess(result);
			}
		});
	}

	/*
	 * Successfully generated preview
	 */
	private void handlePreviewTitleSuccess(String url) {
		/* Build and show the preview frame */
		showPreviewFrame(url);

		/* Update the frame size */
		updateFrameSize();

		/* Hide busy indicator */
		busyIndicator.hide();
	}

	/*
	 * Failed to generate preview
	 */
	private void handlePreviewTitleFailure(Throwable caught) {
		String error = caught.getLocalizedMessage();
		String message = messages.previewError(error);

		/* Show the error panel and hide busy indicator */
		TitleErrorPanel errorPanel = new TitleErrorPanel(message);

		/* Listen for the refresh event */
		errorPanel.addEventListener(new EventListener() {
			public void onRefreshClicked(TitleErrorPanel panel) {
				refresh();
			}
		});
		previewResultContainer.setWidget(errorPanel);

		/* Hide busy indicator */
		busyIndicator.hide();
	}

	/*
	 * Set the preview frame
	 */
	private void showPreviewFrame(String previewURL) {
		/* Build the preview frame */
		previewFrame = new FrameWidget();
		previewFrame.addStyleName(STYLE_PREVIEW_FRAME);
		previewFrame.setUrl(previewURL);

		SimpleContainer frameHolder = new SimpleContainer();
		frameHolder.addStyleName(STYLE_PREVIEW_FRAME_HOLDER);
		frameHolder.setWidget(previewFrame);

		previewResultContainer.setWidget(frameHolder);
	}

	/*
	 * Clear the preview frame
	 */
	private void hidePreviewFrame() {
		previewResultContainer.clear();
		previewFrame = null;
	}

	// ========================================================================
	// ===================== EVENT HANDLING
	// ========================================================================

	@Override
	public HandlerRegistration addViewCancelHandler(ViewCancelHandler handler) {
		return addHandler(handler, ViewCancelEvent.getType());
	}

	@Override
	protected void onLoad() {
		super.onLoad();

		windowRegistration = Window.addResizeHandler(new ResizeHandler() {
			public void onResize(ResizeEvent event) {
				updateFrameSize();
			}
		});
	};

	@Override
	protected void onUnload() {
		windowRegistration.removeHandler();
		windowRegistration = null;
		busyIndicator.hide();

		super.onUnload();
	}

	/*
	 * Handle change in selected specification
	 */
	private void handleSpecificationChange() {
		performPreviewAction();
	}

	/*
	 * Update the size of the asset container widget
	 */
	private void updateFrameSize() {
		DeferredCommand.addCommand(resizeCommand);
	}

	/*
	 * Resize the scroll container
	 */
	private void resizeFrame() {
		if (previewFrame == null) {
			return;
		}

		/* Calculate the size to use */
		int width = Window.getClientWidth() - getInput().getRestrictedWidth() - 5;
		int height = Window.getClientHeight() - getInput().getRestrictedHeight() - 10;
		height -= buttonsContents.getOffsetHeight() + controlsContents.getOffsetHeight();

		/* Set the dimensions */
		previewFrame.setPixelSize(width, height);
	}

	/*
	 * Handle the cancel button click
	 */
	private void handleCancelButtonClick() {
		fireEvent(new ViewCancelEvent());
	}
}
