package com.tandbergtv.cms.portal.content.client.title.view.sites;

import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.cms.portal.content.client.ContentComponent;
import com.tandbergtv.cms.portal.content.client.title.model.UIActionTitle;
import com.tandbergtv.cms.portal.content.client.title.model.UITitle;
import com.tandbergtv.cms.portal.content.client.title.model.compare.TitleCompareRequest;
import com.tandbergtv.cms.portal.content.client.title.model.compare.UITitleCompareData;
import com.tandbergtv.cms.portal.content.client.title.model.compare.UITitleCompareXmlData;
import com.tandbergtv.cms.portal.content.client.title.model.sites.UITitleSites;
import com.tandbergtv.cms.portal.content.client.title.service.ITitleSitesService;
import com.tandbergtv.cms.portal.content.client.title.service.ITitleSitesServiceAsync;
import com.tandbergtv.cms.portal.content.client.title.service.ITitleViewService;
import com.tandbergtv.cms.portal.content.client.title.service.ITitleViewServiceAsync;
import com.tandbergtv.cms.portal.content.client.title.service.ReadDeletedTitleException;
import com.tandbergtv.cms.portal.content.client.title.view.HtmlorXmlDisplayHelper;
import com.tandbergtv.cms.portal.content.client.title.view.TitleView;
import com.tandbergtv.cms.portal.content.client.title.view.TitleViewTab;
import com.tandbergtv.cms.portal.content.client.title.view.compare.ITitleCompareViewController;
import com.tandbergtv.cms.portal.content.client.title.view.compare.TitleCompareView;
import com.tandbergtv.cms.portal.content.client.title.view.compare.TitleCompareViewInput;
import com.tandbergtv.cms.portal.content.client.title.view.series.SeriesTitleView;
import com.tandbergtv.cms.portal.ui.title.client.model.specification.UIAssetSpecificationListItem;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.RefreshWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.RefreshEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.event.RefreshHandler;


/**
 * The Title Sites Tab
 *
 * @author bng
 */
public class TitleSitesTab extends TitleViewTab implements ITitleCompareViewController 
{
	/* The refresh period */
	private static final int DEFAULT_REFRESH_RATE = 180;
	private static final int MINIMUM_REFRESH_RATE = 30;
	private static final String STYLE_LOADING_LABEL = "content-TitleMetadataTab-loadingMessage";
	
	/* Properties */
	private RefreshWidget<LocalTitleSitesPanel> localSitesPanel;
	private RefreshWidget<SitesPanel> masterSitesPanel;

	private TitleCompareView compareView;
	private TitleCompareRequest lastRequest;
	
	/* The title view service */
	private ITitleSitesServiceAsync titleSitesService = GWT.create(ITitleSitesService.class);	
	private ITitleViewServiceAsync viewService = GWT.create(ITitleViewService.class);
	
	/** Tab Name */
	public static final String TAB_NAME = "Sites";
	
	
	/**
	 * Constructor
	 * 
	 * @param parent The title view
	 * @param tabKey The key for the tab
	 */
	public TitleSitesTab(TitleView parent) {
		super(parent);
	}	
	/**
	 * Create the sites panels
	 * 
	 * @see com.tandbergtv.cms.portal.content.client.title.view.TitleViewTab#initialize()
	 */
	@Override
	protected void initialize() {
		this.getContainer().addStyleDependentName("sites");

		LocalTitleSitesPanel panel = new LocalTitleSitesPanel(this);
		localSitesPanel = new RefreshWidget<LocalTitleSitesPanel>(panel);
		localSitesPanel.setRefreshOnLoad(false);
		localSitesPanel.setRefreshPeriod(getRefreshPeriod());
		localSitesPanel.addRefreshHandler(new RefreshHandler() {
			@Override
			public void onRefresh(RefreshEvent event) {
				handleRefreshEvent(event);
			}
		});

		SitesPanel masterPanel = new SitesPanel(this);
		masterSitesPanel = new RefreshWidget<SitesPanel>(masterPanel);
		masterSitesPanel.setRefreshOnLoad(false);
		masterSitesPanel.setRefreshPeriod(getRefreshPeriod());
		masterSitesPanel.addRefreshHandler(new RefreshHandler() {
			@Override
			public void onRefresh(RefreshEvent event) {
				handleRefreshEvent(event);
			}
		});
	}

	private void showLoadingMessage() {
		/* Show a loading message as the main widget */
		String message = this.getViewMessages().sitesLoading();
		LabelWidget label = new LabelWidget(message);
		label.addStyleName(STYLE_LOADING_LABEL);
		getContainer().setWidget(label);
	}

	@Override
	protected String getTabName() {
	    return TAB_NAME;
	}

	@Override
	public String getTabDisplayName() {
		return getViewMessages().sitesTabName();
	}

	public void refreshTab() {
		/* Show the busy indicator */
		showBusyIndicator();

		/* Show the loading message */
		showLoadingMessage();

		/* Get the data from the server */
		refreshTitleSites();
	}
	
	private void refreshTitleSites() 
	{
		// Clear message panel
		getTitleView().showMessage(null);
		
		Long titleId = getViewInput().getTitleId();
		titleSitesService.getTitleSites(titleId, new NeptuneAsyncCallback<UITitle>() {
			/* Success getting title sites */
			public void onNeptuneSuccess(UITitle title) {
				UITitleSites result = title.getSites();
				if (result.getLocal() != null) {
					localSitesPanel.getWidget().handleGetTitleSiteRecordsSuccess(result.getLocal());
					getContainer().setWidget(localSitesPanel);
				}
				else {
					masterSitesPanel.getWidget().handleGetTitleSiteRecordsSuccess(result);
					getContainer().setWidget(masterSitesPanel);
				}
				updateTitleHeader(title.getOverview());
				hideBusyIndicator();		
			};

			/* Failure getting the title sites */
			public void onNeptuneFailure(Throwable caught) {
				if (caught instanceof ReadDeletedTitleException) {
					showSevereError(caught.getMessage());
				} else {
					String error = caught.getLocalizedMessage();
					String message = getViewMessages().sitesFetchFailure(error);

					/* Show the error panel */
					showErrorPanel(message);
				}

				hideBusyIndicator();
			};
		});
	}

	void targetWithRules() {
		Long titleId = getViewInput().getTitleId();
		titleSitesService.targetWithRules(titleId, new NeptuneAsyncCallback<UITitle>() {
			/* Success target with rules */
			public void onNeptuneSuccess(UITitle result) {
				masterSitesPanel.getWidget().handleGetTitleSiteRecordsSuccess(result.getSites());
				if (result.getSites().getActiveSiteNames().size() == 0)
					masterSitesPanel.getWidget().reportNoSitesTargeted();
				else
					masterSitesPanel.getWidget().showTargetingInitiatedMessage();
				updateTitleHeader(result.getOverview());
			};

			/* Failure target with rules */
			public void onNeptuneFailure(Throwable caught) {
				masterSitesPanel.getWidget().reportFailure(caught.getMessage());
			};
		});
	}
	
	void target(List<Integer> siteIds) {
		Long titleId = getViewInput().getTitleId();
		titleSitesService.target(titleId, siteIds, new NeptuneAsyncCallback<UITitle>() {
			/* Success getting title sites */
			public void onNeptuneSuccess(UITitle result) {
				masterSitesPanel.getWidget().handleGetTitleSiteRecordsSuccess(result.getSites());
				masterSitesPanel.getWidget().showTargetingInitiatedMessage();
				updateTitleHeader(result.getOverview());
			};

			/* Failure getting the title sites */
			public void onNeptuneFailure(Throwable caught) {
				masterSitesPanel.getWidget().reportFailure(caught.getMessage());
			};
		});
	}
	
	
	// ========================================================================
	// ===================== AUTO REFRESH
	// ========================================================================

	/*
	 * Determine the refresh period
	 */
	private int getRefreshPeriod() {
		/* Determine the refresh rate */
		NeptuneApplication application = NeptuneApplication.getApplication();
		ContentComponent component = application.getComponent(ContentComponent.class);
		Integer refreshRateValue = component.getTitleSitesViewRefreshRate();
		int refreshRate = DEFAULT_REFRESH_RATE;
		try {
			if (refreshRateValue != null) {
				refreshRate = refreshRateValue;
				if (refreshRate < MINIMUM_REFRESH_RATE)
					refreshRate = MINIMUM_REFRESH_RATE;
			}
		} catch (Exception e) {
			refreshRate = DEFAULT_REFRESH_RATE;
		}

		return (refreshRate * 1000);
	}

	/*
	 * Handle the refresh event
	 */
	private void handleRefreshEvent(RefreshEvent event) {
		/* Determine if visible (the widget and all its parents must be visible) */
		boolean visible = true;
		Widget widget = this.getContainer().getWidget();
		while (widget != null && visible) {
			visible &= widget.isVisible();
			widget = widget.getParent();
		}

		/* Check if the widget needs to be refreshed from the server */
		if (visible) {
			refresh();
		}
	}

	// ========================================================================
	// ===================== SHOWING COMPARE
	// ========================================================================

	/**
	 * Show the master title comparison view
	 * 
	 * @param titleId The titleID of the master title being compared
	 * @param targetedVersion The version of the master title that was targeted
	 */
	public void showMasterTitleComparison(Long masterTitleId, String targetedVersion) 
	{
		compareView = new TitleCompareView(this);
		
		String heading = getViewMessages().compareTitleHeading();		
		if(!masterTitleId.equals(getViewInput().getTitleId()))
		{
			heading = getViewMessages().compareMasterTitleHeading();
		}
		
		TitleCompareViewInput input = buildCompareInput(masterTitleId, targetedVersion, heading);
		compareView.setInput(input);

		UIActionTitle title1 = new UIActionTitle();
		title1.setTitleId(masterTitleId);

		UIActionTitle title2 = new UIActionTitle();
		title2.setTitleId(masterTitleId);
		title2.setVersion(targetedVersion);

		lastRequest = new TitleCompareRequest();
		lastRequest.title1 = title1;
		lastRequest.title2 = title2;
		lastRequest.specName = input.specName;
		if(titleView instanceof SeriesTitleView)
			lastRequest.specificationType = "SERIES";
		else
			lastRequest.specificationType = "PACKAGE";

		compareDiff();
	}

	/**
	 * Show the distributed title comparison view
	 * 
	 * @param titleId The titleID of the site title being compared
	 * @param distributedVersion The version of the site title that was distributed
	 * @param masterTitleView Flag indicating if the user is viewing a master or site title
	 */
	public void showDistributedTitleComparison(Long titleId, String distributedVersion) 
	{
		compareView = new TitleCompareView(this);
		
		String heading = getViewMessages().compareTitleHeading();
		if(!titleId.equals(getViewInput().getTitleId()))
		{
			heading = getViewMessages().compareSiteTitleHeading();
		}
		
		TitleCompareViewInput input = buildCompareInput(titleId, distributedVersion, heading);
		compareView.setInput(input);
		
		UIActionTitle title1 = new UIActionTitle();
		title1.setTitleId(titleId);

		UIActionTitle title2 = new UIActionTitle();
		title2.setTitleId(titleId);
		title2.setVersion(distributedVersion);

		lastRequest = new TitleCompareRequest();
		lastRequest.title1 = title1;
		lastRequest.title2 = title2;
		lastRequest.specName = input.specName;
		if(titleView instanceof SeriesTitleView)
			lastRequest.specificationType = "SERIES";
		else
			lastRequest.specificationType = "PACKAGE";

		compareDiff();
	}

	/*
	 * Build the title compare view input
	 */
	private TitleCompareViewInput buildCompareInput(Long titleId, String version, String heading) 
	{
		TitleCompareViewInput input = new TitleCompareViewInput();
		input.specName = getViewInput().getSpecificationName();
		input.restrictedHeight = getUsedHeight();
		input.restrictedWidth = getUsedWidth();
		input.header = getViewMessages().titleCompareSummaryMessage(heading, getViewMessages().compareRevisionHeading(version));
		input.displayName1 = getViewMessages().compareCurrentRevisionName();
		input.displayName2 = getViewMessages().compareRevisionName(version);
		
		return input;
	}
	
	
	
	public void compareXml()
	{
		compareView.showBusyIndicator();
		
		viewService.compareTitlePreviews(lastRequest.title1, lastRequest.title2, 
				lastRequest.specName, HtmlorXmlDisplayHelper.useHtml(), new NeptuneAsyncCallback<UITitleCompareXmlData>() 
		{
			@Override
			public void onNeptuneFailure(Throwable caught) 
			{
				compareView.showErrorView(caught);
				compareView.hideBusyIndicator();
			}

			@Override
			public void onNeptuneSuccess(UITitleCompareXmlData result) 
			{
				compareView.showXmlView(result);
				compareView.hideBusyIndicator();
			}
		} 
		);
	}


	public void compareDiff()
	{
		compareView.showBusyIndicator();
		
		viewService.compareTitles(lastRequest, new NeptuneAsyncCallback<UITitleCompareData>() 
		{
			@Override
			public void onNeptuneFailure(Throwable caught) 
			{
				compareView.showErrorView(caught);
				compareView.hideBusyIndicator();
			}

			@Override
			public void onNeptuneSuccess(UITitleCompareData data) 
			{
				setSpecName(data);
				compareView.getInput().compareData = data;
				compareView.showDiffView();
				compareView.hideBusyIndicator();
			}
		});
		
		getContainer().setWidget(compareView);
		this.markUpdated();
	}
	
	
	private void setSpecName(UITitleCompareData data)
	{
		// Set default spec name if it is null
		if(lastRequest != null && data != null)
		{
			if(lastRequest.specName == null)
			{
				List<UIAssetSpecificationListItem> specs = data.getSpecifications(); 
				if(specs != null && specs.size() > 0) lastRequest.specName = specs.get(0).getName();
			}
		}
	}

	
	
	// ==============================================================
	// ===================== OTHER OVERRIDDEN METHODS
	// ==============================================================
	@Override
	public TitleView getTitleView() {
		return super.getTitleView();
	}

	
	@Override
	protected void onTabUnselected() {
		masterSitesPanel.getWidget().setPreviousSelections();
		markStale();
	}
	
	@Override
	protected int getTabHeight() {
		return super.getTabHeight();
	}

	@Override
	protected int getTabWidth() {
		return super.getTabWidth();
	}

	@Override
	protected void showBusyIndicator() {
		super.showBusyIndicator();
	}
	
	
	/* Mark the tab as updated, required for package protected access */
	@Override
	protected void markUpdated() {
		super.markUpdated();
	}	
	
	@Override
	protected void hideBusyIndicator() 
	{
		super.hideBusyIndicator();
	}
	
	@Override
	public void onCancel() 
	{
		refresh();
	}
	
	
	@Override
	public void onShowDifferences() 
	{
		compareDiff();
	}
	
	@Override
	public void onShowXml() 
	{
		compareXml();
	}
	
	@Override
	public void onSpecChanged(String newSpec) 
	{
		lastRequest.specName = newSpec;
		compareView.getInput().specName = newSpec;
		compareXml();
	}
}
