package com.ericsson.cms.contractmgmt.client.document;

import java.util.ArrayList;
import java.util.List;

import com.ericsson.cms.contractmgmt.client.ContractComponentConstants;
import com.ericsson.cms.contractmgmt.client.ContractComponentMessages;
import com.ericsson.cms.contractmgmt.client.Permissions;
import com.ericsson.cms.contractmgmt.client.model.UiDocument;
import com.ericsson.cms.contractmgmt.client.model.UiDocumentType;
import com.ericsson.cms.contractmgmt.client.rpc.IFileService;
import com.ericsson.cms.contractmgmt.client.rpc.IFileServiceAsync;
import com.ericsson.cms.contractmgmt.client.tab.ErrorPanel;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ChangeEvent;
import com.google.gwt.event.dom.client.ChangeHandler;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.FileUpload;
import com.google.gwt.user.client.ui.FormPanel;
import com.google.gwt.user.client.ui.FormPanel.SubmitCompleteEvent;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.HasVerticalAlignment;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.CheckBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.FlexTableContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

public class DocumentComponentFactory {

    private static final String TABLE_STYLE = "contract-PlatformTable";
    private static final String ROW_STYLE = "contract-PlatformTable-row";
    private static final String ROW_HEADER_STYLE = "contract-PlatformTable-row-header";
    private static final String ROW_BODY_STYLE = "contract-PlatformTable-row-body";
    private static final String CELL_STYLE = "contract-PlatformTable-cell";
    private static final String CELL_HEADER_STYLE = "contract-PlatformTable-cell-header";

    private final IFileServiceAsync fileService = GWT.create(IFileService.class);
    private final ContractComponentConstants constants = (ContractComponentConstants) GWT
            .create(ContractComponentConstants.class);

    private final ContractComponentMessages messages = (ContractComponentMessages) GWT
            .create(ContractComponentMessages.class);

    private FileUpload fileUpload;

    private static final String SERVER_ERROR = "An error occurred while "
            + "attempting to contact the server. Please check your network " + "connection and try again.";

    private final BusyIndicator busyIndicator = new BusyIndicator();

    private ButtonWidget uploadButton;

    private final FormPanel form = new FormPanel();

    private FlexTableContainer tableDocuments;

    private List<UiDocument> documentsList;

    private final Long recordKey;

    private final ErrorPanel errorPanel;

    private final String text;

    private final UiDocumentType uiDocumentType;

    private ButtonWidget removeButton;

    private HorizontalContainer horizontal;

    private CheckBoxWidget check;

    public DocumentComponentFactory(Long recordKey, String text, UiDocumentType uiDocumentType, ErrorPanel errorPanel) {
        this.recordKey = recordKey;
        this.uiDocumentType = uiDocumentType;
        this.errorPanel = errorPanel;
        this.text = text;
        buildDisclosureDocuments();
    }

    public FormPanel getDocumentPanel(){
        verifyPermissions();
        return form;
    }

    private void buildDisclosureDocuments() {

        fileUpload = new FileUpload();
        fileUpload.setName("uploadFormElement");
        fileUpload.setEnabled(false);

        fileUpload.addChangeHandler(new ChangeHandler() {

            @Override
            public void onChange(ChangeEvent event) {
                if(ClientAuthorizationManager.isAuthorized(Permissions.CONTRACT_MODIFY) && recordKey != null && !fileUpload.getFilename().trim().equalsIgnoreCase("")){
                    uploadButton.setEnabled(true);
                }
            }
        });

        form.setAction(GWT.getModuleBaseURL() + "uploadFile");

        // Because we're going to add a FileUpload widget, we'll need to set the
        // form to use the POST method, and multipart MIME encoding.
        form.setEncoding(FormPanel.ENCODING_MULTIPART);
        form.setMethod(FormPanel.METHOD_POST);

        uploadButton = new ButtonWidget(constants.uploadButtonName(), new ClickHandler() {

            @Override
            public void onClick(ClickEvent event) {
                if (!fileUpload.getFilename().trim().equalsIgnoreCase("") ) {
                    busyIndicator.center();
                    try {
                        form.submit();
                    } catch (Exception e) {
                        errorPanel.addMessage(e.getLocalizedMessage());
                    }
                }
            }

        });
        uploadButton.setEnabled(false);
        VerticalContainer vertical = new VerticalContainer();

        horizontal = new HorizontalContainer();
        horizontal.setVerticalAlignment(HasVerticalAlignment.ALIGN_MIDDLE);
        horizontal.add(fileUpload);
        addSpace(horizontal);
        horizontal.add(uploadButton);

        vertical.add(horizontal);


        tableDocuments = new FlexTableContainer();
        tableDocuments.addStyleName(TABLE_STYLE);

        // Create the header row
        tableDocuments.insertRow(0);
        tableDocuments.getRowFormatter().addStyleName(0, ROW_STYLE);
        tableDocuments.getRowFormatter().addStyleName(0, ROW_HEADER_STYLE);

        createTableDocumentsHeader();
        LabelWidget label = new LabelWidget();
        vertical.add(label);
        vertical.setCellHeight(label, "20px");
        label = new LabelWidget();
        label.setText(text);
        vertical.add(label);
        vertical.add(tableDocuments);
        addButtonsDocument(vertical);
        form.add(vertical);

        form.addSubmitCompleteHandler(new FormPanel.SubmitCompleteHandler() {
            @Override
            public void onSubmitComplete(final SubmitCompleteEvent event) {
                // When the form submission is successfully completed, this
                // event is fired. Assuming the service returned a response of type
                // text/html, we can get the result text here (see the FormPanel
                // documentation for further explanation).
                UiDocument uiDocument = new UiDocument();
                uiDocument.setParentId(recordKey);
                uiDocument.setDocumentType(uiDocumentType);
                fileService.uploadAttachment(constants.filePath(), uiDocument, new AsyncCallback<UiDocument>() {
                    @Override
                    public void onFailure(Throwable caught) {
                        // Show the RPC error message to the user
                        Window.alert(SERVER_ERROR + "<br>\n" + event.getResults());
                        busyIndicator.hide();
                    }

                    @Override
                    public void onSuccess(UiDocument result) {
                        documentsList.add(result);
                        addRowDocuments(result);
                        busyIndicator.hide();

                    }
                });
            }
        });

    }

    private void addSpace(HorizontalContainer container) {
        if (container != null) {
            HorizontalPanel space = new HorizontalPanel();
            LabelWidget label = new LabelWidget();
            space.add(label);
            space.setCellWidth(label, "10");
            container.add(space);
        }
    }

    private void addRowDocuments(UiDocument uiDocument){

        insertRowDocuments();
        int row = getRowDocuments();

        tableDocuments.getRowFormatter().setStyleName(row, TableConstants.STYLE_DATA_EVEN_ROW);

        CheckBoxWidget checkBox = new CheckBoxWidget();
        tableDocuments.setWidget(row, 0, checkBox);
        tableDocuments.getCellFormatter().setStyleName(row, 0,  TableConstants.STYLE_DATA_CHECKBOX );

        tableDocuments.insertCell(row, 1);
        String url = GWT.getModuleBaseURL() + "downloadFile?docId="+ uiDocument.getKey() + "&docType=" + uiDocument.getDocumentType().ordinal();
        tableDocuments.setWidget(row, 1, new HTML("<a href="+ url + ">"+ uiDocument.getFileName() +"</a>"));
        tableDocuments.getCellFormatter().setStyleName(row, 1,  TableConstants.STYLE_DATACELL_TEXT );

    }


    private void createTableDocumentsHeader() {
        check = new CheckBoxWidget();
        check.addClickHandler(new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {
                if (check.getValue()) {
                    for (int i = 0; i < tableDocuments.getRowCount(); i++) {
                        CheckBoxWidget all = (CheckBoxWidget) tableDocuments.getWidget(i, 0);
                        all.setValue(true);
                    }
                } else {
                    for (int i = 0; i < tableDocuments.getRowCount(); i++) {
                        CheckBoxWidget all = (CheckBoxWidget) tableDocuments.getWidget(i, 0);
                        all.setValue(false);
                    }
                }
            }
        });

        tableDocuments.getRowFormatter().setStyleName(0, TableConstants.STYLE_HEADER_ROW);
        tableDocuments.insertCell(0, 0);
        tableDocuments.getFlexCellFormatter().addStyleName(0, 0, CELL_STYLE);
        tableDocuments.getFlexCellFormatter().addStyleName(0, 0, CELL_HEADER_STYLE);
        tableDocuments.setWidget(0, 0, check);

        addHeading(constants.copyrightHolderFilePath(), tableDocuments);

    }

    private void addButtonsDocument(VerticalContainer container) {
        HorizontalContainer buttonsContainer = new HorizontalContainer();
        buttonsContainer.setSpacing(5);

        removeButton = new ButtonWidget(constants.removeButtonName(), new ClickHandler() {

            @Override
            public void onClick(ClickEvent event) {

                if (!isAtLeastOneRecordSelected()) {
                    Window.alert(messages.noRecordSelected());
                } else {
                    List<UiDocument> deleteList = new ArrayList<UiDocument>();

                    for (int i = 1; i < tableDocuments.getRowCount(); i++) {
                        if (((CheckBoxWidget) tableDocuments.getWidget(i, 0)).getValue()) {
                            // if its marked for deletion
                            deleteList.add(documentsList.get(i - 1));
                        }
                    }

                    removeDocument(deleteList);
                    if(check.getValue()){
                        check.setValue(false);
                    }
                }
            }
        });

        removeButton.addStyleDependentName(StyleNames.ACTION_BUTTON_STYLE);
        removeButton.setEnabled(false);

        buttonsContainer.add(removeButton);

        container.add(buttonsContainer);

    }

    private void removeDocument(final List<UiDocument> uiDocumentList ){
        busyIndicator.center();
        final int index = documentsList.indexOf(uiDocumentList.get(0));

        fileService.deleteAttachment(uiDocumentList.get(0).getKey(), uiDocumentType, new AsyncCallback<Boolean>() {

            @Override
            public void onFailure(Throwable caught) {
                errorPanel.addMessage(caught.getLocalizedMessage());
                busyIndicator.hide();
            }

            @Override
            public void onSuccess(Boolean result) {
                documentsList.remove(index);
                busyIndicator.hide();
                removeRowDocuments();
                uiDocumentList.remove(0);
                if(uiDocumentList.size() > 0){
                    removeDocument(uiDocumentList);
                }
            }

        });
    }


    private boolean isAtLeastOneRecordSelected(){
        for(int i = 1; i < tableDocuments.getRowCount(); i++){
            if(((CheckBoxWidget)tableDocuments.getWidget(i, 0)).getValue()){
                return true;
            }
        }

        return false;
    }

    private void insertRowDocuments(){
        tableDocuments.insertRow(tableDocuments.getRowCount());
        tableDocuments.getRowFormatter().addStyleName(getRowDocuments(), ROW_STYLE);
        tableDocuments.getRowFormatter().addStyleName(getRowDocuments(), ROW_BODY_STYLE);
    }

    private void removeRowDocuments() {
        for(int i = 1; i < tableDocuments.getRowCount(); i++){
            if(((CheckBoxWidget)tableDocuments.getWidget(i, 0)).getValue()){
                tableDocuments.removeRow(i);
            }
        }
    }

    private int getRowDocuments(){
        return tableDocuments.getRowCount() - 1;
    }


    private void addHeading(String heading, FlexTableContainer table) {
        /* Add the column to the header row */
        int column = table.getCellCount(0);
        table.insertCell(0, column);

        /* Update the style of the header cells */
        table.getFlexCellFormatter().setStyleName(0, column, CELL_STYLE);
        table.getFlexCellFormatter().setStyleName(0, column, CELL_HEADER_STYLE);

        updateHeading(table, heading, column);
    }

    private void updateHeading(FlexTableContainer table, String heading, int column) {
        /* Create new label widget or update existing widget */
        LabelWidget label = getHeadingWidget(table, column);
        if (label == null) {
            /* Update the widget displayed as the column header */
            label = new LabelWidget(heading);
            label.addStyleName(TableConstants.STYLE_HEADER_NONSORTABLE_COL_TEXT);
            table.setWidget(0, column, label);
        } else {
            label.setText(heading);
        }
    }

    private void verifyPermissions(){
        if(ClientAuthorizationManager.isAuthorized(Permissions.CONTRACT_MODIFY) && recordKey != null){
            fileUpload.setEnabled( true );
            removeButton.setEnabled( true );
        }
    }

    private LabelWidget getHeadingWidget(FlexTableContainer table, int column) {
        Widget widget = table.getWidget(0, column);
        return widget instanceof LabelWidget ? (LabelWidget) widget : null;
    }


    public void enableButtonUpload(){
        uploadButton.setEnabled(true);
    }

    public void disableButtonUpload(){
        uploadButton.setEnabled(false);
    }

    public void enableButtonRemove(){
        removeButton.setEnabled(true);
    }

    public void disableButtonRemove(){
        removeButton.setEnabled(false);
    }

    public void addRow(UiDocument uiDocument){
        addRowDocuments(uiDocument);
    }

    /**
     * @param documentsList the documentsList to set
     */
    public void setDocumentsList(List<UiDocument> documentsList) {
        this.documentsList = documentsList;
    }

}
