package com.ericsson.cms.contractmgmt.client.tab.config;

import java.util.ArrayList;
import java.util.List;

import com.ericsson.cms.contractmgmt.client.ContractComponentConstants;
import com.ericsson.cms.contractmgmt.client.customfield.UiUserDefinedField;
import com.ericsson.cms.contractmgmt.client.model.UiKeyName;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerService;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerServiceAsync;
import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.ColumnBase;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.SortFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.View;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.label.LabelStringView;

/**
 * Data provider which operates on contracts.
 * 
 * @author Daniel Paranhos
 * 
 */
public class ConfigurationDataProvider implements DataProvider<Long, UiUserDefinedFieldRecord> {

    // columns for this table
    private final List<Column<?, UiUserDefinedFieldRecord>> columns;
    private final List<Column<?, UiUserDefinedFieldRecord>> sortColumns;
    private final ColumnBase<String, UiUserDefinedFieldRecord> orderColumn;
    private final ColumnBase<String, UiUserDefinedFieldRecord> nameColumn;
    private final ColumnBase<String, UiUserDefinedFieldRecord> typeColumn;
    private final ColumnBase<String, UiUserDefinedFieldRecord> requiredColumn;
    private final ColumnBase<String, UiUserDefinedFieldRecord> defaultValueColumn;
    private final ColumnBase<String, UiUserDefinedFieldRecord> additionalInfoColumn;

    private ContractComponentConstants constants = (ContractComponentConstants) GWT
            .create( ContractComponentConstants.class );

    private PageFeature parentPageFeature;
    private SortFeatureImpl<Long, UiUserDefinedFieldRecord> parentSortFeature;

    private final String screenName;
    private List<UiUserDefinedFieldRecord> userDefinedFieldRecords;

    // column names
    private static final String ORDER_COLUMN = "fieldOrder";
    private static final String NAME_COLUMN = "name";
    private static final String TYPE_COLUMN = "type";
    private static final String REQUIRED_COLUMN = "required";
    private static final String DEFAULT_COLUMN = "defaultValue";
    private static final String ADDITIONAL_INFO_COLUMN = "additionalInfo";

    // lists panel
    private IBusyNotifier busyNotifier;

    private IContractManagerServiceAsync contractManagerService = GWT.create( IContractManagerService.class );

    public ConfigurationDataProvider( String screenName ) {
        this.screenName = screenName;
        this.userDefinedFieldRecords = new ArrayList<UiUserDefinedFieldRecord>();

        // create the columns - id and name
        columns = new ArrayList<Column<?, UiUserDefinedFieldRecord>>();
        sortColumns = new ArrayList<Column<?, UiUserDefinedFieldRecord>>();

        orderColumn = new ColumnBase<String, UiUserDefinedFieldRecord>( ORDER_COLUMN,
                constants.configurationOrderColumn() ) {
            public View<String> getView( UiUserDefinedFieldRecord record ) {
                return new LabelStringView( getOrder( record.getUserDefinedField() ) );
            }

        };
        orderColumn.setCellStyle( TableConstants.STYLE_DATACELL_NUMERIC );
        columns.add( orderColumn );
        sortColumns.add(orderColumn);

        nameColumn = new ColumnBase<String, UiUserDefinedFieldRecord>( NAME_COLUMN,
                constants.configurationNameColumn() ) {
            public View<String> getView( UiUserDefinedFieldRecord record ) {
                return new LabelStringView( record.getUserDefinedField().getLabel() );
            }

        };
        nameColumn.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( nameColumn );
        sortColumns.add(nameColumn);

        typeColumn = new ColumnBase<String, UiUserDefinedFieldRecord>( TYPE_COLUMN,
                constants.configurationTypeColumn() ) {
            public View<String> getView( UiUserDefinedFieldRecord record ) {
                return new LabelStringView( getType( record.getUserDefinedField() ) );
            }

        };
        typeColumn.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( typeColumn );
        sortColumns.add(typeColumn);

        requiredColumn = new ColumnBase<String, UiUserDefinedFieldRecord>( REQUIRED_COLUMN,
                constants.configurationRequiredColumn() ) {
            public View<String> getView( UiUserDefinedFieldRecord record ) {
                return new LabelStringView( getRequired( record.getUserDefinedField() ) );
            }

        };
        requiredColumn.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( requiredColumn );
        sortColumns.add(requiredColumn);

        defaultValueColumn = new ColumnBase<String, UiUserDefinedFieldRecord>( DEFAULT_COLUMN,
                constants.configurationDefaultValueColumn() ) {
            public View<String> getView( UiUserDefinedFieldRecord record ) {
                return new LabelStringView( getDefaultValue( record.getUserDefinedField() ) );
            }

        };
        defaultValueColumn.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( defaultValueColumn );
        sortColumns.add(defaultValueColumn);

        additionalInfoColumn = new ColumnBase<String, UiUserDefinedFieldRecord>( ADDITIONAL_INFO_COLUMN,
                constants.configurationAdditionalInfoColumn() ) {
            public View<String> getView( UiUserDefinedFieldRecord record ) {
                return new LabelStringView( getAdditionalInfo( record.getUserDefinedField() ) );
            }
        };
        additionalInfoColumn.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( additionalInfoColumn );
    }

    private String getRequired( UiUserDefinedField userDefinedField ) {
        String required = constants.no();
        if ( userDefinedField.isRequired() ) {
            required = constants.yes();
        }

        return required;
    }

    private String getDefaultValue( UiUserDefinedField userDefinedField ) {
        String defaultValue = "";
        if ( userDefinedField.getDefaultValue() != null ) {
        	long key;
        	try {
        		key = Long.parseLong(userDefinedField.getDefaultValue());
    	    	switch (userDefinedField.getType()) {
    			case Alphanumeric:
    			case Text:
    				if (userDefinedField.isDropDown()) {
    					defaultValue = getDisplayableDefaultValue(userDefinedField, key);
    				} else {
    					defaultValue = userDefinedField.getDefaultValue();
    				}
    				break;
    			case Frequency:
    			case Currency:
    				defaultValue = getDisplayableDefaultValue(userDefinedField, key);
    				break;
    	
    			default:
    	            defaultValue = userDefinedField.getDefaultValue();
    				break;
    			}
        	} catch (NumberFormatException e) {
    			// Do I need to do anything?
        		// Just catch the exception and 
        		// return the value at the end.
                defaultValue = userDefinedField.getDefaultValue();
    		}
        }

        return defaultValue;
    }

	/**
	 * @param userDefinedField
	 * @param key
	 * @return 
	 */
	private String getDisplayableDefaultValue(UiUserDefinedField userDefinedField, long key) {
		for (UiKeyName fd : userDefinedField.getKeyNames()) {
			if (fd.getKey().longValue() == key) {
				return fd.getName();
			}
		}
		return userDefinedField.getDefaultValue();
	}

    private String getOrder( UiUserDefinedField userDefinedField ) {
        String order = "";
        if ( userDefinedField.getOrder() != null ) {
            order = userDefinedField.getOrder().toString();
        }

        return order;
    }

    private String getType( UiUserDefinedField userDefinedField ) {
        String type = "";
        switch ( userDefinedField.getType() ) {
            case Alphanumeric:
                type = constants.alphanumeric();
                break;
            case Currency:
                type = constants.currency();
                break;
            case Date:
                type = constants.date();
                break;
            case Frequency:
                type = constants.frequency();
                break;
            case Number:
                type = constants.number();
                break;
            case Percentage:
                type = constants.percentage();
                break;
            case Text:
                type = constants.text();
                break;
        }

        return type;
    }

    private String getAdditionalInfo( UiUserDefinedField userDefinedField ) {
        StringBuilder additionalInfo = new StringBuilder();
        if ( userDefinedField.isDropDown() ) {
            additionalInfo.append( constants.dropDown() );
            additionalInfo.append( ";" );
        }
        if ( ( userDefinedField.getMinChars() != null ) && ( userDefinedField.getMinChars() > 0 ) ) {
            addInfo( userDefinedField, additionalInfo, constants.minChars(), userDefinedField.getMinChars()
                    .toString() );
        }
        if ( ( userDefinedField.getMaxChars() != null ) && ( userDefinedField.getMaxChars() > 0 ) ) {
            addInfo( userDefinedField, additionalInfo, constants.maxChars(), userDefinedField.getMaxChars()
                    .toString() );
        }

        return additionalInfo.toString();
    }

    private void addInfo( UiUserDefinedField userDefinedField, StringBuilder additionalInfo, String label,
            String value ) {
        additionalInfo.append( " " );
        additionalInfo.append( label );
        additionalInfo.append( ":" );
        additionalInfo.append( value );
        additionalInfo.append( ";" );
    }

    public ColumnBase<String, UiUserDefinedFieldRecord> getDefaultSortColumn() {
        return orderColumn;
    }

    public List<Column<?, UiUserDefinedFieldRecord>> getSortColumns() {
        return sortColumns;
    }

    @Override
    public List<Column<?, UiUserDefinedFieldRecord>> getColumns() {
        return columns;
    }

    @Override
    public void getRecord( Long key, final AsyncCallback<UiUserDefinedFieldRecord> callback ) {
        for ( UiUserDefinedFieldRecord record : userDefinedFieldRecords ) {
            if ( record.getKey().equals( key ) ) {
                callback.onSuccess( record );
                break;
            }
        }
    }

    @Override
    public void getRecords( final AsyncCallback<List<UiUserDefinedFieldRecord>> callback ) {
        notifyBusyStart();
        contractManagerService.getUserDefinedFields( parentSortFeature.getSortColumnName(),
                parentSortFeature.isAscending(), screenName,
                new NeptuneAsyncCallback<List<UiUserDefinedField>>() {

                    @Override
                    public void onNeptuneFailure( Throwable caught ) {
                        callback.onFailure( caught );
                    }

                    @Override
                    public void onNeptuneSuccess( List<UiUserDefinedField> result ) {
                        userDefinedFieldRecords.clear();
                        for ( UiUserDefinedField uiUserDefinedField : result ) {
                            userDefinedFieldRecords.add( new UiUserDefinedFieldRecord( uiUserDefinedField ) );
                        }
                        callback.onSuccess( userDefinedFieldRecords );
                        notifyBusyFinish();
                    }

                } );
    }

    @Override
    public void initialize( AsyncCallback<Void> callback ) {
        callback.onSuccess( null );
    }

    @Override
    public boolean isCheckboxEnabled() {
        return true;
    }

    @Override
    public boolean isRecordCountEnabled() {
        return true;
    }

    /**
     * @return the parentPageFeature
     */
    public PageFeature getParentPageFeature() {
        return parentPageFeature;
    }

    /**
     * @param parentPageFeature
     *            the parentPageFeature to set
     */
    public void setParentPageFeature( PageFeature parentPageFeature ) {
        this.parentPageFeature = parentPageFeature;
    }

    /**
     * @return the parentSortFeature
     */
    public SortFeatureImpl<Long, UiUserDefinedFieldRecord> getParentSortFeature() {
        return parentSortFeature;
    }

    /**
     * @param parentSortFeature
     *            the parentSortFeature to set
     */
    public void setParentSortFeature( SortFeatureImpl<Long, UiUserDefinedFieldRecord> parentSortFeature ) {
        this.parentSortFeature = parentSortFeature;
    }

    public static interface IBusyNotifier {
        void startingToBeBusy();

        void doneBeingBusy();
    }

    public void setBusyNotifier( IBusyNotifier busyNotifier ) {
        this.busyNotifier = busyNotifier;
    }

    private void notifyBusyStart() {
        if ( busyNotifier != null ) {
            busyNotifier.startingToBeBusy();
        }
    }

    private void notifyBusyFinish() {
        if ( busyNotifier != null ) {
            busyNotifier.doneBeingBusy();
        }
    }

}
