/**
 * 
 */
package com.ericsson.cms.contractmgmt.client.tab.config;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import com.ericsson.cms.contractmgmt.client.ContractComponentConstants;
import com.ericsson.cms.contractmgmt.client.ContractComponentMessages;
import com.ericsson.cms.contractmgmt.client.customfield.AlphaNumericValidator;
import com.ericsson.cms.contractmgmt.client.customfield.FieldTypeEnum;
import com.ericsson.cms.contractmgmt.client.model.UiKeyName;
import com.google.gwt.core.client.GWT;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.DataTypeWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.TypedDateBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.TypedListBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.TypedTextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.DateConverter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.IntegerConverter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.StringConverter;

/**
 * @author amit.r.seth
 *
 */
public class DefaultValueWidgetFactory {

    private final ContractComponentConstants constants = (ContractComponentConstants) GWT
            .create( ContractComponentConstants.class );

    private final ContractComponentMessages messages = (ContractComponentMessages) GWT
            .create( ContractComponentMessages.class );

    public TypedTextBoxWidget<?> createTextField( String defaultValue, Boolean isDropDown, Integer maxChars, FieldTypeEnum fieldType ) {
        TypedTextBoxWidget<?> textBox = createTypedTextBox( fieldType );

        if ((isDropDown == null || !isDropDown) && maxChars != null && maxChars > 0) {
            textBox.setMaxLength(maxChars);
        }

        if ( defaultValue != null && !defaultValue.trim().equals( "" ) ) {
            textBox.setTextValue( defaultValue );
        }

        return textBox;
    }

    private TypedTextBoxWidget<?> createTypedTextBox( FieldTypeEnum fieldType ) {
        if ( FieldTypeEnum.Number.equals( fieldType ) ) {
            TypedTextBoxWidget<Long> textBoxLong = new TypedTextBoxWidget<Long>( new IntegerConverter() );
            textBoxLong.setErrorToolTip( messages.numberValidationErrorTooltip() );

            return textBoxLong;
        } else if ( FieldTypeEnum.Percentage.equals( fieldType ) ) {
            TypedTextBoxWidget<String> textBoxString = new TypedTextBoxWidget<String>( new PercentageConverter() );
            textBoxString.setErrorToolTip( messages.positiveFloatValidationErrorTooltip() );

            return textBoxString;
        } else {
            TypedTextBoxWidget<String> textBoxString = new TypedTextBoxWidget<String>( new StringConverter() );
            if ( FieldTypeEnum.Alphanumeric.equals( fieldType ) ) {
                textBoxString = new TypedTextBoxWidget<String>( new AlphaNumericValidator() );
                textBoxString.setErrorToolTip( messages.alphaNumericValidationErrorTooltip() );
            }
            return textBoxString;
        }
    }

    public TypedDateBoxWidget createDateField( String defaultValue ) {
        TypedDateBoxWidget dateBox = new TypedDateBoxWidget( new DateConverter( constants.dateFormat(),
                constants.dateFormat() ) );

        dateBox.getTextBox().setMaxLength( constants.dateFormat().length() );
        dateBox.getTextBox().setVisibleLength( constants.dateFormat().length() );
        dateBox.setToolTip( constants.validDateTitle() + " " + constants.dateFormat() );
        dateBox.setErrorToolTip( constants.validDateTitle() + " " + constants.dateFormat() );

        if ( defaultValue != null ) {
            dateBox.setTextValue( defaultValue.trim() );
        }

        return dateBox;
    }

    public TypedListBoxWidget<String> createListBoxField( List<UiKeyName> uiKeyNamesList, String defaultValue ) {
        TypedListBoxWidget<String> listBox = new TypedListBoxWidget<String>( new StringConverter() );
        Collections.sort( uiKeyNamesList, new Comparator<UiKeyName>() {

            @Override
            public int compare( UiKeyName ukn1, UiKeyName ukn2 ) {
                return ukn1.getName().compareTo( ukn2.getName() );
            }
        } );

        listBox.addItem( "", null );
        for ( UiKeyName keyName : uiKeyNamesList ) {
            listBox.addItem( keyName.getName(), keyName.getKey().toString() );
        }

        if ( defaultValue != null && !defaultValue.trim().equals( "" ) ) {
            listBox.setSelectedItem( defaultValue );
        }

        return listBox;
    }

    public DataTypeWidget<?> createDefaultValueWidget(FieldTypeEnum fieldType, Boolean isDropDown, List<UiKeyName> uiKeyNamesList, String defaultValue, Integer maxChars) {
        DataTypeWidget<?> widget = null;
        switch (fieldType) {
        case Currency:
        case Frequency:
            widget = createListBoxField(uiKeyNamesList, defaultValue);
            break;
        case Date:
            widget = createDateField(defaultValue);
            break;
        case Number:
        case Percentage:
            widget = createTextField(defaultValue, isDropDown, maxChars, fieldType);
            break;
        case Text:
        case Alphanumeric:
        default:
            if (isDropDown != null && isDropDown) {
                widget = createListBoxField(uiKeyNamesList, defaultValue);
            } else {
                widget = createTextField(defaultValue, isDropDown, maxChars, fieldType);
            }
            break;
        }

        return widget;
    }
}
