package com.ericsson.cms.contractmgmt.client.tab.config;

import java.util.ArrayList;
import java.util.List;

import com.ericsson.cms.contractmgmt.client.ContractComponentConstants;
import com.ericsson.cms.contractmgmt.client.ContractComponentMessages;
import com.ericsson.cms.contractmgmt.client.Permissions;
import com.ericsson.cms.contractmgmt.client.customfield.UiUserDefinedField;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerService;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerServiceAsync;
import com.ericsson.cms.contractmgmt.client.tab.UiLongKeySerializer;
import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.application.ValidationException;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailViewCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.AbstractDetailFeature;

/**
 * Handles detail view for a given contract.
 * 
 * @author Daniel Paranhos
 * 
 */
public class UserDefinedFieldDetailFeature extends AbstractDetailFeature<Long, UiUserDefinedFieldRecord> {

    private IContractManagerServiceAsync contractManagerService = GWT.create( IContractManagerService.class );
    private final String screenName;

    private ContractComponentConstants constants = (ContractComponentConstants) GWT
            .create( ContractComponentConstants.class );

    private ContractComponentMessages messages = (ContractComponentMessages) GWT
            .create(ContractComponentMessages.class);
    
    private UserDefinedFieldListPanel parentPanel;

    public UserDefinedFieldDetailFeature( UserDefinedFieldListPanel parentPanel, String screenName ) {
        super( new UiLongKeySerializer() );
        this.screenName = screenName;
        this.parentPanel = parentPanel;
    }

    @Override
    public void delete( List<UiUserDefinedFieldRecord> records, final AsyncCallback<Void> callback ) {
        if ( records.size() > 0 ) {
            if ( Window.confirm( messages.deleteFieldConfirmation() ) ) {
                List<UiUserDefinedField> fields = new ArrayList<UiUserDefinedField>();
                for (UiUserDefinedFieldRecord record: records) {
                    fields.add(record.getUserDefinedField());
                }
                parentPanel.clearMessages();
                contractManagerService.deleteUserDefinedFields( fields, new NeptuneAsyncCallback<Void>() {

                    @Override
                    public void onNeptuneFailure( Throwable caught ) {
                        parentPanel.getUserDefinedFieldsTable().refresh();
                        parentPanel.getErrorPanel().showMessage( messages.deleteFieldError() );
                        if (caught instanceof ValidationException ) {
                            ValidationException ve = (ValidationException) caught;
                            for ( String message : ve.getValidationMessages() ) {
                                parentPanel.getErrorPanel().addMessage(message);
                            }
                        } else {
                            callback.onFailure(caught);
                        }
                    }

                    @Override
                    public void onNeptuneSuccess( Void result ) {
                        parentPanel.getSuccessPanel().showMessage( messages.deleteFieldSucceeded() );
                        callback.onSuccess( null );
                    }
                } );
            }
        } else {
            Window.alert( messages.noFieldSelected() );
        }
    }

    @Override
    public void getNew( AsyncCallback<UiUserDefinedFieldRecord> callback ) {
        this.parentPanel.clearMessages();
        callback.onSuccess( new UiUserDefinedFieldRecord( new UiUserDefinedField( this.screenName ) ) );
    }

    @Override
    public DetailView<Long, UiUserDefinedFieldRecord> getView( UiUserDefinedFieldRecord record,
            DetailViewCallback<Long, UiUserDefinedFieldRecord> callback ) {
        this.parentPanel.clearMessages();
        UserDefinedFieldDetailView view = new UserDefinedFieldDetailView( this.parentPanel, record, callback );
        return view;
    }

    /*
     * The detail link is shown only if the user has asset view permissions and the column is the name column.
     */
    @Override
    public boolean hasDetailLink( Column<?, UiUserDefinedFieldRecord> column ) {
        return column.getDisplayName().equals( constants.contractNameColumn() )
                && ClientAuthorizationManager.isAuthorized( Permissions.CONTRACT_VIEW );
    }

    @Override
    public void save( UiUserDefinedFieldRecord record, final AsyncCallback<Void> callback ) {
        parentPanel.clearMessages();
        contractManagerService.saveUserDefinedField(record.getUserDefinedField(),
                new NeptuneAsyncCallback<Void>() {

                    @Override
                    public void onNeptuneFailure(Throwable caught) {
                        parentPanel.getErrorPanel().showMessage( messages.saveFieldError() );
                        if (caught instanceof ValidationException ) {
                            ValidationException ve = (ValidationException) caught;
                            for ( String message : ve.getValidationMessages() ) {
                                parentPanel.getErrorPanel().addMessage(message);
                            }
                        } else {
                            callback.onFailure(caught);
                        }
                    }

                    @Override
                    public void onNeptuneSuccess(Void result) {
                        parentPanel.getSuccessPanel().showMessage( messages.saveFieldSucceeded() );
                        callback.onSuccess(null);
                    }
                });
    }

    @Override
    public boolean showCreateButton() {
        return ClientAuthorizationManager.isAuthorized( Permissions.CONTRACT_CREATE );
    }

    @Override
    public boolean showDeleteButton() {
        return ClientAuthorizationManager.isAuthorized( Permissions.CONTRACT_DELETE );
    }
}
