package com.ericsson.cms.contractmgmt.client.tab.config.dropdown;

import com.ericsson.cms.contractmgmt.client.ContractComponentConstants;
import com.ericsson.cms.contractmgmt.client.customfield.FieldTypeEnum;
import com.ericsson.cms.contractmgmt.client.model.UiKeyNameRecord;
import com.ericsson.cms.contractmgmt.client.tab.ErrorPanel;
import com.ericsson.cms.contractmgmt.client.tab.SuccessPanel;
import com.ericsson.cms.contractmgmt.client.tab.config.dropdown.DropDownDataProvider.IBusyNotifier;
import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.ui.Composite;
import com.tandbergtv.neptune.widgettoolkit.client.menu.WidgetMenuItem.AnchorChangeListener;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.HeaderPanel;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.SortOrder;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Table;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableDetailViewEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableViewChangeHandler;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableViewEvent;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.PageFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.SortFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

/**
 * Table that shows the list of drop-down values.
 * 
 * @author Daniel Paranhos
 * 
 */
public abstract class DropDownListPanel<Record extends UiKeyNameRecord> extends Composite {

    // table related
    private DropDownDataProvider<Record> dataProvider;
    private DropDownDetailFeature<Record> detailFeature;
    private AnchorChangeListener anchorChangeListener;
    private boolean hasPageFeature;
    private boolean hasSortFeature;

    // widgets used in this view
    private VerticalContainer mainContainer;
    private Table<Long, Record> table;
    private BusyIndicator busyIndicator = new BusyIndicator();

    private HeaderPanel header;
    private final SuccessPanel successPanel = new SuccessPanel();
    private final ErrorPanel errorPanel = new ErrorPanel();

    protected ContractComponentConstants constants = (ContractComponentConstants) GWT
            .create( ContractComponentConstants.class );
    protected Long userDefinedFieldId;
    private boolean editing = false;
    protected FieldTypeEnum type = FieldTypeEnum.Text;

    // spacing between the table and message panel
    private static final int SPACING = 2;
    private static final String STYLE_NAME = "content-ContractPanel";

    public DropDownListPanel( boolean hasPageFeature, boolean hasSortFeature ) {
        this.hasPageFeature = hasPageFeature;
        this.hasSortFeature = hasSortFeature; 
        init();
    }

    /*
     * Introduced the Constructor for the CustomListPanel in order to initialize the userDefinedFieldId
     */
    /**
     * @param hasPageFeature
     * @param hasSortFeature
     * @param userDefinedFieldId
     */
    public DropDownListPanel( boolean hasPageFeature, boolean hasSortFeature, Long userDefinedFieldId, FieldTypeEnum type ) {
        this.hasPageFeature = hasPageFeature;
        this.hasSortFeature = hasSortFeature; 
        this.userDefinedFieldId = userDefinedFieldId;
        this.editing = userDefinedFieldId != null ? true : false;
    	if (type != null) {
    		this.type = type;
    	}
        init();
    }

    private void init() {
        this.dataProvider = getDataProvider();
        dataProvider.setBusyNotifier( new IBusyNotifier() {
            @Override
            public void startingToBeBusy() {
                busyIndicator.center();
            }

            @Override
            public void doneBeingBusy() {
                busyIndicator.hide();
            }
        } );
        initWidgets();
        setContainers();
    }

    public SuccessPanel getSuccessPanel() {
        return successPanel;
    }

    public ErrorPanel getErrorPanel() {
        return errorPanel;
    }

    private void setContainers() {
        mainContainer.add( header );
        mainContainer.add(successPanel);
        mainContainer.add(errorPanel);
        mainContainer.add( table );
    }

    private void initWidgets() {
        mainContainer = new VerticalContainer();
        mainContainer.setSpacing( SPACING );
        mainContainer.setStylePrimaryName( STYLE_NAME );
        this.initWidget( mainContainer );

        header = new HeaderPanel( "" );

        table = new Table<Long, Record>( dataProvider );
        table.addTableViewChangeHandler( new TableViewChangeHandler<Long, Record>() {
            @Override
            public void onShowDetailView( TableDetailViewEvent<Long, Record> event ) {
                if ( event.isCreateView() ) {
                    header.setLabel( getCreateHeading() );
                } else {
                    header.setLabel( getEditHeading() + " " + event.getRecord().getKeyName().getName() );
                }
            }

            @Override
            public void onShowListView( TableViewEvent event ) {
                header.setLabel( getListHeading() );
            }
        } );

        // detail feature
        detailFeature = getDropDownDetailFeature();

        if ( hasPageFeature ) {
            PageFeature pageFeature = new PageFeatureImpl();
            dataProvider.setParentPageFeature( pageFeature );
            table.addPageFeature( pageFeature );
        }

        if ( hasSortFeature ) {
            // sort feature
            SortFeatureImpl<Long, Record> sortFeature = new SortFeatureImpl<Long, Record>(
                    dataProvider.getDefaultSortColumn(), SortOrder.ASCENDING );
            dataProvider.setParentSortFeature( sortFeature );
            for ( Column<?, Record> col : dataProvider.getSortColumns() ) {
                sortFeature.addSortableColumn( col );
            }
            dataProvider.setParentSortFeature( sortFeature );
        }

        table.addDetailFeature( detailFeature );

        table.initialize( new NeptuneAsyncCallback<Void>() {
            @Override
            public void onNeptuneFailure( Throwable caught ) {
            }

            @Override
            public void onNeptuneSuccess( Void result ) {
                table.refresh();
            }
        } );

        anchorChangeListener = new DropDownAnchorChangeListener( this );
    }

    protected abstract DropDownDataProvider<Record> getDataProvider();

    protected abstract DropDownDetailFeature<Record> getDropDownDetailFeature();

    public String getCreateHeading() {
        return constants.dropDownCreateHeading();
    }

    public String getEditHeading() {
        return constants.dropDownViewEditHeading();
    }

    public String getListHeading() {
        return constants.dropDownListHeading();
    }

    public Table<Long, Record> getTable() {
        return this.table;
    }

    public AnchorChangeListener getAnchorChangeListener() {
        return this.anchorChangeListener;
    }

    /**
     * @param reverseAnchorChangeListener
     *            the reverseAnchorChangeListener to set
     */
    public void setReverseAnchorChangeListener( AnchorChangeListener reverseAnchorChangeListener ) {
        table.setAnchorChangeNotifier( reverseAnchorChangeListener );
    }

    /**
     * @return the editing
     */
    public boolean isEditing() {
        return editing;
    }

    public void clearMessages() {
        this.successPanel.reset();
        this.errorPanel.reset();
    }
}
