package com.ericsson.cms.contractmgmt.client.tab.contacts;

import java.util.ArrayList;
import java.util.List;

import com.ericsson.cms.contractmgmt.client.ContractComponentConstants;
import com.ericsson.cms.contractmgmt.client.model.ContactSearchResult;
import com.ericsson.cms.contractmgmt.client.model.UiContact;
import com.ericsson.cms.contractmgmt.client.model.UiContactType;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerService;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerServiceAsync;
import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.CheckBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.ColumnBase;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.SortFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.View;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.label.LabelStringView;

/**
 * Data provider which operates on copyright holders.
 * 
 * @author Daniel Paranhos
 * 
 */
public class ContactDataProvider implements DataProvider<Long, UiContactRecord> {

    // columns for this table
    private final List<Column<?, UiContactRecord>> columns;
    private final ColumnBase<String, UiContactRecord> nameColumn;
    private final ColumnBase<String, UiContactRecord> email;
    private final ColumnBase<String, UiContactRecord> office;
    private final ColumnBase<String, UiContactRecord> work;
    private final ColumnBase<String, UiContactRecord> mobile;
    private final ColumnBase<String, UiContactRecord> notify;
    
    private final IContractManagerServiceAsync contractManagerService = GWT.create( IContractManagerService.class );


    private final ContractComponentConstants constants = (ContractComponentConstants) GWT
            .create( ContractComponentConstants.class );

    private PageFeature parentPageFeature;
    private SortFeatureImpl<Long, UiContactRecord> parentSortFeature;

    // column names
    private static final String NAME_COLUMN = "name";
    private static final String EMAIL_COLUMN = "email";
    private static final String PHONE_OFFICE_COLUMN = "office";
    private static final String PHONE_WORK_COLUMN = "work";
    private static final String PHONE_MOBILE_COLUMN = "mobile";
    private static final String NOTIFY_COLUMN = "notify";

    // lists panel
    private ContactListPanel parentPanel;
    private IBusyNotifier busyNotifier;

    public ContactDataProvider( ContactListPanel listsPanel ) {
        this.parentPanel = listsPanel;

        // create the columns - id and name
        columns = new ArrayList<Column<?, UiContactRecord>>();
    

        nameColumn = new ColumnBase<String, UiContactRecord>( NAME_COLUMN,
                constants.copyrightHolderNameColumn() ) {
            @Override
            public View<String> getView( UiContactRecord record ) {
                return new LabelStringView( record.getContact().getName() );
            }
        };
        
        nameColumn.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( nameColumn );
        
        
        email = new ColumnBase<String, UiContactRecord>( EMAIL_COLUMN,
                constants.copyrightHolderEmail() ) {
            @Override
            public View<String> getView( UiContactRecord record ) {
                return new LabelStringView( record.getContact().getEmail() );
            }
        };
        
        email.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( email );
       
        
        office = new ColumnBase<String, UiContactRecord>( PHONE_OFFICE_COLUMN,
                constants.copyrightHolderPhoneOffice() ) {
            @Override
            public View<String> getView( UiContactRecord record ) {
                return new LabelStringView( record.getContact().getPhoneOffice() );
            }
        };
        
        office.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( office );
        
        work = new ColumnBase<String, UiContactRecord>( PHONE_WORK_COLUMN,
                constants.copyrightHolderPhoneWork() ) {
            @Override
            public View<String> getView( UiContactRecord record ) {
                return new LabelStringView( record.getContact().getPhoneWork() );
            }
        };
        
        work.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( work );
        
        mobile = new ColumnBase<String, UiContactRecord>( PHONE_MOBILE_COLUMN,
                constants.copyrightHolderPhoneMobile() ) {
            @Override
            public View<String> getView( UiContactRecord record ) {
                return new LabelStringView( record.getContact().getPhoneMobile() );
            }
        };
        
        mobile.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( mobile );
        
        
        notify = new ColumnBase<String, UiContactRecord>( NOTIFY_COLUMN,
                constants.copyrightHolderNotify() ) {
            @Override
            public View<String> getView( final UiContactRecord record ) {
            	return new View<String>(){

					@Override
					public String getStyleName() {
						// TODO Auto-generated method stub
						return null;
					}

					@Override
					public Widget getWidget() {
						CheckBoxWidget check = new CheckBoxWidget();

						if (record.getContact().isNotify()) {
							check.setValue(true);
						} else {
							check.setValue(false);
						}
						check.setEnabled(false);
						return check;
					}

					@Override
					public void release() {
						// TODO Auto-generated method stub
						
					}};
            }
        };
        
        notify.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( notify );
        
    }

    public ColumnBase<String, UiContactRecord> getDefaultSortColumn() {
        return nameColumn;
    }

    public List<Column<?, UiContactRecord>> getSortColumns() {
        return columns;
    }

    @Override
    public List<Column<?, UiContactRecord>> getColumns() {
        return columns;
    }

    @Override
    public void getRecord( Long key, final AsyncCallback<UiContactRecord> callback ) {
    	if (parentPanel.isEditing) {
			notifyBusyStart();
				
			contractManagerService.getContact(key, new NeptuneAsyncCallback<UiContact>() {

				@Override
				public void onNeptuneFailure(Throwable caught) {
					notifyBusyFinish();
				}

				@Override
				public void onNeptuneSuccess(UiContact result) {
					callback.onSuccess(new UiContactRecord(result));
					notifyBusyFinish();
				}
			});
		} 
    }
    
	private List<UiContactRecord> prepareUIObjects(List<UiContact> result) {
		List<UiContactRecord> records = new ArrayList<UiContactRecord>();
		for (UiContact contact : result) {
			records.add(new UiContactRecord(contact));
		}
		return records;
	}
	
    @Override
    public void getRecords( final AsyncCallback<List<UiContactRecord>> callback ) {
        
		if (parentPanel.isEditing) {
			notifyBusyStart();
			if(parentPanel.getContactType().equals(UiContactType.COPYRIGHT_HOLDER)){
				
				contractManagerService.getCopyrightHolderContactList(parentPanel.getParentId(),getParentSortFeature().getSortColumnName(), 
		                getParentSortFeature().isAscending(), new NeptuneAsyncCallback<ContactSearchResult>() {
					
					@Override
					public void onNeptuneFailure(Throwable caught) {
					    callback.onFailure(caught);
						notifyBusyFinish();
					}
					
					@Override
					public void onNeptuneSuccess(ContactSearchResult result) {
					    parentPanel.getUiContactList().clear();
						parentPanel.getUiContactList().addAll(result.getUiContactList());
						callback.onSuccess(prepareUIObjects(new ArrayList<UiContact>(result.getUiContactList())));
						notifyBusyFinish();
					}
				});
			} else if (parentPanel.getContactType().equals(UiContactType.CONTRACT)){
				contractManagerService.getContractContactList(parentPanel.getParentId(),getParentSortFeature().getSortColumnName(), 
                        getParentSortFeature().isAscending(), new NeptuneAsyncCallback<ContactSearchResult>() {
					
					@Override
					public void onNeptuneFailure(Throwable caught) {
						notifyBusyFinish();
					}
					
					@Override
					public void onNeptuneSuccess(ContactSearchResult result) {
					    parentPanel.getUiContactList().clear();
					    parentPanel.getUiContactList().addAll(result.getUiContactList());
						callback.onSuccess(prepareUIObjects(new ArrayList<UiContact>(result.getUiContactList())));
						notifyBusyFinish();
					}
				});
			} else if(parentPanel.getContactType().equals(UiContactType.DEAL)){
				contractManagerService.getDealContactList(parentPanel.getParentId(),getParentSortFeature().getSortColumnName(), 
                        getParentSortFeature().isAscending(), new NeptuneAsyncCallback<ContactSearchResult>() {
					
					@Override
					public void onNeptuneFailure(Throwable caught) {
						notifyBusyFinish();
					}
					
					@Override
					public void onNeptuneSuccess(ContactSearchResult result) {
					    parentPanel.getUiContactList().clear();
                        parentPanel.getUiContactList().addAll(result.getUiContactList());
						callback.onSuccess(prepareUIObjects(new ArrayList<UiContact>(result.getUiContactList())));
						notifyBusyFinish();
					}
				});
			}
		} else {
			callback.onSuccess(prepareUIObjects(parentPanel.getUiContactList()));
		}
    }
    
    @Override
    public void initialize( AsyncCallback<Void> callback ) {
        callback.onSuccess( null );
    }

    @Override
    public boolean isCheckboxEnabled() {
        return true;
    }

    @Override
    public boolean isRecordCountEnabled() {
        return false;
    }

    /**
     * @return the parentPageFeature
     */
    public PageFeature getParentPageFeature() {
        return parentPageFeature;
    }

    /**
     * @param parentPageFeature
     *            the parentPageFeature to set
     */
    public void setParentPageFeature( PageFeature parentPageFeature ) {
        this.parentPageFeature = parentPageFeature;
    }

    /**
     * @return the parentSortFeature
     */
    public SortFeatureImpl<Long, UiContactRecord> getParentSortFeature() {
        return parentSortFeature;
    }

    /**
     * @param parentSortFeature
     *            the parentSortFeature to set
     */
    public void setParentSortFeature( SortFeatureImpl<Long, UiContactRecord> parentSortFeature ) {
        this.parentSortFeature = parentSortFeature;
    }

    public static interface IBusyNotifier {
        void startingToBeBusy();

        void doneBeingBusy();
    }

    public void setBusyNotifier( IBusyNotifier busyNotifier ) {
        this.busyNotifier = busyNotifier;
    }

    private void notifyBusyStart() {
        if ( busyNotifier != null ) {
        	busyNotifier.startingToBeBusy();
        }
    }

    private void notifyBusyFinish() {
        if ( busyNotifier != null ) {
        	busyNotifier.doneBeingBusy();
        }
    }
	
}
