package com.ericsson.cms.contractmgmt.client.tab.contacts;

import com.ericsson.cms.contractmgmt.client.ContractComponentConstants;
import com.ericsson.cms.contractmgmt.client.Permissions;
import com.ericsson.cms.contractmgmt.client.model.UiContact;
import com.ericsson.cms.contractmgmt.client.tab.ContractFormContainer;
import com.ericsson.cms.contractmgmt.client.tab.ErrorPanel;
import com.ericsson.cms.contractmgmt.client.tab.SuccessPanel;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HasHorizontalAlignment;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.application.ValidationException;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.CheckBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailViewCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.TypedTextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.StringConverter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

public class ContactDetailView extends Composite implements DetailView<Long, UiContactRecord> {

    /** Styles **/

    private static final int CONTACT_NAME_MAX_LENGTH = 50;
    private static final int CONTACT_EMAIL_MAX_LENGTH = 150;
    private static final int CONTACT_PHONE_MAX_LENGTH = 20;
    private static final int CONTACT_PHONE_MIN_LENGTH = 1;

    private final ContractComponentConstants constants = (ContractComponentConstants) GWT
            .create(ContractComponentConstants.class);

    private final UiContactRecord record;
    private final DetailViewCallback<Long, UiContactRecord> callback;
    private final VerticalContainer mainContainer;
    private ContractFormContainer formContainer;
    private final ContactListPanel parentPanel;

    private TypedTextBoxWidget<String> name;
    private TypedTextBoxWidget<String> email;
    private TypedTextBoxWidget<String> phoneOffice;
    private TypedTextBoxWidget<String> phoneWork;
    private TypedTextBoxWidget<String> phoneMobile;
    private CheckBoxWidget notify;

    private ButtonWidget saveButton;
    private ButtonWidget cancelButton;

    private final HorizontalContainer buttonsContainer;

    private final boolean isEditing;

    private final SuccessPanel successPanel = new SuccessPanel();
    private final ErrorPanel errorPanel = new ErrorPanel();

    public ContactDetailView(UiContactRecord record, boolean isEditing, ContactListPanel parentPanel, DetailViewCallback<Long, UiContactRecord> callback) {
        this.record = record;
        this.callback = callback;
        this.isEditing = isEditing;
        this.parentPanel = parentPanel;

        mainContainer = new VerticalContainer();
        mainContainer.add(successPanel);
        mainContainer.add(errorPanel);

        initWidget(mainContainer);
        init();
        mainContainer.add(formContainer);
        buttonsContainer = new HorizontalContainer();
        buttonsContainer.setSpacing(5);
        mainContainer.add(buttonsContainer);
        addButtons();

        if(record.getKey() != null){
            fillForm();
        } else if(record.getContact() != null && record.getContact().getName() != null && record.getContact().getEmail() != null){
            if(parentPanel.getUiContactList().contains(record.getContact())){
                parentPanel.getUiContactList().remove(record.getContact());
            }
            fillForm();
        }
        verifyPermissions();
    }

    private void fillForm() {
        name.setValue(record.getContact().getName());
        email.setValue(record.getContact().getEmail());
        phoneOffice.setValue(record.getContact().getPhoneOffice());
        phoneWork.setValue(record.getContact().getPhoneWork());
        phoneMobile.setValue(record.getContact().getPhoneMobile());
        notify.setValue(record.getContact().isNotify());
    }

    private void init() {
        buildInfo();
    }

    private void buildInfo() {

        formContainer = new ContractFormContainer(HasHorizontalAlignment.ALIGN_LEFT);
        notify = new CheckBoxWidget();
        name = new TypedTextBoxWidget<String>(new StringConverter());
        name.setMaxLength(CONTACT_NAME_MAX_LENGTH);
        email = new TypedTextBoxWidget<String>(new StringConverter());
        email.setMaxLength(CONTACT_EMAIL_MAX_LENGTH);
        phoneOffice = new TypedTextBoxWidget<String>(new StringConverter());
        phoneOffice.setMaxLength(CONTACT_PHONE_MAX_LENGTH);
        phoneWork = new TypedTextBoxWidget<String>(new StringConverter());
        phoneWork.setMaxLength(CONTACT_PHONE_MAX_LENGTH);
        phoneMobile = new TypedTextBoxWidget<String>(new StringConverter());
        phoneMobile.setMaxLength(CONTACT_PHONE_MAX_LENGTH);


        formContainer.addRow(constants.copyrightHolderNotify(), notify);
        formContainer.addRow(constants.copyrightHolderName(), name, true);
        formContainer.addRow(constants.copyrightHolderEmail(), email, true);
        formContainer.addRow(constants.copyrightHolderPhoneOffice(), phoneOffice);
        formContainer.addRow(constants.copyrightHolderPhoneWork(), phoneWork);
        formContainer.addRow(constants.copyrightHolderPhoneMobile(), phoneMobile);
    }

    private void addButtons() {

        saveButton = new ButtonWidget(constants.saveButtonName(), new ClickHandler() {

            @Override
            public void onClick(ClickEvent event) {
                if (validate()) {
                    commit();
                    ContactDetailView.this.callback.save(ContactDetailView.this);
                }
            }

        });
        saveButton.addStyleDependentName(StyleNames.COMMIT_BUTTON_STYLE);
        saveButton.setEnabled(false);
        buttonsContainer.add(saveButton);

        cancelButton = new ButtonWidget(constants.cancelButtonName(), new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {
                errorPanel.reset();
                if(record != null && record.getContact() != null){
                    if(record.getContact().getEmail() != null && !record.getContact().getEmail().isEmpty()){
                        parentPanel.getUiContactList().add(record.getContact());
                    }
                }

                callback.cancel(ContactDetailView.this);
            }
        });
        cancelButton.addStyleDependentName(StyleNames.ACTION_BUTTON_STYLE);
        buttonsContainer.add(cancelButton);

    }

    private boolean validate() {
        errorPanel.reset();
        boolean valid = formContainer.validate();

        if (email.getTextValue() != null
                && (!email.getTextValue().contains("@") || !email.getTextValue().contains("."))) {
            errorPanel.addHeading1("Invalid email");
            valid = false;
        }

        if(phoneMobile.getTextValue() != null && !phoneMobile.getTextValue().isEmpty()){
            if(phoneMobile.getTextValue().length() < CONTACT_PHONE_MIN_LENGTH || phoneMobile.getTextValue().length() > CONTACT_PHONE_MAX_LENGTH){
                errorPanel.addHeading1("Invalid mobile number");
                valid = false;
            } else if(valid){
                valid = validatePhone(phoneMobile.getTextValue());
                if(!valid){
                    errorPanel.addHeading1("Invalid mobile number");
                }
            }
        }

        if(phoneOffice.getTextValue() != null && !phoneOffice.getTextValue().isEmpty()){
            if(phoneOffice.getTextValue().length() < CONTACT_PHONE_MIN_LENGTH || phoneOffice.getTextValue().length() > CONTACT_PHONE_MAX_LENGTH){
                errorPanel.addHeading1("Invalid office number");
                valid = false;
            } else if(valid){
                valid = validatePhone(phoneOffice.getTextValue());
                if(!valid){
                    errorPanel.addHeading1("Invalid office number");
                }
            }
        }

        if(phoneWork.getTextValue() != null && !phoneWork.getTextValue().isEmpty()){
            if(phoneWork.getTextValue().length() < CONTACT_PHONE_MIN_LENGTH || phoneWork.getTextValue().length() > CONTACT_PHONE_MAX_LENGTH){
                errorPanel.addHeading1("Invalid work number");
                valid = false;
            } else if(valid){
                valid = validatePhone(phoneWork.getTextValue());
                if(!valid){
                    errorPanel.addHeading1("Invalid work number");
                }
            }
        }

        if (!valid) {
            errorPanel.addHeading1("Contact with validation errors.");
        }

        return valid;
    }

    @SuppressWarnings("deprecation")
    private boolean validatePhone(String phone){
        boolean valid = true;

        for(Character c : phone.toCharArray()){
            if(!Character.isDigit(c)){
                if(c.equals('+') || c.equals('-') || c.equals('(') || c.equals(')') || Character.isSpace(c)){
                    valid = true;
                } else {
                    return false;
                }
            }
        }

        return valid;
    }

    @Override
    public void commit() {
        if(record.getKey() == null && record.getContact().getName() == null && record.getContact().getEmail() == null){
            UiContact contact = new UiContact();
            contact.setEmail(email.getText());
            contact.setName(name.getText());
            contact.setNotify(notify.getValue());
            contact.setPhoneMobile(phoneMobile.getText());
            contact.setPhoneOffice(phoneOffice.getText());
            contact.setPhoneWork(phoneWork.getText());
            record.setContact(contact);
        } else {
            record.getContact().setEmail(email.getText());
            record.getContact().setName(name.getText());
            record.getContact().setNotify(notify.getValue());
            record.getContact().setPhoneMobile(phoneMobile.getText());
            record.getContact().setPhoneOffice(phoneOffice.getText());
            record.getContact().setPhoneWork(phoneWork.getText());
        }
    }

    @Override
    public UiContactRecord getRecord() {
        return record;
    }

    @Override
    public void revert() {
    }

    @Override
    public String getStyleName() {
        return null;
    }

    @Override
    public Widget getWidget() {
        return this;
    }

    @Override
    public void release() {
    }

    @Override
    public void saveFailed(Throwable throwable) {

        if (throwable instanceof ValidationException) {
            ValidationException ve = (ValidationException) throwable;
            StringBuffer messagesSB = new StringBuffer();
            for (String message : ve.getValidationMessages()) {
                messagesSB.append(message).append("\n");
            }
            errorPanel.showMessage(messagesSB.toString());
        } else {
            errorPanel.showMessage(throwable.getLocalizedMessage());
        }
    }

    @Override
    protected void onUnload() {
        super.onUnload();
    }

    private void verifyPermissions(){
        if(isEditing){
            if(ClientAuthorizationManager.isAuthorized(Permissions.CONTRACT_MODIFY)){
                saveButton.setEnabled(true);
            }
        } else {
            if (ClientAuthorizationManager.isAuthorized(Permissions.CONTRACT_CREATE)){
                saveButton.setEnabled(true);
            }

        }

    }
}
