package com.ericsson.cms.contractmgmt.client.tab.contract;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import com.ericsson.cms.contractmgmt.client.ContractComponentConstants;
import com.ericsson.cms.contractmgmt.client.model.ContractSearchResult;
import com.ericsson.cms.contractmgmt.client.model.UiContract;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerService;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerServiceAsync;
import com.google.gwt.core.client.GWT;
import com.google.gwt.i18n.client.DateTimeFormat;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.ColumnBase;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.SortFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.View;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.label.LabelStringView;

/**
 * Data provider which operates on contracts.
 * 
 * @author Daniel Paranhos
 * 
 */
public class ContractDataProvider implements DataProvider<Long, UiContractRecord> {

	// columns for this table
    private final List<Column<?, UiContractRecord>> columns;
    private final ColumnBase<String, UiContractRecord> nameColumn;
    private final ColumnBase<String, UiContractRecord> typeColumn;
    private final ColumnBase<String, UiContractRecord> expiryColumn;

    private final ContractComponentConstants constants = (ContractComponentConstants) GWT
            .create( ContractComponentConstants.class );
    
    private final IContractManagerServiceAsync contractManagerService = GWT.create( IContractManagerService.class );

    private PageFeature parentPageFeature;
    private SortFeatureImpl<Long, UiContractRecord> parentSortFeature;

    // column names
    private static String NAME_COLUMN = "name";
    private static String TYPE_COLUMN = "contractType";
    private static String EXPIRY_DATE_COLUMN = "expiryDate";

    // lists panel
      private IBusyNotifier busyNotifier;
      
   // Cell style defined dynamically
   private final static String NON_ACTIVE_STYLE = "cell-NonActive";
   private final static String EXPIRED_STYLE = "cell-Expired";
   private final static String NON_ACTIVE_AND_EXPIRED = "cell-Expired-NonActive";
   private  String cellStyle = null;

	
    public ContractDataProvider( ContractListPanel listsPanel ) {

        // create the columns - id and name
        columns = new ArrayList<Column<?, UiContractRecord>>();

        nameColumn = new ColumnBase<String, UiContractRecord>( NAME_COLUMN, constants.contractName() ) {
            @Override
            public View<String> getView( UiContractRecord record ) {
            	ContractDataProvider.this.cellStyle = ContractDataProvider.this.getCellStyle(record);
                return new LabelStringView( String.valueOf( record.getContract().getName() ) );
            }
        };
        nameColumn.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( nameColumn );

        typeColumn = new ColumnBase<String, UiContractRecord>( TYPE_COLUMN,
                constants.contractType() ) {
            @Override
            public View<String> getView( UiContractRecord record ) {
            	LabelStringView lb = new LabelStringView( record.getContract().getType().getName() );
            	if(ContractDataProvider.this.cellStyle != null){
            		lb.setStyleName(ContractDataProvider.this.cellStyle);
            	}
            	return lb;
            }
        };
        typeColumn.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( typeColumn );
        
        expiryColumn = new ColumnBase<String, UiContractRecord>( EXPIRY_DATE_COLUMN,
                constants.contractExpiryDate()) {
            @Override
            public View<String> getView( UiContractRecord record ) {
            	DateTimeFormat dateFormatter = DateTimeFormat.getFormat( constants.dateFormat() );
            	 LabelStringView lb = new LabelStringView( dateFormatter.format( record.getContract().getExpiryDate() ) );
                 if(ContractDataProvider.this.cellStyle != null){
             		lb.setStyleName(ContractDataProvider.this.cellStyle);	
             	}
                 return lb;
            }
        };
        expiryColumn.setCellStyle( TableConstants.STYLE_DATACELL_DATE );
        columns.add( expiryColumn );
        
    }
    
    private final String getCellStyle(UiContractRecord record){
    	
    	if(!record.getContract().isActive() && new Date().after(record.getContract().getExpiryDate())){
    		return NON_ACTIVE_AND_EXPIRED;
    	}
    	else if(!record.getContract().isActive()){
    		return NON_ACTIVE_STYLE;
    	}
    	else if(new Date().after(record.getContract().getExpiryDate())){
    		return EXPIRED_STYLE;
    	}
    	return null;
    
    }

    public ColumnBase<String, UiContractRecord> getDefaultSortColumn() {
        return nameColumn;
    }

    public List<Column<?, UiContractRecord>> getSortColumns() {
        return columns;
    }

    @Override
    public List<Column<?, UiContractRecord>> getColumns() {
        return columns;
    }

    @Override
    public void getRecord( Long key, final AsyncCallback<UiContractRecord> callback ) {
    	notifyBusyStart();
    	contractManagerService.getContract(key, new NeptuneAsyncCallback<UiContract>() {

			@Override
			public void onNeptuneFailure(Throwable caught) {
				callback.onFailure(caught);
				notifyBusyFinish();
			}

			@Override
			public void onNeptuneSuccess(UiContract result) {
				callback.onSuccess(new UiContractRecord(result));
				notifyBusyFinish();
			}
    		
		});
    }

    @Override
    public void getRecords( final AsyncCallback<List<UiContractRecord>> callback ) {
        int length = getParentPageFeature().getPageSize();
        int start = (getParentPageFeature().getPageNumber() - 1) * length;
    	notifyBusyStart();
    	contractManagerService.getContractList(start, length, getParentSortFeature().getSortColumnName(), 
                getParentSortFeature().isAscending(),new NeptuneAsyncCallback<ContractSearchResult>() {
    		
    		@Override
    		public void onNeptuneFailure(Throwable caught) {
    		    getParentPageFeature().setRecordCount(-1);
    			callback.onFailure(caught);
    			notifyBusyFinish();
    		}
    		
    		@Override
    		public void onNeptuneSuccess(ContractSearchResult result) {
    		    getParentPageFeature().setRecordCount(result.getTotalCount());
    			callback.onSuccess(getContractRecordList((ArrayList<UiContract>)result.getUiContractList()));
    			notifyBusyFinish();
    		}
    		
    	});
    	    	
    }
    
    public List<UiContractRecord> getContractRecordList(List<UiContract> uiContractList){
    	List<UiContractRecord> list = new ArrayList<UiContractRecord>();
    	for (UiContract uiContract : uiContractList) {
			UiContractRecord record = new UiContractRecord();
			record.setContract(uiContract);
			list.add(record);
		}
     	
    	return list;
    }

    @Override
    public void initialize( AsyncCallback<Void> callback ) {
        callback.onSuccess( null );
    }

    @Override
    public boolean isCheckboxEnabled() {
        return true;
    }

    @Override
    public boolean isRecordCountEnabled() {
        return true;
    }

    /**
     * @return the parentPageFeature
     */
    public PageFeature getParentPageFeature() {
        return parentPageFeature;
    }

    /**
     * @param parentPageFeature
     *            the parentPageFeature to set
     */
    public void setParentPageFeature( PageFeature parentPageFeature ) {
        this.parentPageFeature = parentPageFeature;
    }

    /**
     * @return the parentSortFeature
     */
    public SortFeatureImpl<Long, UiContractRecord> getParentSortFeature() {
        return parentSortFeature;
    }

    /**
     * @param parentSortFeature
     *            the parentSortFeature to set
     */
    public void setParentSortFeature( SortFeatureImpl<Long, UiContractRecord> parentSortFeature ) {
        this.parentSortFeature = parentSortFeature;
    }

    public static interface IBusyNotifier {
        void startingToBeBusy();

        void doneBeingBusy();
    }

    public void setBusyNotifier( IBusyNotifier busyNotifier ) {
        this.busyNotifier = busyNotifier;
    }

    private void notifyBusyStart() {
        if ( busyNotifier != null ) {
        	busyNotifier.startingToBeBusy();
        }
    }

    private void notifyBusyFinish() {
        if ( busyNotifier != null ) {
        	busyNotifier.doneBeingBusy();
        }
    }

}
