package com.ericsson.cms.contractmgmt.client.tab.contract;

import java.util.ArrayList;
import java.util.List;

import com.ericsson.cms.contractmgmt.client.ContractComponentConstants;
import com.ericsson.cms.contractmgmt.client.ContractComponentMessages;
import com.ericsson.cms.contractmgmt.client.Permissions;
import com.ericsson.cms.contractmgmt.client.model.UiContract;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerService;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerServiceAsync;
import com.ericsson.cms.contractmgmt.client.tab.UiLongKeySerializer;
import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailViewCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.AbstractDetailFeature;

/**
 * Handles detail view for a given contract.
 * 
 * @author Daniel Paranhos
 * 
 */
public class ContractDetailFeature extends AbstractDetailFeature<Long, UiContractRecord> {

    private final ContractComponentConstants constants = (ContractComponentConstants) GWT
            .create( ContractComponentConstants.class );
    private final ContractComponentMessages messages = (ContractComponentMessages) GWT
            .create(ContractComponentMessages.class);
    
    private final IContractManagerServiceAsync contractManagerService = GWT.create( IContractManagerService.class );

    private final ContractListPanel parentPanel;
    
    private Long copyrightHolderId;

    public ContractDetailFeature( ContractListPanel parentPanel ) {
        super( new UiLongKeySerializer() );
        this.parentPanel = parentPanel;
    }

    @Override
    public void delete(List<UiContractRecord> records, final AsyncCallback<Void> callback) {
        if (records != null && !records.isEmpty()) {
            if (Window.confirm(messages.deleteContractConfirmation())) {
                List<UiContract> contractList = new ArrayList<UiContract>();
                for (UiContractRecord record : records) {
                    contractList.add(record.getContract());
                }
                contractManagerService.deleteContracts(contractList,
                        new NeptuneAsyncCallback<Boolean>() {

                            @Override
                            public void onNeptuneFailure(Throwable caught) {
                                parentPanel.getErrorPanel().showMessage( messages.deleteContractError() );
                                callback.onFailure(caught);
                            }

                            @Override
                            public void onNeptuneSuccess(Boolean result) {
                                parentPanel.getSuccessPanel().showMessage( messages.deleteContractSucceeded() );
                                callback.onSuccess(null);
                            }
                        });
            }
        } else {
            Window.alert(messages.noContractSelected());
        }
    }

    @Override
    public void getNew( AsyncCallback<UiContractRecord> callback ) {
        this.parentPanel.clearMessages();
        callback.onSuccess( new UiContractRecord( new UiContract() ) );
    }

    @Override
    public DetailView<Long, UiContractRecord> getView( UiContractRecord record,
            DetailViewCallback<Long, UiContractRecord> callback ) {
        this.parentPanel.clearMessages();
        ContractDetailView view = new ContractDetailView( record, callback );
        return view;
    }

    /*
     * The detail link is shown only if the user has asset view permissions and
     * the column is the name column.
     */
    @Override
    public boolean hasDetailLink( Column<?, UiContractRecord> column ) {
        return column.getDisplayName().equals( constants.contractNameColumn() )
                && ClientAuthorizationManager.isAuthorized( Permissions.CONTRACT_VIEW );
    }

    @Override
    public void save( UiContractRecord record, final AsyncCallback<Void> callback ) {
       	contractManagerService.saveContract(record.getContract(), new NeptuneAsyncCallback<Boolean>() {

			@Override
			public void onNeptuneFailure(Throwable caught) {
				callback.onFailure(caught);
			}

			@Override
			public void onNeptuneSuccess(Boolean result) {
                History.back();
			}
		});
    }

    @Override
    public boolean showCreateButton() {
        return ClientAuthorizationManager.isAuthorized( Permissions.CONTRACT_CREATE );
    }

    @Override
    public boolean showDeleteButton() {
        return ClientAuthorizationManager.isAuthorized( Permissions.CONTRACT_DELETE );
    }
    
    public Long getCopyrightHolderId() {
		return copyrightHolderId;
	}

	public void setCopyrightHolderId(Long copyrightHolderId) {
		this.copyrightHolderId = copyrightHolderId;
	}
}
