package com.ericsson.cms.contractmgmt.client.tab.copyrightholder;

import java.util.ArrayList;
import java.util.List;

import com.ericsson.cms.contractmgmt.client.ContractComponentConstants;
import com.ericsson.cms.contractmgmt.client.model.CopyrightHolderSearchResult;
import com.ericsson.cms.contractmgmt.client.model.UiCopyrightHolder;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerService;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerServiceAsync;
import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.ColumnBase;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.SortFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.View;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.label.LabelStringView;

/**
 * Data provider which operates on copyright holders.
 * 
 * @author Daniel Paranhos
 * 
 */
public class CopyrightHolderDataProvider implements DataProvider<Long, UiCopyrightHolderRecord> {

    // columns for this table
    private final List<Column<?, UiCopyrightHolderRecord>> columns;
    private final ColumnBase<String, UiCopyrightHolderRecord> addressColumn;
    private final ColumnBase<String, UiCopyrightHolderRecord> nameColumn;
    private final ColumnBase<String, UiCopyrightHolderRecord> affiliationColumn;

    private ContractComponentConstants constants = (ContractComponentConstants) GWT
            .create( ContractComponentConstants.class );
    
    private IContractManagerServiceAsync contractManagerService = GWT.create( IContractManagerService.class );


    private PageFeature parentPageFeature;
    private SortFeatureImpl<Long, UiCopyrightHolderRecord> parentSortFeature;

    // column names
    private static String ADDRESS_COLUMN = "address";
    private static String NAME_COLUMN = "name";
    private static String AFFILIATION_COLUMN = "affiliation";

    private IBusyNotifier busyNotifier;

    public CopyrightHolderDataProvider( CopyrightHolderListPanel listsPanel ) {

        columns = new ArrayList<Column<?, UiCopyrightHolderRecord>>();

        nameColumn = new ColumnBase<String, UiCopyrightHolderRecord>( NAME_COLUMN,
                constants.copyrightHolderNameColumn() ) {
            public View<String> getView( UiCopyrightHolderRecord record ) {
                return new LabelStringView( record.getCopyrightHolder().getName() );
            }
        };
        nameColumn.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( nameColumn );
        
        addressColumn = new ColumnBase<String, UiCopyrightHolderRecord>( ADDRESS_COLUMN, constants.copyrightHolderAddress() ) {
            public View<String> getView( UiCopyrightHolderRecord record ) {
                return new LabelStringView( String.valueOf( record.getCopyrightHolder().getAddress() ) );
            }
        };
        addressColumn.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( addressColumn );
        
        affiliationColumn = new ColumnBase<String, UiCopyrightHolderRecord>( AFFILIATION_COLUMN, constants.copyrightHolderAffiliation() ) {
            public View<String> getView( UiCopyrightHolderRecord record ) {
                return new LabelStringView( String.valueOf( record.getCopyrightHolder().getAffiliation() ) );
            }
        };
        affiliationColumn.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( affiliationColumn );
    }

    public ColumnBase<String, UiCopyrightHolderRecord> getDefaultSortColumn() {
        return nameColumn;
    }

    public List<Column<?, UiCopyrightHolderRecord>> getSortColumns() {
        return columns;
    }

    @Override
    public List<Column<?, UiCopyrightHolderRecord>> getColumns() {
        return columns;
    }

    @Override
    public void getRecord( Long key, final AsyncCallback<UiCopyrightHolderRecord> callback ) {
    	notifyBusyStart();
    	contractManagerService.getCopyrightHolder(key, new NeptuneAsyncCallback<UiCopyrightHolder>() {

			@Override
			public void onNeptuneFailure(Throwable caught) {
				callback.onFailure(caught);
				notifyBusyFinish();
			}

			@Override
			public void onNeptuneSuccess(UiCopyrightHolder result) {
				UiCopyrightHolderRecord record = new UiCopyrightHolderRecord();
				record.setCopyrightHolder(result);
				callback.onSuccess(record);
				notifyBusyFinish();
			}
    		
		});
    }

    @Override
    public void getRecords( final AsyncCallback<List<UiCopyrightHolderRecord>> callback ) {
    	int length = getParentPageFeature().getPageSize();
        int start = (getParentPageFeature().getPageNumber() - 1) * length;
    	notifyBusyStart();
    	contractManagerService.getCopyrightHolderList(start, length, getParentSortFeature().getSortColumnName(), 
                getParentSortFeature().isAscending(), new NeptuneAsyncCallback<CopyrightHolderSearchResult>() {

			@Override
			public void onNeptuneFailure(Throwable caught) {
				getParentPageFeature().setRecordCount(-1);
				callback.onFailure(caught);
				notifyBusyFinish();
			}

			@Override
			public void onNeptuneSuccess(CopyrightHolderSearchResult result) {
				getParentPageFeature().setRecordCount(result.getTotalCount());
				List<UiCopyrightHolderRecord> recordList = new ArrayList<UiCopyrightHolderRecord>();
				for(UiCopyrightHolder cp : result.getUiCopyrightHolderList()){
					UiCopyrightHolderRecord record = new UiCopyrightHolderRecord( cp );
					recordList.add(record);
				}
			
				callback.onSuccess(recordList);
				notifyBusyFinish();
			}
    		
		});
    }

    @Override
    public void initialize( AsyncCallback<Void> callback ) {
        callback.onSuccess( null );
    }

    @Override
    public boolean isCheckboxEnabled() {
        return true;
    }

    @Override
    public boolean isRecordCountEnabled() {
        return true;
    }

    /**
     * @return the parentPageFeature
     */
    public PageFeature getParentPageFeature() {
        return parentPageFeature;
    }

    /**
     * @param parentPageFeature
     *            the parentPageFeature to set
     */
    public void setParentPageFeature( PageFeature parentPageFeature ) {
        this.parentPageFeature = parentPageFeature;
    }

    /**
     * @return the parentSortFeature
     */
    public SortFeatureImpl<Long, UiCopyrightHolderRecord> getParentSortFeature() {
        return parentSortFeature;
    }

    /**
     * @param parentSortFeature
     *            the parentSortFeature to set
     */
    public void setParentSortFeature( SortFeatureImpl<Long, UiCopyrightHolderRecord> parentSortFeature ) {
        this.parentSortFeature = parentSortFeature;
    }

    public static interface IBusyNotifier {
        void startingToBeBusy();

        void doneBeingBusy();
    }

    public void setBusyNotifier( IBusyNotifier busyNotifier ) {
        this.busyNotifier = busyNotifier;
    }

    private void notifyBusyStart() {
        if ( busyNotifier != null ) {
        	busyNotifier.startingToBeBusy();
        }
    }

    private void notifyBusyFinish() {
        if ( busyNotifier != null ) {
        	busyNotifier.doneBeingBusy();
        }
    }

}
