package com.ericsson.cms.contractmgmt.client.tab.deal;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import com.ericsson.cms.contractmgmt.client.ContractComponentConstants;
import com.ericsson.cms.contractmgmt.client.model.DealSearchResult;
import com.ericsson.cms.contractmgmt.client.model.UiDeal;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerService;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerServiceAsync;
import com.google.gwt.core.client.GWT;
import com.google.gwt.i18n.client.DateTimeFormat;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.ColumnBase;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.SortFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.View;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.label.LabelStringView;

/**
 * Data provider which operates on deals.
 * 
 * @author Daniel Paranhos
 * 
 */
public class DealDataProvider implements DataProvider<Long, UiDealRecord> {

    // columns for this table
    private final List<Column<?, UiDealRecord>> columns;
    private final ColumnBase<String, UiDealRecord> nameColumn;
    private final ColumnBase<String, UiDealRecord> typeColumn;
    private final ColumnBase<String, UiDealRecord> expiryColumn;

    private final ContractComponentConstants constants = (ContractComponentConstants) GWT
            .create( ContractComponentConstants.class );
    
    private final IContractManagerServiceAsync contractManagerService = GWT.create( IContractManagerService.class );

    private PageFeature parentPageFeature;
    private SortFeatureImpl<Long, UiDealRecord> parentSortFeature;

    // column names
    private static final String NAME_COLUMN = "name";
    private static final String TYPE_COLUMN = "dealType";
    private static final String EXPIRY_DATE_COLUMN = "expiryDate";

    private IBusyNotifier busyNotifier;
    
    // Cell style defined dynamically
    private final static String NON_ACTIVE_STYLE = "cell-NonActive";
    private final static String EXPIRED_STYLE = "cell-Expired";
	private final static String NON_ACTIVE_AND_EXPIRED = "cell-Expired-NonActive";
	private  String cellStyle = null;

    public DealDataProvider( DealListPanel listsPanel ) {

        columns = new ArrayList<Column<?, UiDealRecord>>();

        nameColumn = new ColumnBase<String, UiDealRecord>( NAME_COLUMN, constants.dealName() ) {
            @Override
            public View<String> getView( UiDealRecord record ) {
            	DealDataProvider.this.cellStyle = DealDataProvider.this.getCellStyle(record);
                return new LabelStringView( String.valueOf( record.getDeal().getName() ) );
            }
        };
        nameColumn.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( nameColumn );

        typeColumn = new ColumnBase<String, UiDealRecord>( TYPE_COLUMN,
                constants.dealType() ) {
            @Override
            public View<String> getView( UiDealRecord record ) {
            	LabelStringView lb = new LabelStringView( record.getDeal().getType().getName() );
            	if(DealDataProvider.this.cellStyle != null){
            		lb.setStyleName(DealDataProvider.this.cellStyle);
            	}
            	return lb;
            }
        };
        typeColumn.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( typeColumn );
        
        expiryColumn = new ColumnBase<String, UiDealRecord>( EXPIRY_DATE_COLUMN,
                constants.dealExpiryDate()) {
            @Override
            public View<String> getView( UiDealRecord record ) {
            	DateTimeFormat dateFormatter = DateTimeFormat.getFormat( constants.dateFormat() );
            	 LabelStringView lb = new LabelStringView( dateFormatter.format( record.getDeal().getExpiryDate() ) );
                 if(DealDataProvider.this.cellStyle != null){
             		lb.setStyleName(DealDataProvider.this.cellStyle);	
             	}
                 return lb;
            }
        };
        expiryColumn.setCellStyle( TableConstants.STYLE_DATACELL_DATE );
        columns.add( expiryColumn );
        
    }
    
    private final String getCellStyle(UiDealRecord record){
    	
    	if(!record.getDeal().isActive() && new Date().after(record.getDeal().getExpiryDate())){
    		return NON_ACTIVE_AND_EXPIRED;
    	}
    	else if(!record.getDeal().isActive()){
    		return NON_ACTIVE_STYLE;
    	}
    	else if(new Date().after(record.getDeal().getExpiryDate())){
    		return EXPIRED_STYLE;
    	}
    	return null;
    
    }

    public ColumnBase<String, UiDealRecord> getDefaultSortColumn() {
        return nameColumn;
    }

    public List<Column<?, UiDealRecord>> getSortColumns() {
        return columns;
    }

    @Override
    public List<Column<?, UiDealRecord>> getColumns() {
        return columns;
    }

    @Override
    public void getRecord( Long key, final AsyncCallback<UiDealRecord> callback ) {
    	//orfano
    	notifyBusyStart();
    	contractManagerService.getDeal(key, new NeptuneAsyncCallback<UiDeal>() {

			@Override
			public void onNeptuneFailure(Throwable caught) {
				callback.onFailure(caught);
				notifyBusyFinish();
			}

			@Override
			public void onNeptuneSuccess(UiDeal result) {
				callback.onSuccess(new UiDealRecord(result));
				notifyBusyFinish();
			}
    		
		});
    }

    @Override
    public void getRecords( final AsyncCallback<List<UiDealRecord>> callback ) {
        int length = getParentPageFeature().getPageSize();
        int start = (getParentPageFeature().getPageNumber() - 1) * length;
		notifyBusyStart();
		contractManagerService.getDealsList(start, length, getParentSortFeature().getSortColumnName(), 
                getParentSortFeature().isAscending(),new NeptuneAsyncCallback<DealSearchResult>() {

			@Override
			public void onNeptuneFailure(Throwable caught) {
			    getParentPageFeature().setRecordCount(-1);
				notifyBusyFinish();
				callback.onFailure(caught);
			}

			@Override
			public void onNeptuneSuccess(DealSearchResult result) {
			    getParentPageFeature().setRecordCount(result.getTotalCount());
				notifyBusyFinish();
				List<UiDealRecord> list = new ArrayList<UiDealRecord>();
				for(UiDeal uiDeal : result.getUiDealList()){
					UiDealRecord record = new UiDealRecord();
					record.setDeal(uiDeal);
					list.add(record);
				}
				callback.onSuccess(list);
			}

		});
    }

    @Override
    public void initialize( AsyncCallback<Void> callback ) {
        callback.onSuccess( null );
    }

    @Override
    public boolean isCheckboxEnabled() {
        return true;
    }

    @Override
    public boolean isRecordCountEnabled() {
        return true;
    }

    /**
     * @return the parentPageFeature
     */
    public PageFeature getParentPageFeature() {
        return parentPageFeature;
    }

    /**
     * @param parentPageFeature
     *            the parentPageFeature to set
     */
    public void setParentPageFeature( PageFeature parentPageFeature ) {
        this.parentPageFeature = parentPageFeature;
    }

    /**
     * @return the parentSortFeature
     */
    public SortFeatureImpl<Long, UiDealRecord> getParentSortFeature() {
        return parentSortFeature;
    }

    /**
     * @param parentSortFeature
     *            the parentSortFeature to set
     */
    public void setParentSortFeature( SortFeatureImpl<Long, UiDealRecord> parentSortFeature ) {
        this.parentSortFeature = parentSortFeature;
    }

    public static interface IBusyNotifier {
        void startingToBeBusy();

        void doneBeingBusy();
    }

    public void setBusyNotifier( IBusyNotifier busyNotifier ) {
        this.busyNotifier = busyNotifier;
    }

    private void notifyBusyStart() {
        if ( busyNotifier != null ) {
        	busyNotifier.startingToBeBusy();
        }
    }

    private void notifyBusyFinish() {
        if ( busyNotifier != null ) {
        	busyNotifier.doneBeingBusy();
        }
    }

}
