package com.ericsson.cms.contractmgmt.client.tab.deal;

import java.util.ArrayList;
import java.util.List;

import com.ericsson.cms.contractmgmt.client.ContractComponentConstants;
import com.ericsson.cms.contractmgmt.client.ContractComponentMessages;
import com.ericsson.cms.contractmgmt.client.Permissions;
import com.ericsson.cms.contractmgmt.client.model.UiDeal;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerService;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerServiceAsync;
import com.ericsson.cms.contractmgmt.client.tab.UiLongKeySerializer;
import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailViewCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.AbstractDetailFeature;

/**
 * Handles detail view for a given deal.
 * 
 * @author Daniel Paranhos
 * 
 */
public class DealDetailFeature extends AbstractDetailFeature<Long, UiDealRecord> {


    private final ContractComponentConstants constants = (ContractComponentConstants) GWT
            .create( ContractComponentConstants.class );
    
    private final ContractComponentMessages messages = (ContractComponentMessages) GWT
            .create(ContractComponentMessages.class);

    private final IContractManagerServiceAsync contractManagerService = GWT.create( IContractManagerService.class );

    private final DealListPanel parentPanel;

    public DealDetailFeature( DealListPanel parentPanel ) {
        super( new UiLongKeySerializer() );
        this.parentPanel = parentPanel;
    }

    @Override
    public void delete(List<UiDealRecord> records, final AsyncCallback<Void> callback) {
        if (records != null && !records.isEmpty()) {
            if (Window.confirm(messages.deleteDealConfirmation())) {
                List<UiDeal> dealList = new ArrayList<UiDeal>();
                for (UiDealRecord record : records) {
                    dealList.add(record.getDeal());
                }

                contractManagerService.deleteDeals(dealList,
                        new NeptuneAsyncCallback<Boolean>() {

                            @Override
                            public void onNeptuneFailure(Throwable caught) {
                                parentPanel.getErrorPanel().showMessage( messages.deleteDealError() );
                                callback.onFailure(caught);
                            }

                            @Override
                            public void onNeptuneSuccess(Boolean result) {
                                parentPanel.getSuccessPanel().showMessage( messages.deleteDealSucceeded() );
                                callback.onSuccess(null);
                            }
                        });
            }
        } else {
            Window.alert(messages.noDealSelected());
        }
    }

    @Override
    public void getNew( AsyncCallback<UiDealRecord> callback ) {
        this.parentPanel.clearMessages();
        callback.onSuccess( new UiDealRecord( new UiDeal() ) );
    }

    @Override
    public DetailView<Long, UiDealRecord> getView( UiDealRecord record,
            DetailViewCallback<Long, UiDealRecord> callback ) {
        this.parentPanel.clearMessages();
        DealDetailView view = new DealDetailView( record, callback );
        return view;
    }

    /*
     * The detail link is shown only if the user has asset view permissions and
     * the column is the name column.
     */
    @Override
    public boolean hasDetailLink( Column<?, UiDealRecord> column ) {
        return column.getDisplayName().equals( constants.dealNameColumn() )
                && ClientAuthorizationManager.isAuthorized( Permissions.CONTRACT_VIEW );
    }

    @Override
    public void save( UiDealRecord record, final AsyncCallback<Void> callback ) {
    	contractManagerService.saveDeal(record.getDeal(), new NeptuneAsyncCallback<Boolean>() {

			@Override
			public void onNeptuneFailure(Throwable caught) {
				callback.onFailure(caught);
			}

			@Override
			public void onNeptuneSuccess(Boolean result) {
				History.back();
			}
		});
    }

    @Override
    public boolean showCreateButton() {
        return ClientAuthorizationManager.isAuthorized( Permissions.CONTRACT_CREATE );
    }

    @Override
    public boolean showDeleteButton() {
        return ClientAuthorizationManager.isAuthorized( Permissions.CONTRACT_DELETE );
    }
}
