package com.ericsson.cms.contractmgmt.client.tab.license;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import com.ericsson.cms.contractmgmt.client.Changeable;
import com.ericsson.cms.contractmgmt.client.ContractComponentConstants;
import com.ericsson.cms.contractmgmt.client.ContractComponentMessages;
import com.ericsson.cms.contractmgmt.client.Permissions;
import com.ericsson.cms.contractmgmt.client.customfield.CustomFieldValidator;
import com.ericsson.cms.contractmgmt.client.customfield.UiUserDefinedField;
import com.ericsson.cms.contractmgmt.client.customfield.UiUserDefinedFieldValue;
import com.ericsson.cms.contractmgmt.client.model.UiLicense;
import com.ericsson.cms.contractmgmt.client.model.UiLicenseDetails;
import com.ericsson.cms.contractmgmt.client.tab.license.tree.DeviceTreePanel;
import com.ericsson.cms.contractmgmt.client.tab.license.tree.TreeListener;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.Grid;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.application.ValidationException;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.ScrollContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.DataTypeWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

public abstract class AbstractLicenseTab extends Composite implements TreeListener, Changeable {

    private static final String STYLE_SCROLL_CONTAINER = "contract-licenseTab-scrollContainer";
    private static final String STYLE_DEVICE_SCROLL_CONTAINER = "contract-DeviceTree-scrollContainer";
    private static final String STYLE_SPLIT_CONTAINER = "content-AssetPanel-splitContainer";
    private static final String STYLE_TREE_CELL = "contract-DeviceTree-splitContainer-treeCell";
    private static final String STYLE_CONTENT_CELL = "content-AssetPanel-splitContainer-contentCell";

    private final UiLicense license;
    private final ButtonWidget applyChangesButton;
    private final DeviceTreePanel deviceTreePanel;
    private boolean changed;
    private boolean firstTime;

    protected final LicenseDetailView detailView;
    protected VerticalContainer contentPanel;
    protected UiLicenseDetails currentTreeObject;
    protected final Map<DataTypeWidget<?>, UiUserDefinedField> widgetCustomFieldMap;

    protected BusyIndicator busyIndicator = new BusyIndicator();

    protected ContractComponentConstants constants = (ContractComponentConstants) GWT
            .create( ContractComponentConstants.class );

    protected ContractComponentMessages messages = (ContractComponentMessages) GWT
            .create( ContractComponentMessages.class );

    public AbstractLicenseTab( UiLicense license, final LicenseDetailView detailView ) {
        this.firstTime = true;
        this.license = license;
        this.detailView = detailView;
        this.widgetCustomFieldMap = new HashMap<DataTypeWidget<?>, UiUserDefinedField>();

        contentPanel = new VerticalContainer();

        currentTreeObject = license.getDefaultLicenseDetails();

        Grid splitContainer = new Grid( 1, 2 );
        splitContainer.setStyleName( STYLE_SPLIT_CONTAINER );
        deviceTreePanel = new DeviceTreePanel( license, this );
        ScrollContainer deviceScrollPanel = new ScrollContainer();
        deviceScrollPanel.addStyleName(STYLE_DEVICE_SCROLL_CONTAINER);
        deviceScrollPanel.setPixelSize(0, 0);

        deviceScrollPanel.setWidget(deviceTreePanel);

        splitContainer.setWidget( 0, 0, deviceScrollPanel );
        splitContainer.getCellFormatter().setStyleName( 0, 0, STYLE_TREE_CELL );

        ScrollContainer contentScrollPanel = new ScrollContainer(contentPanel);
        contentScrollPanel.addStyleName(STYLE_SCROLL_CONTAINER);

        splitContainer.setWidget( 0, 1, contentScrollPanel );
        splitContainer.getCellFormatter().setStyleName( 0, 1, STYLE_CONTENT_CELL );

        applyChangesButton = new ButtonWidget( constants.applyChangesButtonName(), new ClickHandler() {

            @Override
            public void onClick( ClickEvent event ) {
                applyChanges( true );
            }
        } );
        applyChangesButton.addStyleDependentName( StyleNames.COMMIT_BUTTON_STYLE );

        initWidget( splitContainer );
    }

    public abstract String getName();

    public abstract String getDisplayName();

    public abstract void validate() throws ValidationException;

    public abstract void commit() throws ValidationException;

    public abstract void fillForm();

    protected void updateUserDefinedFieldsValuesFromForm( Collection<UiUserDefinedFieldValue> oldValuesList ) {
        Set<Entry<DataTypeWidget<?>, UiUserDefinedField>> entrySet = widgetCustomFieldMap.entrySet();
        for ( Entry<DataTypeWidget<?>, UiUserDefinedField> entry : entrySet ) {
            DataTypeWidget<?> dataTypeWidget = entry.getKey();
            UiUserDefinedField userDefinedField = entry.getValue();
            UiUserDefinedFieldValue userDefinedFieldValue = getValueForFieldType( oldValuesList,
                    userDefinedField.getIdUserDefinedField() );
            if ( userDefinedFieldValue != null ) {
                userDefinedFieldValue.setValue( dataTypeWidget.getTextValue() );
            }
        }
    }

    protected void updateFormFromUserDefinedFieldValues( Collection<UiUserDefinedFieldValue> newValuesList ) {

        Set<Entry<DataTypeWidget<?>, UiUserDefinedField>> entrySet = widgetCustomFieldMap.entrySet();
        for ( Entry<DataTypeWidget<?>, UiUserDefinedField> entry : entrySet ) {
            DataTypeWidget<?> dataTypeWidget = entry.getKey();
            UiUserDefinedField userDefinedField = entry.getValue();
            UiUserDefinedFieldValue userDefinedFieldValue = getValueForFieldType( newValuesList,
                    userDefinedField.getIdUserDefinedField() );
            if ( userDefinedFieldValue != null ) {
                dataTypeWidget.setTextValue( userDefinedFieldValue.getValue() );
            }
        }
    }

    private UiUserDefinedFieldValue getValueForFieldType( Collection<UiUserDefinedFieldValue> newValuesList,
            Long idUserDefinedField ) {
        UiUserDefinedFieldValue userDefinedFieldValue = null;
        for ( UiUserDefinedFieldValue newUserDefinedFieldValue : newValuesList ) {
            Long newIdUserDefinedField = newUserDefinedFieldValue.getIdUserDefinedField();
            if ( newIdUserDefinedField.equals( idUserDefinedField ) ) {
                userDefinedFieldValue = newUserDefinedFieldValue;
                break;
            }
        }

        return userDefinedFieldValue;
    }

    protected void copyUserDefinedFieldValues( List<UiUserDefinedFieldValue> toList ) {
        List<UiUserDefinedFieldValue> fromList = getTabUserDefinedFieldValuesList();
        for ( UiUserDefinedFieldValue fromField : fromList ) {
            for ( UiUserDefinedFieldValue toField : toList ) {
                if ( fromField.getIdUserDefinedField().equals( toField.getIdUserDefinedField() ) ) {
                    toField.setValue( fromField.getValue() );
                    break;
                }
            }
        }

    }

    private List<UiUserDefinedFieldValue> getTabUserDefinedFieldValuesList() {
        List<UiUserDefinedFieldValue> valuesList = new ArrayList<UiUserDefinedFieldValue>();
        for ( UiUserDefinedFieldValue uiUserDefinedFieldValue : currentTreeObject.getUserDefinedFieldValues() ) {
            for ( UiUserDefinedField uiUserDefinedField : widgetCustomFieldMap.values() ) {
                if ( uiUserDefinedField.getScreenName().equals( getName() )
                        && uiUserDefinedFieldValue.getIdUserDefinedField().equals(
                                uiUserDefinedField.getIdUserDefinedField() ) ) {
                    valuesList.add( uiUserDefinedFieldValue );
                    break;
                }
            }
        }

        return valuesList;
    }

    @Override
    public void nodeSelected( UiLicenseDetails treeObject ) {
        if (changeEnabled() && changed && Window.confirm(messages.dataLossConfirmation())) {
            applyChanges();
        }
        changed = false;
        detailView.getErrorPanel().reset();

        if ( treeObject != null ) {
            currentTreeObject = treeObject;
            fillForm();
        }
    }

    private boolean changeEnabled() {
        boolean isCreating = getLicense().getKey() == null;
        return isCreating && ClientAuthorizationManager.isAuthorized( Permissions.CONTRACT_CREATE )
                || !isCreating && ClientAuthorizationManager.isAuthorized( Permissions.CONTRACT_MODIFY );
    }

    public UiLicense getLicense() {
        return license;
    }

    protected ButtonWidget getApplyChangesButton() {
        return applyChangesButton;
    }

    public void setSelected( boolean selected ) {
        if ( selected ) {
            deviceTreePanel.updateModifiedItems();
        } else if (changeEnabled() && changed && Window.confirm(messages.dataLossConfirmation())) {
            applyChanges();
            changed = false;
        }
    }

    private void applyChanges() {
        applyChanges( false );
    }

    private void applyChanges( boolean forced ) {
        if ( firstTime || changed || forced ) {
            boolean proceed = true;
            if ( forced && !changed ) {
                proceed = Window.confirm( messages.licenseTabNotChangedConfirm() );
            }
            if ( proceed ) {
                try {
                    detailView.getErrorPanel().reset();
                    validate();
                    commit();
                    firstTime = false;
                    changed = false;
                    deviceTreePanel.updateModifiedItems();
                } catch ( ValidationException e ) {
                    for ( String message : e.getValidationMessages() ) {
                        detailView.getErrorPanel().addMessage( message );
                    }
                }
            }
        }
    }

    public boolean isChanged() {
        return changed;
    }

    @Override
    public void setChanged( boolean changed ) {
        this.changed = changed;
    }

    public void validateCustomFields( List<String> validationMessages ) {
        Set<Entry<DataTypeWidget<?>, UiUserDefinedField>> entrySet = widgetCustomFieldMap.entrySet();
        for ( Entry<DataTypeWidget<?>, UiUserDefinedField> entry : entrySet ) {
            DataTypeWidget<?> dataTypeWidget = entry.getKey();
            UiUserDefinedField field = entry.getValue();
            CustomFieldValidator.validate( dataTypeWidget, field );
            if ( !dataTypeWidget.isValidValue() || dataTypeWidget.hasValidationError() ) {
                String tooltip = dataTypeWidget.getToolTip();
                if ( tooltip == null ) {
                    tooltip = "";
                }
                validationMessages.add( messages.invalidFieldValue( field.getLabel() ) + tooltip );
            }
        }
    }

}
