package com.ericsson.cms.contractmgmt.client.tab.license;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import com.ericsson.cms.contractmgmt.client.ContractComponentConstants;
import com.ericsson.cms.contractmgmt.client.model.LicenseSearchResult;
import com.ericsson.cms.contractmgmt.client.model.UiLicense;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerService;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerServiceAsync;
import com.google.gwt.core.client.GWT;
import com.google.gwt.i18n.client.DateTimeFormat;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.ColumnBase;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.PageFeature;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.SortFeatureImpl;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.View;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.label.LabelStringView;

/**
 * Data provider which operates on licenses.
 * 
 * @author Daniel Paranhos
 * 
 */
public class LicenseDataProvider implements DataProvider<Long, UiLicenseRecord> {

		
    // columns for this table
    private final List<Column<?, UiLicenseRecord>> columns;
    private final ColumnBase<String, UiLicenseRecord> idColumn;
    private final ColumnBase<String, UiLicenseRecord> nameColumn;
    private final ColumnBase<String, UiLicenseRecord> typeColumn;
    private final ColumnBase<String, UiLicenseRecord> expiryColumn;
    private final ColumnBase<String, UiLicenseRecord> activeColumn;

    private ContractComponentConstants constants = (ContractComponentConstants) GWT
            .create( ContractComponentConstants.class );

    private IContractManagerServiceAsync contractManagerService = GWT.create( IContractManagerService.class );

    private PageFeature parentPageFeature;
    private SortFeatureImpl<Long, UiLicenseRecord> parentSortFeature;

    // column names
    private static final String ID_COLUMN = "id";
    private static final String NAME_COLUMN = "name";
    private static final String TYPE_COLUMN = "licenseType";
    private static final String EXPIRY_DATE_COLUMN = "expiryDate";
    private static final String ACTIVE_COLUMN = "active";

    private IBusyNotifier busyNotifier;
    
    // Cell style defined dynamically
    private final static String NON_ACTIVE_STYLE = "cell-NonActive";
    private final static String EXPIRED_STYLE = "cell-Expired";
	private final static String NON_ACTIVE_AND_EXPIRED = "cell-Expired-NonActive";
	private  String cellStyle = null;
	
	
    public LicenseDataProvider( LicenseListPanel listsPanel ) {

        columns = new ArrayList<Column<?, UiLicenseRecord>>();
          
        idColumn = new ColumnBase<String, UiLicenseRecord>( ID_COLUMN, constants.licenseId() ) {
            public View<String> getView( UiLicenseRecord record ) {
            	LicenseDataProvider.this.cellStyle = LicenseDataProvider.this.getCellStyle(record);
            	LabelStringView lb = new LabelStringView( String.valueOf( record.getLicense().getKey() ) );
            	if(LicenseDataProvider.this.cellStyle != null){
            		lb.setStyleName(LicenseDataProvider.this.cellStyle);	
            	}
            	return lb;
            }
        };
        idColumn.setCellStyle( TableConstants.STYLE_DATACELL_NUMERIC );
        columns.add( idColumn );
                
        nameColumn = new ColumnBase<String, UiLicenseRecord>( NAME_COLUMN, constants.licenseNameColumn() ) {
            public View<String> getView( UiLicenseRecord record ) {
            	return new LabelStringView( String.valueOf( record.getLicense().getName() ) );
            }
        };
        nameColumn.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( nameColumn );

        typeColumn = new ColumnBase<String, UiLicenseRecord>( TYPE_COLUMN, constants.licenseTypeColumn() ) {
            public View<String> getView( UiLicenseRecord record ) {
            	LabelStringView lb = new LabelStringView( record.getLicense().getType().getName() );
            	if(LicenseDataProvider.this.cellStyle != null){
            		lb.setStyleName(LicenseDataProvider.this.cellStyle);	
            	}
            	return lb;
            }
        };
        typeColumn.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( typeColumn );

        expiryColumn = new ColumnBase<String, UiLicenseRecord>( EXPIRY_DATE_COLUMN,
                constants.licenseExpiryDateColumn() ) {
            public View<String> getView( UiLicenseRecord record ) {
                DateTimeFormat dateFormatter = DateTimeFormat.getFormat( constants.dateFormat() );
                LabelStringView lb = new LabelStringView( dateFormatter.format( record.getLicense().getExpiryDate() ) );
                if(LicenseDataProvider.this.cellStyle != null){
            		lb.setStyleName(LicenseDataProvider.this.cellStyle);	
            	}
                return lb;
            }
        };
        expiryColumn.setCellStyle( TableConstants.STYLE_DATACELL_DATE );
        columns.add( expiryColumn );

       
        activeColumn = new ColumnBase<String, UiLicenseRecord>( ACTIVE_COLUMN,
                constants.licenseActiveColumn() ) {
            public View<String> getView( UiLicenseRecord record ) {
                if ( record.getLicense().isActive() ) {
                	LabelStringView lb = new LabelStringView( constants.yes() );
                	if(LicenseDataProvider.this.cellStyle != null){
                		lb.setStyleName(LicenseDataProvider.this.cellStyle);	
                	} 
                    return lb;
                } else {
                	LabelStringView lb = new LabelStringView( constants.no() );
                	if(LicenseDataProvider.this.cellStyle != null){
                		lb.setStyleName(LicenseDataProvider.this.cellStyle);	
                	}
                    return lb;
                }
            }
        };
        activeColumn.setCellStyle( TableConstants.STYLE_DATACELL_TEXT );
        columns.add( activeColumn );
        
    }
    
    private final String getCellStyle(UiLicenseRecord record){
    	
    	if(!(record.getLicense().isActive()) && new Date().after(record.getLicense().getExpiryDate())){
    		return NON_ACTIVE_AND_EXPIRED;
    	}
    	else if(!(record.getLicense().isActive())){
    		return NON_ACTIVE_STYLE;
    	}
    	else if(new Date().after(record.getLicense().getExpiryDate())){
    		return EXPIRED_STYLE;
    	}
    	return null;
    
    }
    
   
    public ColumnBase<String, UiLicenseRecord> getDefaultSortColumn() {
        return nameColumn;
    }

    public List<Column<?, UiLicenseRecord>> getSortColumns() {
        return columns;
    }

    @Override
    public List<Column<?, UiLicenseRecord>> getColumns() {
        return columns;
    }

    @Override
    public void getRecord( Long key, final AsyncCallback<UiLicenseRecord> callback ) {
        notifyBusyStart();
        contractManagerService.getFullLicense( key, new NeptuneAsyncCallback<UiLicense>() {

            @Override
            public void onNeptuneFailure( Throwable caught ) {
                callback.onFailure( caught );
                notifyBusyFinish();
            }

            @Override
            public void onNeptuneSuccess( UiLicense result ) {
                callback.onSuccess( new UiLicenseRecord( result ) );
                notifyBusyFinish();
            }

        } );
    }

    @Override
    public void getRecords( final AsyncCallback<List<UiLicenseRecord>> callback ) {
        int length = getParentPageFeature().getPageSize();
        int start = (getParentPageFeature().getPageNumber() - 1) * length;
        notifyBusyStart();
        contractManagerService.getLicenseList( start, length, getParentSortFeature().getSortColumnName(), 
                getParentSortFeature().isAscending(), new NeptuneAsyncCallback<LicenseSearchResult>() {

            @Override
            public void onNeptuneFailure( Throwable caught ) {
                getParentPageFeature().setRecordCount(-1);
                callback.onFailure( caught );
                notifyBusyFinish();
            }

            @Override
            public void onNeptuneSuccess( LicenseSearchResult result ) {
                getParentPageFeature().setRecordCount(result.getTotalCount());
                List<UiLicenseRecord> uiLicenseRecords = new ArrayList<UiLicenseRecord>();
                        for ( UiLicense uiLicense : result.getUiLicensesList() ) {
                    uiLicenseRecords.add( new UiLicenseRecord( uiLicense ) );
                }
                callback.onSuccess( uiLicenseRecords );
                notifyBusyFinish();
            }

        } );
    }

    @Override
    public void initialize( AsyncCallback<Void> callback ) {
        callback.onSuccess( null );
    }

    @Override
    public boolean isCheckboxEnabled() {
        return true;
    }

    @Override
    public boolean isRecordCountEnabled() {
        return true;
    }

    /**
     * @return the parentPageFeature
     */
    public PageFeature getParentPageFeature() {
        return parentPageFeature;
    }

    /**
     * @param parentPageFeature
     *            the parentPageFeature to set
     */
    public void setParentPageFeature( PageFeature parentPageFeature ) {
        this.parentPageFeature = parentPageFeature;
    }

    /**
     * @return the parentSortFeature
     */
    public SortFeatureImpl<Long, UiLicenseRecord> getParentSortFeature() {
        return parentSortFeature;
    }

    /**
     * @param parentSortFeature
     *            the parentSortFeature to set
     */
    public void setParentSortFeature( SortFeatureImpl<Long, UiLicenseRecord> parentSortFeature ) {
        this.parentSortFeature = parentSortFeature;
    }

    public static interface IBusyNotifier {
        void startingToBeBusy();

        void doneBeingBusy();
    }

    public void setBusyNotifier( IBusyNotifier busyNotifier ) {
        this.busyNotifier = busyNotifier;
    }

    private void notifyBusyStart() {
        if ( busyNotifier != null ) {
            busyNotifier.startingToBeBusy();
        }
    }

    private void notifyBusyFinish() {
        if ( busyNotifier != null ) {
            busyNotifier.doneBeingBusy();
        }
    }

}
