package com.ericsson.cms.contractmgmt.client.tab.license;

import java.util.List;

import com.ericsson.cms.contractmgmt.client.ContractComponentConstants;
import com.ericsson.cms.contractmgmt.client.ContractComponentMessages;
import com.ericsson.cms.contractmgmt.client.Permissions;
import com.ericsson.cms.contractmgmt.client.customfield.UiUserDefinedField;
import com.ericsson.cms.contractmgmt.client.model.UiLicense;
import com.ericsson.cms.contractmgmt.client.model.UiLicenseDetails;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerService;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerServiceAsync;
import com.ericsson.cms.contractmgmt.client.tab.UiLongKeySerializer;
import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.application.ValidationException;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailViewCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.AbstractDetailFeature;

/**
 * Handles detail view for a given license.
 * 
 * @author Daniel Paranhos
 * 
 */
public class LicenseDetailFeature extends AbstractDetailFeature<Long, UiLicenseRecord> {

    private final LicenseListPanel parentPanel;
   
    private final IContractManagerServiceAsync contractManagerService = GWT.create( IContractManagerService.class );

    private final ContractComponentConstants constants = (ContractComponentConstants) GWT
            .create( ContractComponentConstants.class );

    private final ContractComponentMessages messages = (ContractComponentMessages) GWT
            .create( ContractComponentMessages.class );

    public LicenseDetailFeature( LicenseListPanel parentPanel ) {
        super( new UiLongKeySerializer() );
        this.parentPanel = parentPanel;
    }

    @Override
    public void delete( List<UiLicenseRecord> records, final AsyncCallback<Void> callback ) {
        if ( records.size() > 0 ) {
            if ( Window.confirm( messages.deleteLicenseConfirmation() ) ) {
                this.parentPanel.showBusyIndicator();
                contractManagerService.deleteLicenses( records, new NeptuneAsyncCallback<Void>() {

                    @Override
                    public void onNeptuneFailure( Throwable caught ) {
                    	parentPanel.getLicenseTable().refresh();
                        parentPanel.getErrorPanel().showMessage( messages.deleteLicenseError() );
                        if (caught instanceof ValidationException) {
                            ValidationException ve = (ValidationException) caught;
                            for ( String message : ve.getValidationMessages() ) {
                            	parentPanel.getErrorPanel().addMessage(message);
                            }
                        } else {
                        	callback.onFailure( caught );
                        }
                        parentPanel.hideBusyIndicator();
                    }

                    @Override
                    public void onNeptuneSuccess( Void result ) {
                        parentPanel.getSuccessPanel().showMessage( messages.deleteLicenseSucceeded() );
                        callback.onSuccess( null );
                        parentPanel.hideBusyIndicator();
                    }
                } );
            }
        } else {
            Window.alert( messages.noLicenseSelected() );
        }
    }

    @Override
    public void getNew( final AsyncCallback<UiLicenseRecord> callback ) {
        this.parentPanel.clearMessages();

        this.parentPanel.showBusyIndicator();
        contractManagerService
                .getLicenseUserDefinedFields( new NeptuneAsyncCallback<List<UiUserDefinedField>>() {

                    @Override
                    public void onNeptuneFailure( Throwable caught ) {
                        callback.onFailure( caught );
                        parentPanel.hideBusyIndicator();
                    }

                    @Override
                    public void onNeptuneSuccess( List<UiUserDefinedField> result ) {
                        UiLicense license = new UiLicense();
                        license.getUserDefinedFields().addAll( result );
                        license.resetDefaultValues();

                        UiLicenseDetails defaultLicenseDetails = new UiLicenseDetails( license );
                        license.getLicenseDetailsList().add( defaultLicenseDetails );

                        callback.onSuccess( new UiLicenseRecord( license ) );
                        parentPanel.hideBusyIndicator();
                    }
                } );
    }

    @Override
    public DetailView<Long, UiLicenseRecord> getView( UiLicenseRecord record,
            DetailViewCallback<Long, UiLicenseRecord> callback ) {
        this.parentPanel.clearMessages();
        LicenseDetailView view = new LicenseDetailView( this.parentPanel, record, callback );
        return view;
    }

    /*
     * The detail link is shown only if the user has contract view permissions and the column is the name
     * column.
     */
    @Override
    public boolean hasDetailLink( Column<?, UiLicenseRecord> column ) {
        return column.getDisplayName().equals( constants.licenseNameColumn() )
                && ClientAuthorizationManager.isAuthorized( Permissions.CONTRACT_VIEW );
    }

    @Override
    public void save( final UiLicenseRecord record, final AsyncCallback<Void> callback ) {
        this.parentPanel.showBusyIndicator();
        contractManagerService.saveLicense( record, new NeptuneAsyncCallback<Void>() {

            @Override
            public void onNeptuneFailure( Throwable caught ) {
                callback.onFailure( caught );
                parentPanel.hideBusyIndicator();
            }

            @Override
            public void onNeptuneSuccess( Void result ) {
            	parentPanel.hideBusyIndicator();
            	History.back();
            	
            }
        } );
    }

    @Override
    public boolean showCreateButton() {
        return ClientAuthorizationManager.isAuthorized( Permissions.CONTRACT_CREATE );
    }

    @Override
    public boolean showDeleteButton() {
        return ClientAuthorizationManager.isAuthorized( Permissions.CONTRACT_DELETE );
    }
}
