package com.ericsson.cms.contractmgmt.client.tab.license;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.ericsson.cms.contractmgmt.client.ContractComponentConstants;
import com.ericsson.cms.contractmgmt.client.ContractComponentMessages;
import com.ericsson.cms.contractmgmt.client.Permissions;
import com.ericsson.cms.contractmgmt.client.customfield.CustomFieldFactory;
import com.ericsson.cms.contractmgmt.client.customfield.UiUserDefinedField;
import com.ericsson.cms.contractmgmt.client.customfield.UiUserDefinedFieldValue;
import com.ericsson.cms.contractmgmt.client.model.UiDeal;
import com.ericsson.cms.contractmgmt.client.model.UiDeliveryLocation;
import com.ericsson.cms.contractmgmt.client.model.UiDevice;
import com.ericsson.cms.contractmgmt.client.model.UiLicense;
import com.ericsson.cms.contractmgmt.client.model.UiLicenseDetails;
import com.ericsson.cms.contractmgmt.client.model.UiPlatform;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerService;
import com.ericsson.cms.contractmgmt.client.rpc.IContractManagerServiceAsync;
import com.ericsson.cms.contractmgmt.client.tab.ErrorPanel;
import com.ericsson.cms.contractmgmt.client.tab.SuccessPanel;
import com.ericsson.cms.contractmgmt.client.tab.license.tab.AssetEnhancementTab;
import com.ericsson.cms.contractmgmt.client.tab.license.tab.DistributionTab;
import com.ericsson.cms.contractmgmt.client.tab.license.tab.LicenseDetailsUtils;
import com.ericsson.cms.contractmgmt.client.tab.license.tab.PricingTab;
import com.ericsson.cms.contractmgmt.client.tab.license.tab.StorageMetadataTab;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.BeforeSelectionEvent;
import com.google.gwt.event.logical.shared.BeforeSelectionHandler;
import com.google.gwt.event.logical.shared.SelectionEvent;
import com.google.gwt.event.logical.shared.SelectionHandler;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HasHorizontalAlignment;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.application.ValidationException;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailView;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.DetailViewCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.TabContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

public class LicenseDetailView extends Composite implements DetailView<Long, UiLicenseRecord> {

    private static final String STYLE_TABCONTAINER = "license-tabContainer";
    private static final String ASSETS_ALREADY_ADDED = "licenseAssetsAlreadyAdded";
    private static final String INVALID_ASSETS = "licenseInvalidAssets";
    private static final String ASSETS_ASSOCIATED_WITH_LICENSES = "licenseAssetsAssociated";
    private static final String STYLE_NAME = "content-ContractPanel";

    private final ContractComponentConstants constants = (ContractComponentConstants) GWT
            .create( ContractComponentConstants.class );
    private final ContractComponentMessages messages = (ContractComponentMessages) GWT
            .create( ContractComponentMessages.class );

    private final IContractManagerServiceAsync contractManagerService = GWT.create( IContractManagerService.class );

    private final UiLicenseRecord record;
    private final DetailViewCallback<Long, UiLicenseRecord> callback;
    private final LicenseListPanel parentPanel;
    private TabContainer tabContainer;
    private final List<AbstractLicenseTab> tabs;
    private int oldSelectedTab;
    private PricingTab pricingTab;
    private DistributionTab distributionTab;
    private AssetEnhancementTab assetEnhancementTab;
    private StorageMetadataTab storageMetadataTab;
    private final VerticalContainer mainContainer;
    private final VerticalContainer contentContainer;
    private HorizontalContainer buttonsContainer;

    private LicenseMainForm licenseMainForm;

    private final SuccessPanel successPanel = new SuccessPanel();
    private final ErrorPanel errorPanel = new ErrorPanel();

    private final BusyIndicator busyIndicator = new BusyIndicator();

    public LicenseDetailView( LicenseListPanel parentPanel, final UiLicenseRecord record,
            DetailViewCallback<Long, UiLicenseRecord> callback ) {
        this.parentPanel = parentPanel;
        this.record = record;
        this.callback = callback;

        tabs = new ArrayList<AbstractLicenseTab>();
        mainContainer = new VerticalContainer();
        mainContainer.add( successPanel );
        mainContainer.add( errorPanel );
        contentContainer = new VerticalContainer();
        contentContainer.setStylePrimaryName(STYLE_NAME);
        mainContainer.add( contentContainer );

        if (record.getKey() != null) {
            // get the full license object
            busyIndicator.center();
            contractManagerService.getFullLicense( record.getKey(), new NeptuneAsyncCallback<UiLicense>() {

                @Override
                public void onNeptuneFailure( Throwable caught ) {
                    errorPanel.showMessage( caught.getLocalizedMessage() );
                    busyIndicator.hide();
                }

                @Override
                public void onNeptuneSuccess( UiLicense result ) {
                    record.setLicense( result );
                    createMainForm();
                    getErrorsFromSession();
                    busyIndicator.hide();
                }

            } );
        } else {
            createMainForm();
        }

        initWidget( mainContainer );
    }

    private void getErrorsFromSession() {
        contractManagerService.getLicenseErrorsFromHttpSession( record.getKey(),
                new AsyncCallback<Map<String, Object>>() {

            @Override
            public void onFailure( Throwable caught ) {
                Window.alert( constants.errorsMapError() );
            }

            @SuppressWarnings("unchecked")
            @Override
            public void onSuccess( Map<String, Object> errorsMap ) {
                Integer assetsAlreadyAdded = (Integer) errorsMap.get( ASSETS_ALREADY_ADDED + record.getKey() );
                successPanel.reset();
                if ( assetsAlreadyAdded != null && assetsAlreadyAdded > 0 ) {
                    successPanel.setVisible( true );
                    successPanel.addMessage( constants.assetsAlreadyAdded() );
                }
                Integer invalidAssets = (Integer) errorsMap.get( INVALID_ASSETS + record.getKey() );
                if ( invalidAssets != null && invalidAssets > 0 ) {
                    successPanel.setVisible( true );
                    successPanel.addMessage( constants.invalidAssetsProvider() );
                }

                List<String> assetsLicenseAssociated = (List<String>) errorsMap.get(ASSETS_ASSOCIATED_WITH_LICENSES + record.getKey());
                if(assetsLicenseAssociated != null && !assetsLicenseAssociated.isEmpty()){
                    errorPanel.setVisible(true);
                    for(String message : assetsLicenseAssociated){
                        errorPanel.addMessage(message);
                    }
                }
            }
        }
                );
    }

    private void createMainForm() {
        licenseMainForm = new LicenseMainForm( record.getLicense(), errorPanel,
                HasHorizontalAlignment.ALIGN_LEFT );

        showMainLicensePage();
    }

    private void showMainLicensePage() {
        contentContainer.clear();
        contentContainer.add( licenseMainForm );
        addMainFormButtons();
    }

    private void addMainFormButtons() {

        buttonsContainer = new HorizontalContainer();
        buttonsContainer.setSpacing( 5 );
        contentContainer.add( buttonsContainer );

        addNextButton( buttonsContainer );
        addDeleteButton( buttonsContainer );
        addCancelButton( buttonsContainer );
    }

    private void createLicenceTabs() {
        contentContainer.clear();
        initTabs();

        buttonsContainer = new HorizontalContainer();
        buttonsContainer.setSpacing( 5 );
        contentContainer.add( buttonsContainer );

        addSaveButton( buttonsContainer );
        addDeleteButton( buttonsContainer );
        addBackButton( buttonsContainer );
        addCancelButton( buttonsContainer );
    }

    private void addNextButton( HorizontalContainer buttonsContainer ) {
        ButtonWidget nextButton = new ButtonWidget( constants.configureLicenseNextButton(),
                new ClickHandler() {

            @Override
            public void onClick( ClickEvent event ) {
                if ( record.getLicense().getDeal().getKey() != null ) {
                    busyIndicator.center();
                    contractManagerService.getDeal( record.getLicense().getDeal().getKey(),
                            new NeptuneAsyncCallback<UiDeal>() {

                        @Override
                        public void onNeptuneFailure( Throwable caught ) {
                            errorPanel.showMessage( caught.getLocalizedMessage() );
                            busyIndicator.hide();
                        }

                        @Override
                        public void onNeptuneSuccess( UiDeal result ) {
                            licenseMainForm.setDeal( result );
                            try {
                                licenseMainForm.validateWithMessages();
                                commit();
                                buildLicenseDetailsList();
                                createLicenceTabs();
                                successPanel.reset();
                                errorPanel.reset();
                                busyIndicator.hide();
                            } catch ( ValidationException e ) {
                                errorPanel.reset();
                                successPanel.reset();
                                for ( String message : e.getValidationMessages() ) {
                                    errorPanel.addMessage( message );
                                }
                                busyIndicator.hide();
                            }
                        }

                    } );
                } else {
                    try {
                        errorPanel.reset();
                        licenseMainForm.validateWithMessages();
                    } catch ( ValidationException e ) {
                        for ( String message : e.getValidationMessages() ) {
                            errorPanel.addMessage( message );
                        }
                    }
                }
            }

        } );
        nextButton.addStyleDependentName( StyleNames.COMMIT_BUTTON_STYLE );
        nextButton.ensureDebugId("license-next");
        buttonsContainer.add( nextButton );
    }

    private void addCancelButton( HorizontalContainer buttonsContainer ) {
        ButtonWidget cancelButton = new ButtonWidget( constants.cancelButtonName(), new ClickHandler() {
            @Override
            public void onClick( ClickEvent event ) {
                errorPanel.reset();
                successPanel.reset();
                History.back();
            }
        } );
        buttonsContainer.add( cancelButton );
    }

    private void addBackButton( HorizontalContainer buttonsContainer ) {
        ButtonWidget backButton = new ButtonWidget( constants.backButtonName(), new ClickHandler() {
            @Override
            public void onClick( ClickEvent event ) {
                for ( AbstractLicenseTab tab : tabs ) {
                    if ( tab.isChanged() ) {
                        tab.nodeSelected( null );
                    }
                }
                errorPanel.reset();
                LicenseDetailView.this.parentPanel.getErrorPanel().reset();
                showMainLicensePage();
            }
        } );
        buttonsContainer.add( backButton );
    }

    private void addSaveButton( HorizontalContainer buttonsContainer ) {
        boolean isCreating = record.getKey() == null;
        if ( isCreating && ClientAuthorizationManager.isAuthorized( Permissions.CONTRACT_CREATE )
                || !isCreating && ClientAuthorizationManager.isAuthorized( Permissions.CONTRACT_MODIFY ) ) {
            ButtonWidget saveButton = new ButtonWidget( constants.saveButtonName(), new ClickHandler() {

                @Override
                public void onClick( ClickEvent event ) {
                    parentPanel.getErrorPanel().reset();
                    errorPanel.reset();
                    if ( validateTabs() ) {
                        commit();
                        LicenseDetailView.this.callback.save( LicenseDetailView.this );
                    }
                }

            } );
            saveButton.addStyleDependentName( StyleNames.COMMIT_BUTTON_STYLE );
            saveButton.ensureDebugId("license-save");
            buttonsContainer.add( saveButton );
        }
    }

    private void addDeleteButton( HorizontalContainer buttonsContainer ) {
        boolean isEditing = record.getKey() != null;
        if ( isEditing && ClientAuthorizationManager.isAuthorized( Permissions.CONTRACT_DELETE ) ) {
            ButtonWidget deleteButton = new ButtonWidget( constants.deleteButtonName(), new ClickHandler() {

                @Override
                public void onClick( ClickEvent event ) {
                    deleteLicense();
                }

            } );
            deleteButton.addStyleDependentName( StyleNames.DATALOSS_BUTTON_STYLE );
            buttonsContainer.add( deleteButton );
        }
    }

    private void deleteLicense() {
        if ( Window.confirm( messages.deleteLicenseConfirmation() ) ) {
            busyIndicator.center();
            contractManagerService.deleteLicense( record.getKey(), new NeptuneAsyncCallback<Void>() {

                @Override
                public void onNeptuneFailure( Throwable caught ) {
                    errorPanel.showMessage( messages.deleteLicenseError() );
                    if (caught instanceof ValidationException) {
                        ValidationException ve = (ValidationException) caught;
                        for ( String message : ve.getValidationMessages() ) {
                            errorPanel.addMessage(message);
                        }
                    }
                    busyIndicator.hide();
                }

                @Override
                public void onNeptuneSuccess( Void result ) {
                    parentPanel.getSuccessPanel().showMessage( messages.deleteLicenseSucceeded() );
                    busyIndicator.hide();
                    callback.cancel( LicenseDetailView.this );
                }
            } );
        }
    }

    private void buildLicenseDetailsList() {
        if ( record.getKey() == null ) {
            createLicenseDetailsList();
        } else {
            updateLicenseDetailsList();
        }
    }

    private void updateLicenseDetailsList() {
        UiLicense uiLicense = record.getLicense();
        List<UiLicenseDetails> licenseDetailsListTemp = new ArrayList<UiLicenseDetails>();
        createMissingUserDefinedFieldValues( uiLicense, uiLicense.getDefaultLicenseDetails() );
        UiLicenseDetails defaultLicenseDetails = uiLicense.getDefaultLicenseDetails();
        if ( defaultLicenseDetails != null ) {
            licenseDetailsListTemp.add( uiLicense.getDefaultLicenseDetails() );
        } else {
            defaultLicenseDetails = createLicenseDetails( uiLicense, null, null, null );
            licenseDetailsListTemp.add( defaultLicenseDetails );
        }

        for ( UiDeliveryLocation deliveryLocation : uiLicense.getDeliveryLocations() ) {
            UiLicenseDetails licenseDetails = findLicenseDetails( uiLicense.getLicenseDetailsList(),
                    deliveryLocation );
            if ( licenseDetails != null ) {
                createMissingUserDefinedFieldValues( uiLicense, licenseDetails );
                if(licenseDetails.getPlatform() == null && licenseDetails.getDevice() == null){
                    licenseDetails.getChildren().clear();
                }
                licenseDetailsListTemp.add( licenseDetails );
            } else {
                UiLicenseDetails newLicenseDetails = createLicenseDetails( uiLicense, deliveryLocation, null,
                        null );
                LicenseDetailsUtils.copy( defaultLicenseDetails, newLicenseDetails );
                licenseDetailsListTemp.add( newLicenseDetails );
            }

            for ( UiPlatform platform : uiLicense.getPlatforms() ) {
                licenseDetails = findLicenseDetails( uiLicense.getLicenseDetailsList(), deliveryLocation,
                        platform );
                if ( licenseDetails != null ) {
                    platform.setDeliveryLocation( deliveryLocation );
                    createMissingUserDefinedFieldValues( uiLicense, licenseDetails );
                    licenseDetailsListTemp.add( licenseDetails );
                } else {
                    platform.setDeliveryLocation( deliveryLocation );
                    UiLicenseDetails newLicenseDetails = createLicenseDetails( uiLicense, deliveryLocation,
                            platform, null );
                    LicenseDetailsUtils.copy( defaultLicenseDetails, newLicenseDetails );
                    licenseDetailsListTemp.add( newLicenseDetails );
                }

                for ( UiDevice device : uiLicense.getDevices() ) {
                    if ( device.getPlatform().getKey().equals( platform.getKey() ) ) {
                        licenseDetails = findLicenseDetails( uiLicense.getLicenseDetailsList(),
                                deliveryLocation, device );
                        if ( licenseDetails != null ) {
                            createMissingUserDefinedFieldValues( uiLicense, licenseDetails );
                            licenseDetailsListTemp.add( licenseDetails );
                        } else {
                            UiLicenseDetails newLicenseDetails = createLicenseDetails( uiLicense,
                                    deliveryLocation, platform, device );
                            LicenseDetailsUtils.copy( defaultLicenseDetails, newLicenseDetails );
                            licenseDetailsListTemp.add( newLicenseDetails );
                        }
                    }
                }
            }
        }

        uiLicense.getLicenseDetailsList().clear();
        uiLicense.getLicenseDetailsList().addAll( licenseDetailsListTemp );
    }

    private void createMissingUserDefinedFieldValues( UiLicense license, UiLicenseDetails licenseDetails ) {
        for ( UiUserDefinedField field : license.getUserDefinedFields() ) {
            if ( !field.getScreenName().equals( CustomFieldFactory.LICENSE_SCREEN ) ) {
                UiUserDefinedFieldValue uiUserDefinedFieldValue = findUserDefinedFieldValue( licenseDetails,
                        field.getIdUserDefinedField() );
                if ( uiUserDefinedFieldValue == null ) {
                    uiUserDefinedFieldValue = new UiUserDefinedFieldValue();
                    uiUserDefinedFieldValue.setIdUserDefinedField( field.getIdUserDefinedField() );
                    uiUserDefinedFieldValue.setValue( field.getDefaultValue() );
                    licenseDetails.getUserDefinedFieldValues().add( uiUserDefinedFieldValue );
                }
            }
        }
    }

    private UiUserDefinedFieldValue findUserDefinedFieldValue( UiLicenseDetails licenseDetails,
            Long idUserDefinedField ) {
        UiUserDefinedFieldValue uiUserDefinedFieldValue = null;
        for ( UiUserDefinedFieldValue fieldValue : licenseDetails.getUserDefinedFieldValues() ) {
            if ( fieldValue.getIdUserDefinedField().equals( idUserDefinedField ) ) {
                uiUserDefinedFieldValue = fieldValue;
                break;
            }
        }

        return uiUserDefinedFieldValue;
    }

    private UiLicenseDetails findLicenseDetails( List<UiLicenseDetails> licenseDetailsList,
            UiDeliveryLocation deliveryLocation ) {
        UiLicenseDetails licenseDetails = null;
        for ( UiLicenseDetails ld : licenseDetailsList ) {
            if (ld.getDeliveryLocation() != null && deliveryLocation != null) {
                if (ld.getDeliveryLocation().getName().equals(deliveryLocation.getName()) // NOSONAR
                        && ld.getPlatform() == null && ld.getDevice() == null) {
                    licenseDetails = ld;
                    break;
                }
            }
        }
        return licenseDetails;
    }

    private UiLicenseDetails findLicenseDetails( List<UiLicenseDetails> licenseDetailsList,
            UiDeliveryLocation deliveryLocation, UiPlatform platform ) {
        UiLicenseDetails licenseDetails = null;
        for ( UiLicenseDetails ld : licenseDetailsList ) {
            if (ld.getDeliveryLocation() != null && ld.getPlatform() != null) {
                if (ld.getDeliveryLocation().getName().equals(deliveryLocation.getName())   // NOSONAR
                        && ld.getPlatform().getKey().equals(platform.getKey()) && ld.getDevice() == null) {
                    licenseDetails = ld;
                    break;
                }
            }
        }
        return licenseDetails;
    }

    private UiLicenseDetails findLicenseDetails( List<UiLicenseDetails> licenseDetailsList,
            UiDeliveryLocation deliveryLocation, UiDevice device ) {
        UiLicenseDetails licenseDetails = null;
        for ( UiLicenseDetails ld : licenseDetailsList ) {
            if (ld.getDeliveryLocation() != null && ld.getPlatform() != null && ld.getDevice() != null) {
                if (ld.getDeliveryLocation().getName().equals(deliveryLocation.getName())   // NOSONAR
                        && ld.getPlatform().getKey().equals(device.getPlatform().getKey())
                        && ld.getDevice().getKey().equals(device.getKey())) {
                    licenseDetails = ld;
                    break;
                }
            }
        }
        return licenseDetails;
    }

    private void createLicenseDetailsList() {
        UiLicense uiLicense = record.getLicense();
        uiLicense.getLicenseDetailsList().clear();
        UiLicenseDetails defaultLicenseDetails = new UiLicenseDetails( uiLicense );
        uiLicense.getLicenseDetailsList().add( defaultLicenseDetails );

        for ( UiDeliveryLocation deliveryLocation : record.getLicense().getDeliveryLocations() ) {
            UiLicenseDetails licenseDetails = createLicenseDetails( uiLicense, deliveryLocation, null, null );
            uiLicense.getLicenseDetailsList().add( licenseDetails );
            for ( UiPlatform platform : record.getLicense().getPlatforms() ) {
                platform.setDeliveryLocation( deliveryLocation );
                licenseDetails = createLicenseDetails( uiLicense, deliveryLocation, platform, null );
                uiLicense.getLicenseDetailsList().add( licenseDetails );
                for ( UiDevice device : record.getLicense().getDevices() ) {
                    if ( device.getPlatform().getKey().equals( platform.getKey() ) ) {
                        licenseDetails = createLicenseDetails( uiLicense, deliveryLocation, platform, device );
                        uiLicense.getLicenseDetailsList().add( licenseDetails );
                    }
                }
            }
        }

    }

    private UiLicenseDetails createLicenseDetails( UiLicense license, UiDeliveryLocation deliveryLocation,
            UiPlatform platform, UiDevice device ) {
        UiLicenseDetails licenseDetails = new UiLicenseDetails( license );
        licenseDetails.setDevice( device );
        licenseDetails.setPlatform( platform );
        licenseDetails.setDeliveryLocation( deliveryLocation );

        return licenseDetails;
    }

    private boolean validateTabs() {
        errorPanel.reset();
        boolean valid = true;
        boolean addErrorHeader = false;
        for ( AbstractLicenseTab tab : tabs ) {
            try {
                tab.validate();
            } catch ( ValidationException e ) {
                valid = false;
                if ( !addErrorHeader ) {
                    addErrorHeader = true;
                    errorPanel.addHeading1( messages.tabsValidationError() );
                }
                errorPanel.addHeading2( tab.getDisplayName() );
            }
        }
        return valid;
    }

    private void initTabs() {

        tabContainer = new TabContainer();
        tabContainer.addBeforeSelectionHandler( new BeforeSelectionHandler<Integer>() {

            @Override
            public void onBeforeSelection( BeforeSelectionEvent<Integer> event ) {
                if ( changeEnabled() ) {
                    if ( tabs.get( oldSelectedTab ).isChanged() ) {
                        event.cancel();
                        tabs.get( oldSelectedTab ).nodeSelected( null );
                    } else {
                        oldSelectedTab = event.getItem();
                    }
                }
            }
        } );
        tabContainer.addSelectionHandler( new SelectionHandler<Integer>() {
            @Override
            public void onSelection( SelectionEvent<Integer> event ) {
                for ( int i = 0; i < tabs.size(); ++i ) {
                    tabs.get( i ).setSelected( i == event.getSelectedItem() );
                }
            }
        } );
        tabContainer.addStyleName( STYLE_TABCONTAINER );

        tabs.clear();
        addTabs();

        contentContainer.add( tabContainer );
    }

    private boolean changeEnabled() {
        boolean isCreating = record.getKey() == null;
        return isCreating && ClientAuthorizationManager.isAuthorized( Permissions.CONTRACT_CREATE )
                || !isCreating && ClientAuthorizationManager.isAuthorized( Permissions.CONTRACT_MODIFY );
    }

    private void addTabs() {
        storageMetadataTab = new StorageMetadataTab( record.getLicense(), this );
        tabs.add( storageMetadataTab );
        tabContainer.add( storageMetadataTab, storageMetadataTab.getDisplayName() );

        pricingTab = new PricingTab( record.getLicense(), this );
        tabs.add( pricingTab );
        tabContainer.add( pricingTab, pricingTab.getDisplayName() );

        distributionTab = new DistributionTab( record.getLicense(), this );
        tabs.add( distributionTab );
        tabContainer.add( distributionTab, distributionTab.getDisplayName() );

        assetEnhancementTab = new AssetEnhancementTab( record.getLicense(), this );
        tabs.add( assetEnhancementTab );
        tabContainer.add( assetEnhancementTab, assetEnhancementTab.getDisplayName() );

        tabContainer.selectTab( 0 );
    }


    @Override
    public void commit() {
        licenseMainForm.commit();
    }

    @Override
    public UiLicenseRecord getRecord() {
        return record;
    }

    @Override
    public void revert() {
    }

    @Override
    public String getStyleName() {
        return null;
    }

    @Override
    public Widget getWidget() {
        return this;
    }

    @Override
    public void release() {
    }

    @Override
    public void saveFailed( Throwable throwable ) {
        if ( throwable instanceof ValidationException ) {
            ValidationException ve = (ValidationException) throwable;
            for ( String message : ve.getValidationMessages() ) {
                this.parentPanel.getErrorPanel().addMessage( message );
            }
        } else {
            this.parentPanel.getErrorPanel().showMessage( throwable.getLocalizedMessage() );
        }
    }

    @Override
    protected void onUnload() {
        super.onUnload();
    }

    public ErrorPanel getErrorPanel() {
        return this.errorPanel;
    }

}
