package com.ericsson.cms.contractmgmt.client.tab.license.tree;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.ericsson.cms.contractmgmt.client.ContractComponentConstants;
import com.ericsson.cms.contractmgmt.client.ContractComponentMessages;
import com.ericsson.cms.contractmgmt.client.model.UiLicenseDetails;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.logical.shared.SelectionEvent;
import com.google.gwt.event.logical.shared.SelectionHandler;
import com.google.gwt.user.client.ui.HasVerticalAlignment;
import com.google.gwt.user.client.ui.TreeItem;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TreeWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;

public class DeviceTree extends TreeWidget {

    private static final String STYLE_NAME = "contract-DeviceTree";
    private static final String STYLE_ITEM = "contract-DeviceTree-item";
    private static final String STYLE_ITEM_DISABLED = "contract-DeviceTree-item-disabled";

    private final Map<String, TreeItem> deliveryLocationMap = new HashMap<String, TreeItem>();
    private final Map<String, TreeItem> platformMap = new HashMap<String, TreeItem>();
    private final Map<String, TreeItem> deviceMap = new HashMap<String, TreeItem>();

    private TreeItem currentSelectedTreeItem;

    protected ContractComponentConstants constants = (ContractComponentConstants) GWT
            .create( ContractComponentConstants.class );

    protected ContractComponentMessages messages = (ContractComponentMessages) GWT
            .create( ContractComponentMessages.class );

    /**
     * Construct a device tree
     */
    public DeviceTree( List<UiLicenseDetails> licenseDetailsList, UiLicenseDetails defaultLicenseDetails,
            final TreeListener treeListener ) {
        addStyleName( STYLE_NAME );

        addSelectionHandler( new SelectionHandler<TreeItem>() {
            @Override
            public void onSelection( SelectionEvent<TreeItem> event ) {
                if ( treeListener != null ) {
                    Object userObject = event.getSelectedItem().getUserObject();
                    if ( userObject != null ) {
                        treeListener.nodeSelected( (UiLicenseDetails) userObject );
                        currentSelectedTreeItem = event.getSelectedItem();
                    } else {
                        if ( currentSelectedTreeItem != null ) {
                            setSelectedItem( currentSelectedTreeItem );
                        }
                    }
                }
            }
        } );

        addTreeItems( licenseDetailsList, defaultLicenseDetails );

        // Open all tree elements
        Iterator<TreeItem> iterator = treeItemIterator();
        while ( iterator.hasNext() ) {
            TreeItem item = iterator.next();
            item.setState( true );
        }

    }

    private void addTreeItems( List<UiLicenseDetails> licenseDetails, UiLicenseDetails defaultLicenseDetails ) {

        // Find all DeliveryLocation nodes
        for ( UiLicenseDetails ld : licenseDetails ) {
            // Node is as DeliveryLocation
            if (ld.getDeliveryLocation() != null && ld.getPlatform() == null && ld.getDevice() == null) {
                TreeItem locationTreeItem = createDeliveryLocationNode( ld );
                deliveryLocationMap.put( ld.getDeliveryLocation().getName(), locationTreeItem );
                defaultLicenseDetails.addChild( ld );
            }
        }

        // Find all Platform nodes
        for ( UiLicenseDetails ld : licenseDetails ) {
            // Node is as Platform
            if (ld.getDeliveryLocation() != null && ld.getPlatform() != null && ld.getDevice() == null) {
                TreeItem locationTreeItem = deliveryLocationMap.get( ld.getDeliveryLocation().getName() );
                if ( locationTreeItem != null ) {
                    TreeItem platformTreeItem = createPlatformNode( locationTreeItem, ld );
                    platformMap.put( ld.getDeliveryLocation().getName() + "-" + ld.getPlatform().getKey(),
                            platformTreeItem );
                    UiLicenseDetails parent = (UiLicenseDetails) locationTreeItem.getUserObject();
                    parent.addChild( ld );
                }
            }
        }

        // Find all Device nodes
        for ( UiLicenseDetails ld : licenseDetails ) {
            // Node is as Device
            if (ld.getDeliveryLocation() != null && ld.getPlatform() != null && ld.getDevice() != null) {
                TreeItem platformTreeItem = platformMap.get( ld.getDeliveryLocation().getName() + "-"
                        + ld.getPlatform().getKey() );
                if ( platformTreeItem != null ) {
                    TreeItem deviceTreeItem = createDeviceNode( platformTreeItem, ld );
                    deviceMap.put( ld.getDeliveryLocation().getName() + "-" + ld.getPlatform().getKey() + "-"
                            + ld.getDevice().getKey(), deviceTreeItem );
                    UiLicenseDetails parent = (UiLicenseDetails) platformTreeItem.getUserObject();
                    parent.addChild( ld );
                }
            }
        }

        // Find all empty DeliveryLocation nodes:
        // now Platforms are not mandatory anymore so the addAllNode would apply to them.
        for ( UiLicenseDetails ld : licenseDetails ) {
            // Node is as DeliveryLocation
            if (ld.getDeliveryLocation() != null && ld.getPlatform() == null && ld.getDevice() == null) {
                TreeItem locationTreeItem = deliveryLocationMap.get(ld.getDeliveryLocation().getName());
                if (locationTreeItem != null && ld.getAllChildren().isEmpty()) {
                    addAllNode(locationTreeItem, constants.allPlatform());
                    addAllNode(locationTreeItem.getChild(locationTreeItem.getChildCount() - 1), constants.allDevices());
                }
            }
        }

        // Find all empty Platform nodes
        for ( UiLicenseDetails ld : licenseDetails ) {
            // Node is as Platform
            if (ld.getDeliveryLocation() != null && ld.getPlatform() != null && ld.getDevice() == null) {
                TreeItem platformTreeItem = platformMap.get( ld.getDeliveryLocation().getName() + "-"
                        + ld.getPlatform().getKey() );
                if (platformTreeItem != null && ld.getAllChildren().isEmpty()) {
                    addAllNode(platformTreeItem);
                }
            }
        }

    }

    private void addAllNode( TreeItem platformTreeItem ) {
        TreeItem allNode = new TreeItem( buildTreeItemWidget( constants.allDevices(), true ) );
        allNode.setText( constants.allDevices() );

        platformTreeItem.addItem( allNode );
    }

    private void addAllNode( TreeItem treeItem, String label ) {
        TreeItem allNode = new TreeItem( buildTreeItemWidget( label, true ) );
        allNode.setText( label );

        treeItem.addItem( allNode );
    }

    public TreeItem createDeliveryLocationNode( UiLicenseDetails licenseDetails ) {
        /* Build the tree item */
        TreeItem locationTreeItem = new TreeItem( buildTreeItemWidget( licenseDetails.getDeliveryLocation()
                .getName(), false ) );
        locationTreeItem.setText( licenseDetails.getDeliveryLocation().getName() );
        locationTreeItem.setUserObject( licenseDetails );
        addItem( locationTreeItem );

        return locationTreeItem;
    }

    /*
     * Build the child tree items for the given DeliveryLocation tree item
     */
    private TreeItem createPlatformNode( TreeItem locationTreeItem, UiLicenseDetails licenseDetails ) {
        TreeItem platformTreeItem = new TreeItem( buildTreeItemWidget(
                licenseDetails.getPlatform().getName(), false ) );
        platformTreeItem.setText( licenseDetails.getPlatform().getName() );
        platformTreeItem.setUserObject( licenseDetails );
        locationTreeItem.addItem( platformTreeItem );

        return platformTreeItem;
    }

    /*
     * Build the child tree items for the given Platform tree item
     */
    private TreeItem createDeviceNode( TreeItem platformTreeItem, UiLicenseDetails licenseDetails ) {
        TreeItem deviceTreeItem = new TreeItem( buildTreeItemWidget( licenseDetails.getDevice().getName(),
                false ) );
        deviceTreeItem.setText( licenseDetails.getDevice().getName() );
        deviceTreeItem.setUserObject( licenseDetails );
        platformTreeItem.addItem( deviceTreeItem );

        return deviceTreeItem;
    }

    /**
     * 
     * Build the tree item widget
     * 
     * @param item
     * @param disabled
     * @param modified
     * 
     * @return
     */
    public Widget buildTreeItemWidget( String name, boolean disabled ) {
        HorizontalContainer container = new HorizontalContainer();
        if ( !disabled ) {
            container.addStyleName( STYLE_ITEM );
        } else {
            container.addStyleName( STYLE_ITEM_DISABLED );
        }
        container.setVerticalAlignment( HasVerticalAlignment.ALIGN_MIDDLE );

        LabelWidget label = new LabelWidget( name );
        container.add( label );

        return container;
    }

    public void selectNone() {
        TreeItem selectedItem = getSelectedItem();
        if ( selectedItem != null ) {
            selectedItem.setSelected( false );
        }
    }

}
