package com.ericsson.cms.epgmgmt.client.model;

import com.ericsson.cms.epgmgmt.client.model.enumeration.ScheduleType;
import com.ericsson.cms.epgmgmt.client.model.enumeration.TriggerType;
import com.google.gwt.user.client.rpc.IsSerializable;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Record;

public class UITaskTrigger  implements Record<Long>, IsSerializable{

    private static final int MAX_SCHEDULER_COUNT = 5;

    private static final long MINUTE_IN_SEC = 60L;

    private static final long HOUR_IN_SEC = 3600L;

    private static final long DAY_IN_SEC = 86400L;

    private static final int MAX_SCHEDULE_NUM = 5;

    private long id;

    private TriggerType triggerType;

    private long frequency;

    private long days;

    private long hours;

    private long minutes;

    private ScheduleType scheduleType;
    private long[] scheduleHours;
    private long[] scheduleMinutes;
    private long[] scheduleDaysOrWeeks ;
    private long scheduleCount;

    private long triggeredByTask;
    private long triggeredByPartner;

    public UITaskTrigger() {
        setTriggerType(TriggerType.FREQUENCY);
        setScheduleType(ScheduleType.DAILY);
        this.triggeredByPartner = -1;
        this.triggeredByTask = -1;
    }

    /**
     * Copy from another {@link UITaskTrigger}, if the pd is null, IllegalArgumentException will be throw.
     * 
     * @param taskTrigger
     */
    public UITaskTrigger(UITaskTrigger taskTrigger) {
        if (taskTrigger == null) {
            throw new IllegalArgumentException("parameter 'task trigger' cannot be null");
        }
        this.triggerType = taskTrigger.getTriggerType();
        this.frequency = taskTrigger.getFrequency();
        this.days = taskTrigger.getDays();
        this.hours = taskTrigger.getHours();
        this.minutes = taskTrigger.getMinutes();
        this.scheduleType = taskTrigger.getScheduleType();
        this.scheduleCount = taskTrigger.getScheduleCount();
        this.triggeredByTask = taskTrigger.getTriggeredByTask();
        this.triggeredByPartner = taskTrigger.getTriggeredByPartner();

        this.scheduleHours = new long[(int) scheduleCount];
        this.scheduleMinutes = new long[(int) scheduleCount];
        this.scheduleDaysOrWeeks = new long[(int) scheduleCount];
        for (int i = 0; i < scheduleCount; i++) {
            this.scheduleHours[i] = taskTrigger.getScheduleHours()[i];
            this.scheduleMinutes[i] = taskTrigger.getScheduleMinutes()[i];
            this.scheduleDaysOrWeeks[i] = taskTrigger.getScheduleDaysOrWeeks()[i];
        }
    }

    public long getId() {
        return id;
    }

    public void setId(long id) {
        this.id = id;
    }

    public TriggerType getTriggerType() {
        return triggerType;
    }

    public final void setTriggerType(TriggerType triggerType) {
        this.triggerType = triggerType;
    }

    public long getFrequency() {
        frequency = (days * DAY_IN_SEC) + (hours * HOUR_IN_SEC) + (minutes * MINUTE_IN_SEC);
        return frequency;
    }

    public final void setFrequency(long frequency) {
        this.frequency = frequency;
        this.days = frequency / DAY_IN_SEC;
        long remainingSecHoursMinutes = frequency - (days * DAY_IN_SEC);
        this.hours = remainingSecHoursMinutes / HOUR_IN_SEC;
        long remainingSecMinutes = remainingSecHoursMinutes - (hours * HOUR_IN_SEC);
        if ((remainingSecHoursMinutes > 0) && (remainingSecMinutes < MINUTE_IN_SEC) && (days <= 0) && (hours <= 0)) {
            this.minutes = 1L;
        } else {
            this.minutes = remainingSecMinutes / MINUTE_IN_SEC;
        }
    }

    public long getDays() {
        return days;
    }

    public void setDays(long days) {
        this.days = days;
    }

    public long getHours() {
        return hours;
    }

    public void setHours(long hours) {
        this.hours = hours;
    }

    public long getMinutes() {
        return minutes;
    }

    public void setMinutes(long minutes) {
        this.minutes = minutes;
    }

    public ScheduleType getScheduleType() {
        return scheduleType;
    }

    public final void setScheduleType(ScheduleType scheduleType) {
        this.scheduleType = scheduleType;
    }

    // For entity storage
    public String getEntityTriggerValue() {
        StringBuilder triggerValue = new StringBuilder("");
        if (TriggerType.FREQUENCY.equals(triggerType)) {
            triggerValue.append(String.valueOf(getFrequency()));
        } else if (TriggerType.SCHEDULE.equals(triggerType)) {
            final String cronSeperator = " ";
            for (int i = 0; i < scheduleCount; i++) {
                final long second = 0L;
                String cronHead = second + cronSeperator + scheduleMinutes[i] + cronSeperator + scheduleHours[i] + cronSeperator;
                switch (scheduleType) {
                case DAILY:
                    //0 %d %d * * ?
                    triggerValue.append(cronHead + "* * ?|");
                    break;
                case WEEKLY:
                    //0 %d %d ? * %d
                    triggerValue.append(cronHead + "? * " + scheduleDaysOrWeeks[i] + "|");
                    break;
                case MONTHLY:
                    //0 %d %d %d * ?
                    triggerValue.append(cronHead + scheduleDaysOrWeeks[i] + " * ?|");
                    break;
                }
            }
            if (triggerValue.length() != 0) {
                triggerValue.deleteCharAt(triggerValue.length() - 1);
            }
        }
        return triggerValue.toString();
    }

    // For UI loading
    public void setEntityTriggerValue(String triggerValue) {
        if (TriggerType.FREQUENCY.equals(triggerType)) {
            long frequency = MINUTE_IN_SEC;
            try {
                frequency = Long.parseLong(triggerValue);
            } catch (NumberFormatException ex) { //NOSONAR
            }
            setFrequency(frequency);
        } else if (TriggerType.SCHEDULE.equals(triggerType)) {
            if (triggerValue == null) {
                return;
            }

            String[] crons = triggerValue.split("\\|");
            if (crons.length  == 0) {
                return;
            }
            scheduleType = getScheduleType(crons[0]);
            parseTriggerValues(crons);
        }
    }

    private void parseTriggerValues(String[] crons) {
        scheduleCount = Math.min(MAX_SCHEDULE_NUM, crons.length);
        scheduleHours = new long[(int) scheduleCount];
        scheduleMinutes = new long[(int) scheduleCount];
        scheduleDaysOrWeeks = new long[(int) scheduleCount];

        for (int i = 0; i < scheduleCount; i++) {
            String cron = crons[i];
            String[] es = cron.split("\\s+");
            if (es.length <= MAX_SCHEDULER_COUNT) {
                continue;
            }
            try {
                scheduleMinutes[i] = Long.parseLong(es[1]);
                scheduleHours[i] = Long.parseLong(es[2]);

                if (scheduleType == ScheduleType.MONTHLY) {
                    scheduleDaysOrWeeks[i] = Long.parseLong(es[3]); //NOSONAR
                } else if (scheduleType == ScheduleType.WEEKLY) {
                    scheduleDaysOrWeeks[i] = Long.parseLong(es[5]); //NOSONAR
                }
            } catch (NumberFormatException ex) { //NOSONAR
            }
        }
    }

    /*
     * Example:
     *      DAILY:      0 1 5 * * ?  (two stars)
     *      WEEKLY:     0 1 5 ? * 1  (doesn't end with ?)
     *      MONTHLY:    0 1 5 2 * ?
     */
    private ScheduleType getScheduleType(String cron) {
        if (cron.contains("* *")) {
            return ScheduleType.DAILY;
        }

        if (!cron.endsWith("?")) {
            return ScheduleType.WEEKLY;
        }

        return ScheduleType.MONTHLY;
    }

    public long[] getScheduleHours() {
        return scheduleHours;
    }

    public void setScheduleHours(long[] scheduleHours) {
        if (scheduleHours != null) {
            this.scheduleHours = new long[scheduleHours.length];
            for (int i = 0; i < scheduleHours.length; i++) {
                this.scheduleHours[i] = scheduleHours[i];
            }
        }
    }

    public long[] getScheduleMinutes() {
        return scheduleMinutes;
    }

    public void setScheduleMinutes(long[] scheduleMinutes) {
        if (scheduleMinutes != null) {
            this.scheduleMinutes = new long[scheduleMinutes.length];
            for (int i = 0; i < scheduleMinutes.length; i++) {
                this.scheduleMinutes[i] = scheduleMinutes[i];
            }
        }
    }

    public long[] getScheduleDaysOrWeeks() {
        return scheduleDaysOrWeeks;
    }

    public void setScheduleDaysOrWeeks(long[] scheduleDaysOrWeeks) {
        if (scheduleDaysOrWeeks != null) {
            this.scheduleDaysOrWeeks = new long[scheduleDaysOrWeeks.length];
            for (int i = 0; i < scheduleDaysOrWeeks.length; i++) {
                this.scheduleDaysOrWeeks[i] = scheduleDaysOrWeeks[i];
            }
        }
    }

    public long getScheduleCount() {
        return scheduleCount;
    }

    public void setScheduleCount(long scheduleCount) {
        this.scheduleCount = scheduleCount;
    }

    @Override
    public Long getKey() {
        return id;
    }

    public void setTriggeredByTask(long taskId) {
        this.triggeredByTask = taskId;
    }
    public void setTriggeredByPartner(long partnerId){
        this.triggeredByPartner = partnerId;
    }


    public long getTriggeredByTask() {
        return triggeredByTask;
    }

    public long getTriggeredByPartner() {
        return triggeredByPartner;
    }
}
