package com.ericsson.cms.epgmgmt.client.partner;

import java.util.List;
import java.util.Map;

import com.ericsson.cms.epgmgmt.client.Permissions;
import com.ericsson.cms.epgmgmt.client.i18n.EPGComponentConstants;
import com.ericsson.cms.epgmgmt.client.i18n.EPGComponentMessages;
import com.ericsson.cms.epgmgmt.client.model.PartnerConstantUtils;
import com.ericsson.cms.epgmgmt.client.model.UIEndPointEnum;
import com.ericsson.cms.epgmgmt.client.model.UIPartner;
import com.ericsson.cms.epgmgmt.client.model.UITask;
import com.ericsson.cms.epgmgmt.client.rpc.IEPGPartnerService;
import com.ericsson.cms.epgmgmt.client.rpc.IEPGPartnerServiceAsync;
import com.ericsson.cms.epgmgmt.client.rpc.IEPGTaskService;
import com.ericsson.cms.epgmgmt.client.rpc.IEPGTaskServiceAsync;
import com.ericsson.cms.epgmgmt.client.task.AbstractTaskTab;
import com.ericsson.cms.epgmgmt.client.utils.DefaultAsyncCallback;
import com.ericsson.cms.epgmgmt.client.utils.MessageAreaUtils;
import com.ericsson.cms.epgmgmt.client.utils.PromptWindow;
import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.History;
import com.google.gwt.user.client.Timer;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Image;
import com.google.gwt.user.client.ui.Label;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.application.NeptuneApplication;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.messagearea.MessageArea;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.feature.impl.AnchorTokenizer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.SimpleContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

public abstract class AbstractPartnerTabContainer extends Composite {

    protected final PartnerListPanel partnerList;
    protected final SimpleContainer mainPanel;
    protected AbstractPartnerCreateTab createTab;
    protected PartnerUpdateTab updateTab;

    protected final BusyIndicator busyIndicator = new BusyIndicator();
    protected final IEPGPartnerServiceAsync partnerService = GWT.create(IEPGPartnerService.class);
    protected final IEPGTaskServiceAsync taskService = GWT.create(IEPGTaskService.class);
    protected final EPGComponentConstants constants = (EPGComponentConstants) GWT.create(EPGComponentConstants.class);
    protected static final String STYLE_TABCONTAINER = "epg-tabContainer";
    protected Long partnerId = 0L;
    protected final MessageArea messageArea;
    protected final EPGComponentMessages constantsMsg = (EPGComponentMessages) GWT.create(EPGComponentMessages.class);
    protected UIPartner partner;

    public AbstractPartnerTabContainer(PartnerListPanel partnerList, MessageArea messageArea) {
        this.partner = getEmptyPartner();
        this.partnerList = partnerList;
        this.messageArea = messageArea;
        NeptuneApplication.getApplication().getSecurity();
        mainPanel = new SimpleContainer();
        mainPanel.setWidth("100%");
        mainPanel.setHeight("100%");

        initWidgets();
        initWidget(mainPanel);
    }

    private void initWidgets() {
        createTab = getCreateTab();
        createTab.setMessageArea(messageArea);

        updateTab = new PartnerUpdateTab(this, partner);
        updateTab.setMessageArea(messageArea);
        updateTab.init();
    }

    public void setHeight(int height) {
        createTab.setHeight(height);
        updateTab.setHeight(height);
    }

    /**
     * Validate the plugin properties, if no violation occur then submit the task.
     */
    public void validateAndSubmit() {
        final UITask task = createTab.populateTask();
        taskService.validateTask(task, new NeptuneAsyncCallback<List<String>>() {
            @Override
            public void onNeptuneFailure(Throwable caught) {
                MessageAreaUtils.handleException(caught, messageArea);
            }

            @Override
            public void onNeptuneSuccess(List<String> result) {
                if (result.size() > 0) {
                    MessageAreaUtils.displayErrorMessages("", result, "", messageArea);
                    return;
                }
                findConflictEndpointAndSubmit(task);
            }
        });
    }

    private void findConflictEndpointAndSubmit(final UITask task) {
        taskService.getConflictEndpoint(task, new DefaultAsyncCallback<UITask>(messageArea) {
            @Override
            public void onNeptuneSuccess(UITask conflicted) {
                if (conflicted == null) {
                    onCommit();
                    return;
                }

                String msg = constantsMsg.conflictFileEndpoint(conflicted.getName(), conflicted.getPartnerType(),
                        conflicted.getPartnerName());
                if (UIEndPointEnum.file.name().equals(task.getEndpoint().getEndpointType())) {
                    MessageAreaUtils.displayErrorMessage("", msg, "", messageArea);
                } else if (Window.confirm(msg)) {
                    onCommit();
                }
            }
        });
    }

    protected void onCommit() {
        clear();
        isButtonVisible();

        if (partnerId == 0) {
            onCommitCreate();
        } else {
            onCommitUpdate();
        }
    }

    protected void onCommitCreate() {
        busyIndicator.center();
        partner = createTab.populatePartner();
        partnerService.createPartner(partner, new NeptuneAsyncCallback<Long>() {
            @Override
            public void onNeptuneFailure(Throwable caught) {
                MessageAreaUtils.handleException(caught, messageArea);
                busyIndicator.hide();
            }

            @Override
            public void onNeptuneSuccess(final Long partnerId) {
                UITask task = createTab.populateTask();
                task.setPartnerId(partnerId);
                taskService.createTask(task, new NeptuneAsyncCallback<Long>() {
                    @Override
                    public void onNeptuneFailure(Throwable caught) {
                        busyIndicator.hide();
                        MessageAreaUtils.handleException(caught, messageArea);
                    }

                    @Override
                    public void onNeptuneSuccess(Long result) {
                        setConfirmationMessage(constantsMsg.partnerSuccess(PartnerConstantUtils
                                .getPartnerType(getPartner())));
                        Timer sleepTimer = new Timer() {
                            @Override
                            public void run() {
                                setPartnerId(partnerId);
                                viewCreatePartnerDetails(partnerId);
                                busyIndicator.hide();
                            };
                        };
                        sleepTimer.schedule(5000);
                        updateTab.saveSuccess();
                    }
                });
            }
        });
    }

    abstract protected void onCommitUpdate();

    /**
     * @param partnerId
     */
    public void viewCreatePartnerDetails(Long partnerId) {
        this.clear();
        this.partnerId = partnerId;
        History.newItem(getAnchorName() + "." + "id" + "=" + String.valueOf(partnerId), false);
        partnerList.reload(partnerId, null);
        isButtonVisible();
        getPartnerDetail(partnerId);
    }

    public void viewCreatePartner() {
        History.newItem(getAnchorName(), false);
        partnerId = 0L;
        clear();
        createTab.reload();
        displayCreate();
    }

    public void displayEmpty() {
        clear();
        mainPanel.clear();
    }

    private void displayCreate() {
        clear();
        mainPanel.setWidget(createTab);
    }

    public void displayAnMessagePanel() {
        clear();

        // Create an empty panel
        String msg = constantsMsg.noPartnerExist(this.partner.getPartnerType().toString());

        VerticalContainer messageContainer = new VerticalContainer();
        Image emptyImageRow = new Image("cms_epgmgmt_ui/images/layer-indicator-blank.png");
        Image emptyImageCollumn = new Image("cms_epgmgmt_ui/images/layer-indicator-blank.png");
        HorizontalPanel messagePanel = new HorizontalPanel();
        messagePanel.add(emptyImageCollumn);
        messagePanel.add(new Label(msg));

        messageContainer.add(emptyImageRow);
        messageContainer.add(messagePanel);
        mainPanel.setWidget(messageContainer);
    }

    private void displayUpdate() {
        clear();
        mainPanel.setWidget(updateTab);
    }

    abstract protected void onDelete();

    public void disableButtons() {
        updateTab.disableButtons();
    }

    public void clear() {
        messageArea.reset();
        setConfirmationMessage("");
    }

    public void loadPartnerDetails(Long partnerId) {
        // set anchor
        this.clear();
        this.partnerId = partnerId;
        History.newItem(getAnchorName() + "." + "id" + "=" + String.valueOf(partnerId), false);
        isButtonVisible();
        getPartnerDetail(partnerId);
    }

    public void isButtonVisible() {
        createTab.setButtonVisible(ClientAuthorizationManager.isAuthorized(Permissions.EPG_CREATE));
        updateTab.setButtonVisible(ClientAuthorizationManager.isAuthorized(Permissions.EPG_MODIFY));
    }

    protected String saveWarningMessage() {
        return constantsMsg.partnerSaveWarningMessage(PartnerConstantUtils.getPartnerActionType(partner));
    }

    private void getPartnerDetail(Long partnerId) {

        busyIndicator.center();
        partnerService.getPartner(partnerId, new NeptuneAsyncCallback<UIPartner>() {
            @Override
            public void onNeptuneFailure(Throwable caught) {
                Window.alert(constants.loadingError() + " | " + constants.reason() + ": "
                        + (caught != null ? caught.getLocalizedMessage() : ""));
                busyIndicator.hide();
            }

            @Override
            public void onNeptuneSuccess(UIPartner partner) {
                busyIndicator.hide();

                if (null == partner) {
                    return;
                }

                displayUpdate();

                partnerList.reload(partner.getId(), null);

                // set id of TabContainer
                setPartnerId(partner.getId());

                updateTab.refreshPartner(partner);
                setPartner(partner);

                if (!partner.isActive()) {
                    PromptWindow.getInstance().prompt(
                            constantsMsg.cannotEditTaskWhilePartnerIsDisabled(partner.getPartnerType().toString()),
                            3000);
                }
            }
        });
    }

    private boolean isBlank(String s) {
        return (s == null) || s.isEmpty();
    }

    public void anchorChanged(String anchor) {
        if (isBlank(anchor)) {
            reset();
            partnerList.selectFirst(true);
        } else {
            AnchorTokenizer at = new AnchorTokenizer();
            Map<String, String> paramMap = at.parseAnchor(anchor);
            Long id = Long.parseLong(paramMap.get("id"));

            if ((id != null) && (id >= 0)) {
                getPartnerDetail(id);
            }
        }
    }

    protected void setConfirmationMessage(String message) {
        messageArea.setInfoMessage(message);
        updateTabSize();
    }

    protected void updateTabSize() {
        createTab.updateScrollPanelSize();
        updateTab.updateTabSize();
    }

    private void reset() {
        messageArea.reset();
    }

    public UIPartner getPartner() {
        return partner;
    }

    public void setPartner(UIPartner partner) {
        this.partner = partner;
    }

    public Long getPartnerId() {
        return partnerId;
    }

    public void setPartnerId(Long partnerId) {
        this.partnerId = partnerId;
    }

    @Override
    protected void onLoad() {
        super.onLoad();
    };

    @Override
    protected void onUnload() {
        if (busyIndicator != null) {
            busyIndicator.hide();
        }

        super.onUnload();
    }

    /**
     * Return the instance of the create tab.
     * 
     * @return
     */
    protected abstract AbstractPartnerCreateTab getCreateTab();

    /**
     * Return the instance of the task tab.
     * 
     * @param partner
     * @return
     */
    protected abstract AbstractTaskTab getTaskTab(UIPartner partner);

    /**
     * Return the model of an empty partner.
     * 
     * @return
     */
    protected abstract UIPartner getEmptyPartner();

    /**
     * Get the anchor name according to the partner type.
     * 
     * @return
     */
    protected abstract String getAnchorName();
}
