package com.ericsson.cms.epgmgmt.client.partner;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import com.ericsson.cms.epgmgmt.client.SyncCallback;
import com.ericsson.cms.epgmgmt.client.i18n.EPGComponentConstants;
import com.ericsson.cms.epgmgmt.client.model.UIPartner;
import com.ericsson.cms.epgmgmt.client.model.enumeration.PartnerType;
import com.ericsson.cms.epgmgmt.client.rpc.IEPGPartnerService;
import com.ericsson.cms.epgmgmt.client.rpc.IEPGPartnerServiceAsync;
import com.google.gwt.core.client.GWT;
import com.google.gwt.dom.client.Style.Unit;
import com.google.gwt.event.logical.shared.SelectionEvent;
import com.google.gwt.event.logical.shared.SelectionHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.HasVerticalAlignment;
import com.google.gwt.user.client.ui.Image;
import com.google.gwt.user.client.ui.TreeItem;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.security.NeptuneSecurity;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TreeWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;

public class PartnerListPanel extends TreeWidget {

    private final PartnerType type;
    private TreeItem root;
    private final BusyIndicator busyIndicator = new BusyIndicator();
    private AbstractPartnerTabContainer tabContainer;
    private final EPGComponentConstants constants = (EPGComponentConstants) GWT.create(EPGComponentConstants.class);
    private final IEPGPartnerServiceAsync service = GWT.create(IEPGPartnerService.class);

    public PartnerListPanel(NeptuneSecurity security, PartnerType type) {
        this.type = type;
        addSelectionHandler(new SelectionHandler<TreeItem>() {
            @Override
            public void onSelection(SelectionEvent<TreeItem> event) {
                Long id = 0L;
                TreeItem selectedItem = event.getSelectedItem();
                UIPartner partner = (UIPartner) selectedItem.getUserObject();
                if (partner == null) {
                    return;
                }
                id = partner.getId();

                tabContainer.loadPartnerDetails(id);
            }
        });
        root = new PartnerListItem("");
        addItem(root);
    }

    private class PartnerListItem extends TreeItem {

        private static final String STYLE_ITEM_SELECTED = "epg-treeItem-selected";
        private static final String RELATIVE_POSITION_LEFT_20PX = "epg-relative-position-left-20px";

        public PartnerListItem(String text) {
            super(new HorizontalContainer());
        }

        public PartnerListItem(UIPartner nodeData) {
            super(new HorizontalContainer());
            setUserObject(nodeData);
            init();
        }

        private void init() {
            // container for treeItem
            HorizontalContainer container = (HorizontalContainer) this.getWidget();
            container.setStyleName(RELATIVE_POSITION_LEFT_20PX);
            container.setSpacing(2);
            container.setVerticalAlignment(HasVerticalAlignment.ALIGN_MIDDLE);
            UIPartner nodeData = (UIPartner) this.getUserObject();
            if(!nodeData.isActive()){
                Image pro = new Image("cms_epgmgmt_ui/images/prohibit.png");
                container.add(pro);
            }
            container.add(new Image(initPicture()));
            container.add(initLabel());
        }

        private LabelWidget initLabel() {
            UIPartner nodeData = (UIPartner) this.getUserObject();
            LabelWidget label = new LabelWidget();
            label.setText(nodeData.getShortName());
            label.setTitle(nodeData.getName());
            label.getElement().getStyle().setPaddingTop(4, Unit.PX);
            label.getElement().getStyle().setPaddingBottom(4, Unit.PX);
            label.setWidth("100%");
            return label;
        }


        private String initPicture() {
            UIPartner partner = (UIPartner)this.getUserObject();
            String url;
            if(partner.getPartnerType()==PartnerType.Provider){
                url = "cms_epgmgmt_ui/images/provider";
            }else{
                url ="cms_epgmgmt_ui/images/receiver";
            }
            if(partner.isActive()){
                url += ".png";
            }else{
                url += "-disabled.png";
            }
            return url;
        }

        @Override
        public void setSelected(boolean selected) {
            super.setSelected(selected);
            if (selected) {
                getWidget().addStyleName(STYLE_ITEM_SELECTED);
            } else {
                getWidget().removeStyleName(STYLE_ITEM_SELECTED);
            }
        }
    }

    /**
     * @param partnerId
     * @param tabid
     * @param isParent
     */
    public void reload(final Long partnerId, final Integer tabId) {
        reload(partnerId, tabId, null);
    }

    public void reload(final Long partnerId, final Integer tabId, SyncCallback callback) {
        busyIndicator.center();

        if (type == PartnerType.Provider) {
            service.getProviders(new LoadPartnersCallback(partnerId, tabId, callback));
        } else {
            service.getPublishers(new LoadPartnersCallback(partnerId, tabId, callback));
        }
    }

    private class LoadPartnersCallback extends NeptuneAsyncCallback<List<UIPartner>> {

        private final Long partnerId;
        private final Integer tabId;
        private final SyncCallback callback;

        public LoadPartnersCallback(Long partnerId, Integer tabId, SyncCallback callback) {
            this.partnerId = partnerId;
            this.tabId = tabId;
            this.callback = callback;
        }

        @Override
        public void onNeptuneFailure(Throwable caught) {
            busyIndicator.hide();
            Window.alert(constants.loadingError() + " | " + constants.reason() + ": "
                    + (caught != null ? caught.getLocalizedMessage() : ""));
        }

        @Override
        public void onNeptuneSuccess(List<UIPartner> result) {
            renderPartnerList(result);
            if (partnerId != null) {
                selectPartnerId(partnerId, tabId);
            }

            if (callback != null) {
                callback.execute();
            }

            busyIndicator.hide();
        }
    }

    private void renderPartnerList(List<UIPartner> partners) {
        removeItems();
        Collections.sort(partners, new Comparator<UIPartner>() {
            @Override
            public int compare(UIPartner o1, UIPartner o2) {
                return o1.getName().compareTo(o2.getName());
            }
        });

        for (UIPartner partner : partners) {
            PartnerListItem item = new PartnerListItem(partner);
            if (!partner.isActive()) {
                item.addStyleName("epg-disabled");
            }
            addItem(item);
        }
    }

    private void selectPartnerId(long id, Integer parentSiteId) {
        PartnerListItem item = null;
        item = findSelectedPartnerById(id, null);

        if (item != null) {
            item.getTree().setSelectedItem(item, false);
            item.getTree().ensureSelectedItemVisible();
        }
    }

    private PartnerListItem findSelectedPartnerById(long id, TreeItem parentItem) {
        PartnerListItem selectedItem = null;
        if (parentItem != null) {
            int numChildItems = parentItem.getChildCount();
            for (int i = 0; i < numChildItems; i++) {
                TreeItem childItem = parentItem.getChild(i);
                if (childItem instanceof PartnerListItem) {
                    UIPartner uiNode = (UIPartner) childItem.getUserObject();
                    if (uiNode != null) {
                        long candidateId = uiNode.getId();
                        if (id == candidateId) {
                            selectedItem = (PartnerListItem) childItem;
                        } else {
                            selectedItem = findSelectedPartnerById(id, childItem);
                        }
                        if (selectedItem != null) {
                            break;
                        }
                    }
                }
            }
        } else {
            int numRootItems = PartnerListPanel.this.getItemCount();
            for (int i = 0; i < numRootItems; i++) {
                TreeItem rootItem = PartnerListPanel.this.getItem(i);
                if (rootItem instanceof PartnerListItem) {
                    UIPartner uiNode = (UIPartner) rootItem.getUserObject();
                    if (uiNode != null) {
                        long candidateId = uiNode.getId();
                        if (id == candidateId) {
                            selectedItem = (PartnerListItem) rootItem;
                        } else {
                            selectedItem = findSelectedPartnerById(id, rootItem);
                        }
                        if (selectedItem != null) {
                            break;
                        }
                    }
                }
            }
        }

        return selectedItem;
    }

    /**
     * Select the first item.
     * 
     * @param isWhenSingleItem
     *            Only when there's 1 item in the list then it will be selected.
     */
    public void selectFirst(final boolean isWhenSingleItem) {
        reload(null, null, new SyncCallback() {
            @Override
            public void execute() {
                if (isWhenSingleItem && getItemCount() == 1 || !isWhenSingleItem && getItemCount() > 0) {
                    setSelectedItem(getItem(0), true);
                } else {
                    tabContainer.displayAnMessagePanel();
                }
            }
        });
    }

    public void setTabContainer(AbstractPartnerTabContainer tabContainer) {
        this.tabContainer = tabContainer;
    }
}
