package com.ericsson.cms.epgmgmt.client.plugin;

import java.util.List;

import com.ericsson.cms.epgmgmt.client.Permissions;
import com.ericsson.cms.epgmgmt.client.i18n.EPGComponentConstants;
import com.ericsson.cms.epgmgmt.client.i18n.EPGComponentMessages;
import com.ericsson.cms.epgmgmt.client.model.UIPluginInfo;
import com.ericsson.cms.epgmgmt.client.model.UIPluginMgmtResult;
import com.ericsson.cms.epgmgmt.client.plugin.EPGPluginDataProvider.IBusyNotifier;
import com.ericsson.cms.epgmgmt.client.rpc.IPluginMgmtService;
import com.ericsson.cms.epgmgmt.client.rpc.IPluginMgmtServiceAsync;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.ScrollPanel;
import com.gwtext.client.widgets.form.Label;
import com.tandbergtv.neptune.widgettoolkit.client.application.ClientAuthorizationManager;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.HeaderPanel;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.messagearea.MessageArea;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Table;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.TabContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.style.StyleNames;

public class EPGPluginListViewPanel extends Composite {
    // spacing between the table and message panel
    private static final int SPACING = 2;
    private static final String STYLE_NAME = "content-EPGPanel";
    private static final String BUTTON_PANEL_STYLE = "epg-tabContainer-buttonPanel";
    private static final String NEWLINE = "\n";
    private static final String COMMA = ", ";

    public static final String ACTION_UNINSTALL = "uninstall";
    public static final String ACTION_DEACTIVATE = "deactivate";
    public static final String ACTION_ACTIVATE = "activate";

    private final EPGComponentConstants constants = (EPGComponentConstants) GWT.create(EPGComponentConstants.class);
    private final MessageArea messageArea = new MessageArea();;
    private Table<Long, UIPluginInfo> pluginTable;
    private final BusyIndicator busyIndicator = new BusyIndicator();

    private final IPluginMgmtServiceAsync pluginMgmtService = GWT.create(IPluginMgmtService.class);
    private final EPGComponentMessages epgMessages = GWT.create(EPGComponentMessages.class);

    public EPGPluginListViewPanel() {
        if (!ClientAuthorizationManager.isAuthorized(Permissions.EPG_PLUGIN_MGMT)) {
            Window.alert(epgMessages.permissionDenied());
            this.initWidget(new Label());
            return;
        }
        init();
    }

    private void init() {
        VerticalContainer mainContainer = new VerticalContainer();
        mainContainer.setSpacing(SPACING);
        mainContainer.setStylePrimaryName(STYLE_NAME);
        HeaderPanel header = new HeaderPanel(constants.epgFormatHeader());

        // Add widgets to the main container
        mainContainer.add(header);
        mainContainer.add(createListPanel());
        mainContainer.add(createButtonPanel());
        messageArea.reset();
        mainContainer.add(messageArea);
        refreshTable();
        this.initWidget(mainContainer);
    }

    private void refreshTable() {
        if (!pluginTable.isInitialized()) {
            pluginTable.initialize(new NeptuneAsyncCallback<Void>() {
                @Override
                public void onNeptuneFailure(Throwable caught) {
                    messageArea.setErrorMessage(caught.getLocalizedMessage());
                }

                @Override
                public void onNeptuneSuccess(Void result) {
                    pluginTable.refresh();
                }
            });
        } else {
            pluginTable.refresh();
        }
    }

    private TabContainer createListPanel() {
        TabContainer container = new TabContainer();
        container.setWidth("100%");
        container.setHeight("100%");
        container.addStyleName("epg-tabContainer");
        createTable();
        container.add(new ScrollPanel(pluginTable), constants.epgFormatHeader());
        container.selectTab(0);
        return container;
    }

    private Table<Long, UIPluginInfo> createTable() {
        EPGPluginDataProvider dataProvider = new EPGPluginDataProvider(this);
        dataProvider.setBusyNotifier(new IBusyNotifier() {
            @Override
            public void startingToBeBusy() {
                busyIndicator.center();
            }

            @Override
            public void doneBeingBusy() {
                busyIndicator.hide();
            }
        });
        pluginTable = new Table<Long, UIPluginInfo>(dataProvider);
        return pluginTable;
    }

    private HorizontalContainer createButtonPanel() {
        HorizontalContainer buttonPanel = new HorizontalContainer();
        buttonPanel.setStyleName(BUTTON_PANEL_STYLE);
        // install
        ButtonWidget installButton = new ButtonWidget(constants.buttonLabelInstall());
        installButton.addStyleDependentName(StyleNames.ACT_TOWARDS_SAVE_BUTTON_STYLE);
        installButton.addClickHandler(new ClickHandler(){
            @Override
            public void onClick(ClickEvent event) {
                PluginFileUploadpopup popup = new PluginFileUploadpopup(pluginTable);
                popup.center();
            }            
        });
        buttonPanel.add(installButton);
        // refresh
        ButtonWidget refreshButton = new ButtonWidget(constants.buttonLabelRefresh());
        refreshButton.addStyleDependentName(StyleNames.ACTION_BUTTON_STYLE);
        refreshButton.addClickHandler(new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {
                pluginTable.refresh();
            }
        });
        buttonPanel.add(refreshButton);
        
        return buttonPanel;
    }

    public void uninstallPlugin(final UIPluginInfo row) {
        busyIndicator.center();
        pluginMgmtService.uninstallPlugin(row.getId(), new UninstallAsyncCallback(row, ACTION_UNINSTALL));
    }

    public void deactivatePlugin(final UIPluginInfo row) {
        busyIndicator.center();
        pluginMgmtService.deactivatePlugin(row.getId(), new UninstallAsyncCallback(row, ACTION_DEACTIVATE));
    }

    public void checkPlugin(final UIPluginInfo row, String action) {
        busyIndicator.center();
        pluginMgmtService.checkPlugin(row.getId(), new UninstallAsyncCallback(row, action));
    }

    public void activatePlugin(final UIPluginInfo row) {
        busyIndicator.center();
        pluginMgmtService.activiatePlugin(row.getId(), new UninstallAsyncCallback(row, ACTION_ACTIVATE));
    }

    private String buildMessage(String message, List<UIPluginMgmtResult.TaskVO> tasks) {
        StringBuffer buffer = new StringBuffer();
        buffer.append(message);
        buffer.append(NEWLINE);
        for (UIPluginMgmtResult.TaskVO partner : tasks) {
            buffer.append(partner.getName());
            buffer.append(COMMA);
        }
        buffer.setLength(buffer.length() - 2);
        return buffer.toString();
    }

    class UninstallAsyncCallback implements AsyncCallback<UIPluginMgmtResult> {
        private final UIPluginInfo row;
        private final String action;

        public UninstallAsyncCallback(UIPluginInfo row, String type) {
            this.row = row;
            this.action = type;
        }

        @Override
        public void onFailure(Throwable caught) {
            messageArea.setErrorMessage(caught.getLocalizedMessage());
            busyIndicator.hide();
        }

        @Override
        public void onSuccess(UIPluginMgmtResult result) {
            busyIndicator.hide();
            String msg;
            switch (result.getResult()) {
            case UIPluginMgmtResult.COMFIRM:
                if (result.getTasks().isEmpty()) {
                    msg = epgMessages.confMgmtPluginNoPartner(action, row.getFormatName());
                } else {
                    msg = epgMessages.confirmMgmtPlugin(action, row.getFormatName(), row.getType().toLowerCase());
                }
                if (Window.confirm(buildMessage(msg, result.getTasks()))) {
                    if (action.equals(ACTION_DEACTIVATE)) {
                        deactivatePlugin(row);
                    } else {
                        uninstallPlugin(row);
                    }
                } else {
                    pluginTable.refresh();
                }
                break;
            case UIPluginMgmtResult.HAS_RUNNING_JOB:
                msg = epgMessages.alertRunningJob(row.getType().toLowerCase(), action);
                Window.alert(buildMessage(msg, result.getRunningTasks()));
                pluginTable.refresh();
                break;
            case UIPluginMgmtResult.OTHER_VERSION_ACTIVE:
                Window.alert(epgMessages.alertOtherVerActive());
                pluginTable.refresh();
                break;
            case UIPluginMgmtResult.SUCC:
                pluginTable.refresh();
            }
        }
    }
}
