package com.ericsson.cms.epgmgmt.client.task;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map.Entry;
import java.util.Set;

import com.ericsson.cms.epgmgmt.client.i18n.EPGComponentConstants;
import com.ericsson.cms.epgmgmt.client.i18n.EPGComponentMessages;
import com.ericsson.cms.epgmgmt.client.model.UIOrderedTask;
import com.ericsson.cms.epgmgmt.client.model.UIPartner;
import com.ericsson.cms.epgmgmt.client.model.UIPluginProperty;
import com.ericsson.cms.epgmgmt.client.model.UITask;
import com.ericsson.cms.epgmgmt.client.model.UITaskSetting;
import com.ericsson.cms.epgmgmt.client.model.enumeration.JobStatus;
import com.ericsson.cms.epgmgmt.client.model.enumeration.PartnerType;
import com.ericsson.cms.epgmgmt.client.model.util.TableUtils;
import com.ericsson.cms.epgmgmt.client.rpc.IEPGJobService;
import com.ericsson.cms.epgmgmt.client.rpc.IEPGJobServiceAsync;
import com.ericsson.cms.epgmgmt.client.rpc.IEPGPluginManagerService;
import com.ericsson.cms.epgmgmt.client.rpc.IEPGPluginManagerServiceAsync;
import com.ericsson.cms.epgmgmt.client.rpc.IEPGTaskService;
import com.ericsson.cms.epgmgmt.client.rpc.IEPGTaskServiceAsync;
import com.ericsson.cms.epgmgmt.client.utils.DateConvert;
import com.ericsson.cms.epgmgmt.client.utils.DefaultAsyncCallback;
import com.ericsson.cms.epgmgmt.client.utils.MessageAreaUtils;
import com.ericsson.cms.epgmgmt.client.widget.AlignableCheckBoxView;
import com.ericsson.cms.epgmgmt.client.widget.CustomizableHorizontalView;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.HasHorizontalAlignment;
import com.google.gwt.user.client.ui.Image;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.CheckBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.LabelWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.messagearea.MessageArea;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.Column;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.ColumnBase;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.DataProvider;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.TableConstants;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.View;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.table.view.label.LabelStringView;

public abstract class AbstractTaskDataProvider implements DataProvider<Long, UIOrderedTask> {

    protected static final String STYLE_EPG_DISABLED = "epg-disabled";
    private static final String TASK_NAME = "taskName";
    private static final String PROPERTY_TRIGGER_TYPE = "trigger.type";
    protected static final String TRIGGER_TYPE = "triggerType";
    private static final String STATUS = "Status";
    private static final String PROPERTY_RUNNING_NODE = "runningNode";
    private static final String NEXT_EXECUTION = "nextExecutionTime";
    private static final String PROPERTY_ACTIVATED = "activated";
    private static final String PROPERTY_ACTION = "action";
    private static final String LAST_EXECUTION = "lastExecution";
    private static final String ORDER = "order";
    protected UIPartner partner;
    private final EPGComponentConstants constants = GWT.create(EPGComponentConstants.class);
    protected final EPGComponentMessages messages = GWT.create(EPGComponentMessages.class);
    private final IEPGJobServiceAsync jobService = GWT.create(IEPGJobService.class);
    protected final IEPGTaskServiceAsync taskService = GWT.create(IEPGTaskService.class);
    private final IEPGPluginManagerServiceAsync pluginService = GWT.create(IEPGPluginManagerService.class);
    private final List<Column<?, UIOrderedTask>> columns;
    private BusyIndicator busyIndicator;
    protected MessageArea messageArea;
    final AbstractTaskListViewPanel parent;
    protected static final String PROVIDER = "provider";
    protected static final String INGEST_TASK = "ingestTask";

    public AbstractTaskDataProvider(UIPartner partner, AbstractTaskListViewPanel parent) {
        this.parent = parent;
        this.partner = partner;
        columns = new ArrayList<Column<?, UIOrderedTask>>();
        columns.add(createLinkColumn(TASK_NAME, constants.providerTaskNameLabel()));
        if (partner.getPartnerType() == PartnerType.Provider) {
            columns.add(createOrderColumn(ORDER, constants.providerTaskOrderLabel()));
        }
        columns.add(createFormatColumn());
        columns.add(createLabelColumn(PROPERTY_TRIGGER_TYPE, constants.providerTaskTriggerTypeLable(), TRIGGER_TYPE));
        columns.add(createStatusColumn());
        columns.add(createNodeColumn());
        columns.add(createDateColumn(LAST_EXECUTION, constants.providerTaskLastExecutionLable(), LAST_EXECUTION,
                constants.activityLogTimeFormat()));
        columns.add(TableUtils.<UIOrderedTask> createDateColumn(NEXT_EXECUTION,
                constants.providerTaskNextExecutionLable(), NEXT_EXECUTION, constants.activityLogTimeFormat()));
        columns.add(createEnableColumn());
        columns.add(createActionColumn());
    }

    private Column<?, UIOrderedTask> createFormatColumn() {
        ColumnBase<String, UIOrderedTask> c = new ColumnBase<String, UIOrderedTask>("formatName",
                constants.taskFormatNameLabel()) {
            @Override
            public View<String> getView(final UIOrderedTask task) {
                final CustomizableHorizontalView v = new CustomizableHorizontalView();
                Object value = task.getProperty("formatName");
                LabelWidget l = new LabelWidget(value == null ? "" : value.toString());
                if (!partner.isActive() || !task.isEnabled()) {
                    l.setStyleName(STYLE_EPG_DISABLED);
                }
                v.getContainer().add(l);

                notifyBusyStatus(true);

                pluginService.isPluginActive(task.getFormatId(), new NeptuneAsyncCallback<Boolean>() {

                    @Override
                    public void onNeptuneFailure(Throwable caught) {
                        MessageAreaUtils.handleException(caught, messageArea);
                        notifyBusyStatus(false);
                    }

                    @Override
                    public void onNeptuneSuccess(Boolean result) {
                        if (!result) {
                            ImageWidget img = new ImageWidget("cms_epgmgmt_ui/images/warning.png");
                            img.setTitle(messages.updateNeededCauseFormatIsDisabled(task.getFormatName()));
                            v.getContainer().insert(img, 0);
                        }
                        notifyBusyStatus(false);
                    }
                });

                return v;
            }
        };
        c.setCellStyle(TableConstants.STYLE_DATACELL_TEXT);
        return c;
    }

    protected ColumnBase<String, UIOrderedTask> createLabelColumn(String name, String label, final String propertyName) {
        ColumnBase<String, UIOrderedTask> c = new ColumnBase<String, UIOrderedTask>(name, label) {
            @Override
            public View<String> getView(UIOrderedTask task) {
                Object value = task.getProperty(propertyName);
                LabelStringView v = new LabelStringView(value == null ? "" : value.toString());
                if (!partner.isActive() || !task.isEnabled()) {
                    v.setStyleName(STYLE_EPG_DISABLED);
                }
                return v;
            }
        };
        c.setCellStyle(TableConstants.STYLE_DATACELL_TEXT);
        return c;
    }

    private ColumnBase<String, UIOrderedTask> createDateColumn(String name, String label, final String propertyName,
            final String format) {
        ColumnBase<String, UIOrderedTask> c = new ColumnBase<String, UIOrderedTask>(name, label) {
            @Override
            public View<String> getView(UIOrderedTask task) {
                Object value = task.getProperty(propertyName);
                LabelStringView v = new LabelStringView(value == null ? "" : DateConvert.formatDate(format, new Date(
                        (Long) value)));
                if (!partner.isActive() || !task.isEnabled()) {
                    v.setStyleName(STYLE_EPG_DISABLED);
                }
                return v;
            }
        };
        c.setCellStyle(TableConstants.STYLE_DATACELL_TEXT);
        return c;
    }

    private ColumnBase<String, UIOrderedTask> createLinkColumn(final String name, final String label) {
        ColumnBase<String, UIOrderedTask> col = new ColumnBase<String, UIOrderedTask>(name, label) {
            @Override
            public View<String> getView(final UIOrderedTask task) {
                CustomizableHorizontalView v = new CustomizableHorizontalView();
                if (partner.getPartnerType() == PartnerType.Provider) {
                    for (int i = 0; i < (task.getLevel() - 1); i++) {
                        v.getContainer().add(
                                i == (task.getLevel() - 2) ? new Image("cms_epgmgmt_ui/images/layer-indicator.png")
                                : new Image("cms_epgmgmt_ui/images/blank.png"));
                    }
                }

                if (!task.isEnabled()) {
                    v.getContainer().add(new ImageWidget("cms_epgmgmt_ui/images/prohibit.png"));
                }

                LabelWidget l = new LabelWidget(task.getName());
                v.getContainer().add(l);

                l.setStyleName("epg-table-data-link"
                        + (partner.isActive() && task.isEnabled() ? "" : " " + STYLE_EPG_DISABLED));
                l.addClickHandler(new ClickHandler() {
                    @Override
                    public void onClick(ClickEvent event) {
                        parent.getMessageArea().reset();
                        AbstractTaskDialog taskDialog = getTaskDialog(task, partner, parent);
                        parent.setTaskDialog(taskDialog);
                        taskDialog.center();
                    }
                });
                return v;
            }
        };
        col.setCellStyle(TableConstants.STYLE_DATACELL_TEXT);
        return col;
    }

    private Column<?, UIOrderedTask> createOrderColumn(String name, String label) {
        ColumnBase<String, UIOrderedTask> col = new ColumnBase<String, UIOrderedTask>(name, label) {
            @Override
            public View<String> getView(final UIOrderedTask task) {
                if ((task.getOrder() == 1) && task.getChildTasks().isEmpty()) {
                    return new LabelStringView("");
                }
                LabelStringView v = new LabelStringView(String.valueOf(task.getOrder()));
                v.getWidget().setHorizontalAlignment(HasHorizontalAlignment.ALIGN_CENTER);
                if (!partner.isActive() || !task.isEnabled()) {
                    v.setStyleName(STYLE_EPG_DISABLED);
                }
                return v;
            }
        };
        col.setCellStyle(TableConstants.STYLE_DATACELL_TEXT);
        return col;
    }

    private ColumnBase<String, UIOrderedTask> createStatusColumn() {
        ColumnBase<String, UIOrderedTask> textColumn = new TaskStatusColumn(STATUS,
                constants.providerTaskStatusLable(), true);
        textColumn.setCellStyle(TableConstants.STYLE_DATACELL_TEXT);
        return textColumn;
    }

    private ColumnBase<String, UIOrderedTask> createNodeColumn() {
        ColumnBase<String, UIOrderedTask> column = new ColumnBase<String, UIOrderedTask>(PROPERTY_RUNNING_NODE,
                constants.providerTaskNodeLabel()) {
            @Override
            public View<String> getView(UIOrderedTask task) {
                return new LabelStringView(task.getJobStatus().getNodeAddress() == null ? "" : task.getJobStatus()
                        .getNodeAddress());
            }
        };
        column.setCellStyle(TableConstants.STYLE_DATACELL_TEXT);
        return column;
    }

    private Column<?, UIOrderedTask> createActionColumn() {
        ColumnBase<String, UIOrderedTask> actionColumn = new TaskActionColumn(this, PROPERTY_ACTION,
                constants.actionLabel());
        actionColumn.setCellStyle(TableConstants.STYLE_DATACELL_TEXT);
        return actionColumn;
    }

    private ColumnBase<Boolean, UIOrderedTask> createEnableColumn() {
        ColumnBase<Boolean, UIOrderedTask> activeColumn = new ColumnBase<Boolean, UIOrderedTask>(PROPERTY_ACTIVATED,
                constants.providerTaskEnableLabel()) {
            @Override
            public View<Boolean> getView(final UIOrderedTask task) {
                AlignableCheckBoxView view = new AlignableCheckBoxView(task.isEnabled(), true,
                        HasHorizontalAlignment.ALIGN_CENTER);
                if (!partner.isActive()) {
                    view.getCheckbox().setEnabled(false);
                    return view;
                }
                view.getCheckbox().addValueChangeHandler(new ActivateValueChangeHandler(task));
                return view;
            }
        };
        activeColumn.setCellStyle(TableConstants.STYLE_DATACELL_TEXT);
        return activeColumn;
    }

    private void enableTask(long taskId, final boolean isEnabled) {
        taskService.activateTask(taskId, isEnabled, new AsyncCallback<Void>() {

            @Override
            public void onFailure(Throwable caught) {
                Window.alert(caught.getLocalizedMessage());
            }

            @Override
            public void onSuccess(Void result) {
                messageArea.setInfoMessage("Task " + (isEnabled ? "activated" : "deactivated"));
                parent.refresh();
            }
        });
    }

    @Override
    public void initialize(AsyncCallback<Void> callback) {
        callback.onSuccess(null);
    }

    @Override
    public List<Column<?, UIOrderedTask>> getColumns() {
        return columns;
    }

    @Override
    public void getRecords(final AsyncCallback<List<UIOrderedTask>> callback) {
        notifyBusyStatus(true);
        taskService.getTasks(partner.getId(), new AsyncCallback<List<UITask>>() {

            @Override
            public void onFailure(Throwable caught) {
                callback.onFailure(caught);
                notifyBusyStatus(false);
            }

            @Override
            public void onSuccess(List<UITask> result) {
                callback.onSuccess(IngestTaskSorter.sortByFollowUpOrder(result));
                notifyBusyStatus(false);
            }
        });
    }



    protected void notifyBusyStatus(boolean isBusy) {
        if (busyIndicator != null) {
            if (isBusy) {
                busyIndicator.center();
            } else {
                busyIndicator.hide();
            }
        }
    }

    @Override
    public void getRecord(Long key, AsyncCallback<UIOrderedTask> callback) {
        //
    }

    @Override
    public boolean isRecordCountEnabled() {
        return true;
    }

    @Override
    public boolean isCheckboxEnabled() {
        return false;
    }

    public void setBusyNotifier(BusyIndicator busyNotifier) {
        this.busyIndicator = busyNotifier;
    }

    public void setPartner(UIPartner partner) {
        this.partner = partner;
    }

    public UIPartner getPartner() {
        return partner;
    }

    public void setMessageArea(MessageArea messageArea) {
        this.messageArea = messageArea;
    }

    abstract AbstractTaskDialog getTaskDialog(UITask task, UIPartner partner, AbstractTaskListViewPanel parent);

    void cancelTask(UITask task) {
        String ingestAction = task.getType().equals("INGEST") ? constants.provider() : constants.publisher();
        if (Window.confirm(messages.cancelMessage(ingestAction))) {
            jobService.cancel(task.getId(), new DefaultAsyncCallback<Void>(null) {
                @Override
                public void onNeptuneSuccess(Void result) {
                    parent.refresh();
                }
            });
        }
    }

    void cloneTask(UITask task) {
        parent.getMessageArea().reset();
        UITask clonedTask = new UITask(task);
        clonedTask.getEndpoint().setId(0L);
        clonedTask.setId(0L);
        clonedTask.setName(task.getName() + "_" + "Cloned");
        for (UIPluginProperty value : clonedTask.getPluginPropertyValues()) {
            value.setId(0L);
        }

        for (Entry<String, UITaskSetting> e : clonedTask.getTaskSettings().entrySet()) {
            e.getValue().setId(0L);
        }
        clonedTask.setEnabled(false);
        AbstractTaskDialog taskDialog = getTaskDialog(clonedTask, partner, parent);
        parent.setTaskDialog(taskDialog);
        taskDialog.center();
    }

    void deleteTask(final UITask task) {
        taskService.getEnabledSubscribedTasks(task.getId(), new DefaultAsyncCallback<Set<UITask>>(null) {
            @Override
            public void onNeptuneSuccess(Set<UITask> result) {
                boolean hasSubscrbiedPulbisherTasks = result.size() != 0;
                boolean updateConfirmed = true;
                String info = "";
                if (hasSubscrbiedPulbisherTasks) {
                    info += messages.subscriberTasksMessage("task", tasks2String(result)) + "\n";
                }

                info += messages.deleteTaskConfirmation(task.getName());

                updateConfirmed = Window.confirm(info);
                if (updateConfirmed) {
                    taskService.deleteTask(task.getId(), new DefaultAsyncCallback<Void>(null) {
                        @Override
                        public void onNeptuneSuccess(Void result) {
                            messageArea.setInfoMessage("Task deleted.");
                            parent.refresh();
                        }
                    });
                }
            }
        });
    }

    void startTask(UITask task) {
        jobService.startTask(task.getId(), new DefaultAsyncCallback<Void>(null) {
            @Override
            public void onNeptuneSuccess(Void result) {
                parent.refresh();
            }
        });

    }

    private class ActivateValueChangeHandler implements ValueChangeHandler<Boolean> {

        private final UITask task;

        public ActivateValueChangeHandler(UITask task) {
            this.task = task;
        }

        @Override
        public void onValueChange(final ValueChangeEvent<Boolean> event) {
            boolean enable = event.getValue();
            final CheckBoxWidget checkbox = (CheckBoxWidget) event.getSource();
            if (enable) {
                enableTask(checkbox);
            } else {
                confirmDisableTask(task, checkbox);
            }
        }

        private void enableTask(final CheckBoxWidget checkbox) {
            pluginService.isPluginActive(task.getFormatId(), new AsyncCallback<Boolean>() {
                @Override
                public void onFailure(Throwable caught) {
                    checkbox.setValue(false);
                    Window.alert(caught.getLocalizedMessage());
                }

                @Override
                public void onSuccess(Boolean result) {
                    if (!result) {
                        checkbox.setValue(false);
                        Window.alert(messages.enableTaskWhilePluginIsDisabled());
                        return;
                    }
                    AbstractTaskDataProvider.this.enableTask(task.getId(), true);
                }
            });
        }
    }
    private void confirmDisableTask(final UITask task,final CheckBoxWidget checkbox){
        taskService.getTask(task.getId(), new DefaultAsyncCallback<UITask>(messageArea){

            @Override
            public void onNeptuneSuccess(UITask result) {
                if (result == null){
                    return;
                }
                String message= messages.disableTaskConfirm();
                if (result.getJobStatus().getJobStatus() == JobStatus.RUNNING) {
                    message = messages.disableRunningTaskConfirm();
                }
                if(Window.confirm(message)){
                    disableTask(task, checkbox);
                }else{
                    checkbox.setValue(true);
                }
            }
        });
    }

    private void disableTask(final UITask task, final CheckBoxWidget checkbox) {
        final long taskId = task.getId();

        //for disabing export task, no need to confirm
        if(task.getType().equalsIgnoreCase("Export")) {
            enableTask(taskId, false);
            return;
        }

        //for disabing ingest task, need to confirm
        taskService.getEnabledSubscribedTasks(taskId, new NeptuneAsyncCallback<Set<UITask>>() {
            @Override
            public void onNeptuneFailure(Throwable caught) {
            }

            @Override
            public void onNeptuneSuccess(Set<UITask> tasks) {
                boolean needConfirm = tasks.size() !=0;
                boolean updateConfirmed = true;
                if(needConfirm) {
                    String info = messages.subscriberTasksMessage("task", tasks2String(tasks)) + "";
                    info += " Continue?";
                    updateConfirmed = Window.confirm(info);
                }

                if(updateConfirmed) {
                    enableTask(taskId, false);
                }
                else {
                    //cancel the disable
                    checkbox.setValue(true);
                }
            }
        });
    }

    private String tasks2String(Set<UITask> tasks) {
        StringBuilder sb = new StringBuilder();

        for (UITask task : tasks) {
            sb.append(task.getPartnerName() + ":" + task.getName() + ", ");
        }
        String str = sb.toString();
        str = str.substring(0, str.length() - 2);

        return str;
    }
}
