package com.ericsson.cms.epgmgmt.client.task;

import java.util.List;

import com.ericsson.cms.epgmgmt.client.i18n.EPGComponentConstants;
import com.ericsson.cms.epgmgmt.client.i18n.EPGComponentMessages;
import com.ericsson.cms.epgmgmt.client.model.UIEndPoint;
import com.ericsson.cms.epgmgmt.client.model.UIEndPointEnum;
import com.ericsson.cms.epgmgmt.client.model.UIPartner;
import com.ericsson.cms.epgmgmt.client.model.UITask;
import com.ericsson.cms.epgmgmt.client.model.enumeration.JobType;
import com.ericsson.cms.epgmgmt.client.partner.ExecutionLogTab;
import com.ericsson.cms.epgmgmt.client.rpc.IEPGTaskService;
import com.ericsson.cms.epgmgmt.client.rpc.IEPGTaskServiceAsync;
import com.ericsson.cms.epgmgmt.client.utils.DefaultAsyncCallback;
import com.ericsson.cms.epgmgmt.client.utils.MessageAreaUtils;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.HasHorizontalAlignment;
import com.google.gwt.user.client.ui.PopupPanel;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.gwtext.client.widgets.ToolTip;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ButtonWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.ImageWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.messagearea.MessageArea;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;

public abstract class AbstractTaskDialog extends PopupPanel {

    private static final String DIALOG_WIDTH = "650px";
    public static final String TAB_HEIGHT = "550px";
    protected UITask history;
    protected UITask current;
    protected final UIPartner partner;
    protected ButtonWidget createButton;
    protected MessageArea messageArea;
    protected AbstractTaskDialog dialog = this;
    protected BusyIndicator busyIndicator = new BusyIndicator();
    protected final AbstractTaskListViewPanel parent;
    protected final EPGComponentConstants constants = GWT.create(EPGComponentConstants.class);
    protected final EPGComponentMessages messages = GWT.create(EPGComponentMessages.class);
    protected final IEPGTaskServiceAsync taskService = GWT.create(IEPGTaskService.class);

    public AbstractTaskDialog(UIPartner partner, AbstractTaskListViewPanel parent) {
        this.parent = parent;
        this.partner = partner;
        history = createEmptyModel();
        history.setPartnerId(partner.getId());
        setModelDefaultValue(history);
        current = new UITask(history);
        init();
    }

    public AbstractTaskDialog(UITask task, UIPartner partner, AbstractTaskListViewPanel parent) {
        this.parent = parent;
        this.partner = partner;
        history = task;
        current = new UITask(history);
        init();
    }

    private void init() {
        setModal(true);
        setAnimationEnabled(true);
        setGlassEnabled(true);
        setWidth(DIALOG_WIDTH);

        VerticalContainer vc = new VerticalContainer();

        initCloseButton(vc);
        initMessageArea(vc);
        initContent(vc);
        initButton(vc);

        add(vc);
    }

    private void initMessageArea(VerticalContainer vc) {
        messageArea = new MessageArea();
        messageArea.reset();
        HorizontalContainer hc = new HorizontalContainer();
        hc.setWidth("100%");
        hc.add(messageArea);
        vc.add(hc);
    }

    private void initButton(VerticalContainer vc) {
        HorizontalContainer outer = new HorizontalContainer();
        outer.setHorizontalAlignment(HasHorizontalAlignment.ALIGN_CENTER);
        outer.setWidth("100%");

        HorizontalContainer hc = new HorizontalContainer();

        if (!partner.isActive()) {
            hc.add(new ToolTip(messages.cannotEditTaskWhilePartnerIsDisabled(partner.getPartnerType().toString())));
        }

        createButton = new ButtonWidget(constants.buttonLabelSave());
        createButton.addStyleDependentName("act-towards-save-button");
        createButton.addClickHandler(getCreateButtonClickHandler());
        createButton.setEnabled(partner.isActive());
        hc.add(createButton);

        ButtonWidget cancelButton = new ButtonWidget(current.getId() == 0 ? constants.buttonLabelCancel()
                : constants.buttonLabelClose());
        cancelButton.addStyleDependentName("dataloss-button");
        cancelButton.addClickHandler(new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {
                dialog.hide();
            }
        });
        hc.add(cancelButton);

        outer.add(hc);
        vc.add(outer);
    }

    private UITask createEmptyModel() {
        UITask ut = new UITask();
        ut.setEnabled(true);
        UIEndPoint uep = new UIEndPoint();
        ut.setEndpoint(uep);
        return ut;
    }

    private void initCloseButton(VerticalPanel vp) {
        HorizontalContainer hc = new HorizontalContainer();
        hc.setWidth("100%");
        hc.setHorizontalAlignment(HasHorizontalAlignment.ALIGN_RIGHT);
        hc.add(createCloseButton());
        vp.add(hc);
    }

    private ImageWidget createCloseButton() {
        ImageWidget cb = new ImageWidget();
        ClickHandler handler = new ClickHandler() {
            @Override
            public void onClick(ClickEvent event) {
                dialog.hide();
            }
        };
        cb.setUrl("cms_epgmgmt_ui/images/delete.gif");
        cb.setHeight("18px");
        cb.setWidth("18px");
        cb.addClickHandler(handler);
        cb.addStyleName("epg-exportsetting-verticalpanel-widget");
        return cb;
    }

    public void clearMessageArea() {
        messageArea.reset();
    }

    public abstract UITask populateTask();

    abstract protected void initContent(VerticalContainer vc);

    abstract protected ClickHandler getCreateButtonClickHandler();

    abstract protected void setModelDefaultValue(UITask model);

    protected void submit(UITask task) {
        if (!createButton.isEnabled()) {
            return;
        }

        busyIndicator.center();
        createButton.setEnabled(false);
        if (task.getId() == 0) {
            // Create
            taskService.createTask(task, new NeptuneAsyncCallback<Long>() {
                @Override
                public void onNeptuneFailure(Throwable caught) {
                    busyIndicator.hide();
                    createButton.setEnabled(true);
                    MessageAreaUtils.handleException(caught, messageArea);
                }

                @Override
                public void onNeptuneSuccess(Long result) {
                    busyIndicator.hide();
                    dialog.hide();
                    createButton.setEnabled(true);
                    parent.refresh();
                }
            });
        } else {
            // Update
            taskService.updateTask(task, new NeptuneAsyncCallback<Void>() {
                @Override
                public void onNeptuneFailure(Throwable caught) {
                    busyIndicator.hide();
                    createButton.setEnabled(true);
                    MessageAreaUtils.handleException(caught, messageArea);
                }

                @Override
                public void onNeptuneSuccess(Void result) {
                    busyIndicator.hide();
                    messageArea.setInfoMessage("Task updated.");
                    dialog.hide();
                    createButton.setEnabled(true);
                    parent.refresh();
                }
            });
        }
    }

    /**
     * Validate the plugin properties, if no violation occur then submit the task.
     */
    protected void validateAndSubmit(final UITask task) {
        taskService.validateTask(task, new NeptuneAsyncCallback<List<String>>() {
            @Override
            public void onNeptuneFailure(Throwable caught) {
                MessageAreaUtils.handleException(caught, messageArea);
            }

            @Override
            public void onNeptuneSuccess(List<String> result) {
                if (result.size() > 0) {
                    MessageAreaUtils.displayErrorMessages("", result, "", messageArea);
                    return;
                }
                findConflictEndpointAndSubmit(task);
            }
        });
    }

    protected void findConflictEndpointAndSubmit(final UITask task) {
        taskService.getConflictEndpoint(task, new DefaultAsyncCallback<UITask>(messageArea) {
            @Override
            public void onNeptuneSuccess(UITask conflicted) {
                if (conflicted == null) {
                    submit(task);
                    return;
                }

                String msg = messages.conflictFileEndpoint(conflicted.getName(), conflicted.getPartnerType(),
                        conflicted.getPartnerName());
                if (UIEndPointEnum.file.name().equals(task.getEndpoint().getEndpointType())) {
                    MessageAreaUtils.displayErrorMessage("", msg, "", messageArea);
                } else if (Window.confirm(msg)) {
                    submit(task);
                }
            }
        });
    }

    protected ExecutionLogTab createExeLogTab(JobType jobType) {
        ExecutionLogTab logTab = ExecutionLogTab.createForTask(jobType);
        logTab.setMessageArea(messageArea);
        logTab.refreshData(history);
        return logTab;
    }
}
