package com.ericsson.cms.epgmgmt.client.task;

import static com.ericsson.cms.epgmgmt.client.model.UITaskSetting.PROVIDER_INCLUDE_KEY;
import static com.ericsson.cms.epgmgmt.client.model.UITaskSetting.PUBLISHER_FILTERS_KEY;

import java.util.ArrayList;
import java.util.List;

import com.ericsson.cms.epgmgmt.client.EPGFormContainer;
import com.ericsson.cms.epgmgmt.client.i18n.EPGComponentConstants;
import com.ericsson.cms.epgmgmt.client.i18n.EPGComponentMessages;
import com.ericsson.cms.epgmgmt.client.model.UIEndPoint;
import com.ericsson.cms.epgmgmt.client.model.UIEndPointEnum;
import com.ericsson.cms.epgmgmt.client.model.UIEpgFormat;
import com.ericsson.cms.epgmgmt.client.model.UIPartner;
import com.ericsson.cms.epgmgmt.client.model.UITask;
import com.ericsson.cms.epgmgmt.client.model.UITaskSetting;
import com.ericsson.cms.epgmgmt.client.model.enumeration.JobType;
import com.ericsson.cms.epgmgmt.client.rpc.IEPGPartnerService;
import com.ericsson.cms.epgmgmt.client.rpc.IEPGPartnerServiceAsync;
import com.ericsson.cms.epgmgmt.client.rpc.IEPGPluginManagerService;
import com.ericsson.cms.epgmgmt.client.rpc.IEPGPluginManagerServiceAsync;
import com.google.gwt.core.client.GWT;
import com.google.gwt.event.dom.client.BlurEvent;
import com.google.gwt.event.dom.client.BlurHandler;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Widget;
import com.tandbergtv.neptune.widgettoolkit.client.remote.NeptuneAsyncCallback;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.CheckBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.basic.TextBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.composite.BusyIndicator;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.HorizontalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.container.VerticalContainer;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.TypedListBoxWidget;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.IntegerConverter;
import com.tandbergtv.neptune.widgettoolkit.client.widget.datatype.converter.StringConverter;

public class TaskBasicSettingForm extends EPGFormContainer {

    protected static final Long NONE_SELECTED_VALUE = -1L;

    private UITask model;
    private final JobType type;
    private final IEPGPartnerServiceAsync partnerService = GWT.create(IEPGPartnerService.class);
    private final EPGComponentConstants constants = GWT.create(EPGComponentConstants.class);
    private final EPGComponentMessages constantsMsg = (EPGComponentMessages) GWT.create(EPGComponentMessages.class);
    private final IEPGPluginManagerServiceAsync pluginService = GWT.create(IEPGPluginManagerService.class);
    private final TextBoxWidget taskName = new TextBoxWidget();
    private final CheckBoxWidget isTaskEnabled = new CheckBoxWidget();
    private final TypedListBoxWidget<Long> formatTypeList = new TypedListBoxWidget<Long>(new IntegerConverter());
    private final TypedListBoxWidget<String> retrievalTypeList = new TypedListBoxWidget<String>(new StringConverter());
    private final VerticalContainer endPointOptionValueWidget = new VerticalContainer();
    private final EndPointWidget endPointWidget = new EndPointWidget();
    private final BusyIndicator busyIndicator;
    private final TypedListBoxWidget<Long> providerListBox = new TypedListBoxWidget<Long>(new IntegerConverter());
    private ExportSettingPanel exportSettingPanel;
    private TaskTriggerSettingForm triggerForm;

    public TaskBasicSettingForm(JobType type, UITask model) {
        super(HorizontalContainer.ALIGN_LEFT);
        this.type = type;
        this.model = model;
        busyIndicator = new BusyIndicator();
        initForm();
        bindEventHandler();
    }

    private void initForm() {
        isTaskEnabled.setValue(model.isEnabled());
        addRow(constants.enable(), isTaskEnabled);
        taskName.setValue(model.getName());
        addRow(constants.providerTaskNameLabel(), taskName, true);
        loadProviderListBox();

        // specify the provider to task
        if (JobType.EXPORT.equals(type)) {
            addRow(constants.specifiedProvider(), providerListBox, true);
        }

        addRow(constants.formatType(), loadFormattype(model.getFormatId()), true);
        endPointOptionValueWidget.add(setRetrievaltype(retrievalTypeList, model.getEndpoint().getEndpointType()));
        endPointOptionValueWidget.add(endPointWidget);
        addRow(constants.retrievalType(), endPointOptionValueWidget, true);

    }

    private Widget setRetrievaltype(final TypedListBoxWidget<String> retrievalTypeList, String selectedValue) {
        retrievalTypeList.clear();
        // Default is Blank
        retrievalTypeList.addItem("", "0");

        for (UIEndPointEnum uiEndPointDetails : UIEndPointEnum.values()) {
            retrievalTypeList.addItem(uiEndPointDetails.name(), uiEndPointDetails.name());
        }

        retrievalTypeList.setSelectedItem(selectedValue);

        if (selectedValue != null && !"0".equals(selectedValue)) {
            endPointWidget.setData(retrievalTypeList.getSelectedItem(), model.getEndpoint(), true);
            endPointWidget.setVisible(true);
        } else {
            endPointWidget.setVisible(false);
        }

        return retrievalTypeList;
    }

    private void bindEventHandler() {
        taskName.addBlurHandler(new BlurHandler() {
            @Override
            public void onBlur(BlurEvent event) {
                model.setName(taskName.getValue());
            }
        });
        isTaskEnabled.addValueChangeHandler(new ValueChangeHandler<Boolean>() {
            @Override
            public void onValueChange(ValueChangeEvent<Boolean> event) {
                model.setEnabled(event.getValue());
            }
        });
        formatTypeList.addValueChangeHandler(new ValueChangeHandler<Long>() {
            @Override
            public void onValueChange(ValueChangeEvent<Long> event) {
                model.setFormatId(event.getValue() == null ? 0 : event.getValue());
            }
        });
        retrievalTypeList.addValueChangeHandler(new ValueChangeHandler<String>() {
            @Override
            public void onValueChange(ValueChangeEvent<String> event) {
                String selectedValue = retrievalTypeList.getSelectedItem();
                if (selectedValue != "0") {
                    UIEndPoint uiEndPoint = new UIEndPoint();
                    if (selectedValue.equals(model.getEndpoint().getEndpointType())) {
                        uiEndPoint = model.getEndpoint();
                    }
                    endPointWidget.setData(selectedValue, uiEndPoint, true);
                    endPointWidget.setVisible(true);
                } else {
                    endPointWidget.setVisible(false);
                }
            }
        });
        providerListBox.addValueChangeHandler(new ValueChangeHandler<Long>() {
            @Override
            public void onValueChange(ValueChangeEvent<Long> event) {
                exportSettingPanel.setProviderId(providerListBox.getSelectedItem());
                triggerForm.setProviderId(providerListBox.getSelectedItem());
            }
        });
    }

    public void loadModel(UITask model) {
        this.model = model;
        isTaskEnabled.setValue(model.isEnabled());
        taskName.setValue(model.getName());
        loadFormattype(model.getFormatId());
        if (retrievalTypeList.getItemCount() > 0 && model.getEndpoint().getEndpointType() != null) {
            retrievalTypeList.setValue(model.getEndpoint().getEndpointType(), true);
        }
        loadProviderListBox();
    }

    private void loadProviderListBox() {
        if (JobType.INGEST.toString().equalsIgnoreCase(model.getType())) {
            return;
        }
        partnerService.getProviders(new NeptuneAsyncCallback<List<UIPartner>>() {
            @Override
            public void onNeptuneFailure(Throwable caught) {
                Window.alert("Get ingest tasks failed!! \n " + caught);
            }

            @Override
            public void onNeptuneSuccess(List<UIPartner> result) {
                providerListBox.clear();
                if (result != null && result.size() > 0) {
                    providerListBox.addItem(" ", NONE_SELECTED_VALUE);
                    for (UIPartner t : result) {
                        providerListBox.addItem(t.getName(), t.getId());
                    }

                    Long providerId = getSpecifiedProvider();
                    if (providerId == null) {
                        return;
                    }

                    providerListBox.setSelectedItem(providerId);

                    if (exportSettingPanel != null) {
                        exportSettingPanel.setProviderId(providerId);
                    }
                    if (triggerForm != null) {
                        triggerForm.setProviderId(providerId);
                    }
                }
            }

            private Long getSpecifiedProvider() {
                Long providerId = null;
                UITaskSetting taskSetting = model.getTaskSettings().get(PUBLISHER_FILTERS_KEY);
                if (taskSetting != null) {
                    @SuppressWarnings("unchecked")
                    List<String> providerIds = (List<String>) taskSetting.getSetting().get(PROVIDER_INCLUDE_KEY);
                    if (providerIds != null && !providerIds.isEmpty()) {
                        providerId = Long.valueOf(providerIds.get(0));
                    }
                }
                return providerId;
            }
        });
    }

    private TypedListBoxWidget<Long> loadFormattype(final long selectedValue) {
        busyIndicator.center();
        pluginService.getPlugins(type, new NeptuneAsyncCallback<List<UIEpgFormat>>() {

            @Override
            public void onNeptuneFailure(Throwable caught) {
                Window.alert(constants.loadingError() + " | " + constants.reason() + ": "
                        + (caught != null ? caught.getLocalizedMessage() : ""));
                busyIndicator.hide();
            }

            @Override
            public void onNeptuneSuccess(List<UIEpgFormat> result) {
                formatTypeList.clear();
                formatTypeList.addItem("", 0L);
                long id = 0L;

                for (UIEpgFormat epgFormat : result) {
                    if (epgFormat.isEnabled()) {
                        formatTypeList.addItem(epgFormat.getName() + " v" + epgFormat.getVersion(), epgFormat.getId());
                        if (selectedValue == epgFormat.getId()) {
                            id = selectedValue;
                        }
                    }
                }

                // set existing information
                formatTypeList.setSelectedItem(id);
                model.setFormatId(id);
                busyIndicator.hide();
            }
        });

        return formatTypeList;
    }

    public List<String> validateField() {
        List<String> validationMessages = new ArrayList<String>();
        if (taskName.getValue() == null || "".equals(taskName.getValue().trim())) {
            validationMessages.add(constantsMsg.requiredTaskName());
        }

        // format type is required
        if (formatTypeList.getValue() == 0L) {
            validationMessages.add(constantsMsg.requiredFormatTypeName());
        }
        // retrieval type is required
        if ("0".equals(retrievalTypeList.getValue())) {
            validationMessages.add(constantsMsg.requiredRetrievalProtocolName());
        }

        // endpoint widget field validation
        if (retrievalTypeList.getSelectedItem() != null) {
            List<String> validationMsg = endPointWidget.validate();
            if (validationMsg.size() > 0) {
                for (String valMsg : validationMsg) {
                    validationMessages.add(valMsg);
                }
            }
        }
        return validationMessages;
    }

    public UITask populateModel() {
        model.setName(taskName.getValue());
        model.setType(type.toString());
        model.setFormatId(formatTypeList.getSelectedItem());
        model.setEnabled(isTaskEnabled.getValue());
        UIEndPoint endPoint = endPointWidget.setEndPointForm();
        endPoint.setEndpointType(retrievalTypeList.getSelectedItem());
        if (model.getEndpoint() != null) {
            endPoint.setId(model.getEndpoint().getId());
        }
        model.setEndpoint(endPoint);
        return model;
    }

    public TypedListBoxWidget<Long> getFormatTypeList() {
        return formatTypeList;
    }

    public CheckBoxWidget getIsTaskEnabled() {
        return isTaskEnabled;
    }

    public void setExportSettingPanel(ExportSettingPanel exportSettingPanel) {
        this.exportSettingPanel = exportSettingPanel;
        if (providerListBox.getSelectedItem() != null) {
            exportSettingPanel.setProviderId(providerListBox.getSelectedItem());
        }
    }

    public TaskTriggerSettingForm getTriggerForm() {
        return triggerForm;
    }

    public void setTriggerForm(TaskTriggerSettingForm triggerForm) {
        this.triggerForm = triggerForm;
        if (providerListBox.getSelectedItem() != null) {
            triggerForm.setProviderId(providerListBox.getSelectedItem());
        }
    }
}
